% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/univLogReg.R
\name{big_univLogReg}
\alias{big_univLogReg}
\title{Column-wise logistic regression}
\usage{
big_univLogReg(
  X,
  y01.train,
  ind.train = rows_along(X),
  ind.col = cols_along(X),
  covar.train = NULL,
  tol = 1e-08,
  maxiter = 20,
  ncores = 1
)
}
\arguments{
\item{X}{An object of class \link[=FBM-class]{FBM}.}

\item{y01.train}{Vector of responses, corresponding to \code{ind.train}.
\strong{Must be only 0s and 1s.}}

\item{ind.train}{An optional vector of the row indices that are used,
for the training part. If not specified, all rows are used.
\strong{Don't use negative indices.}}

\item{ind.col}{An optional vector of the column indices that are used.
If not specified, all columns are used. \strong{Don't use negative indices.}}

\item{covar.train}{Matrix of covariables to be added in each model to correct
for confounders (e.g. the scores of PCA), corresponding to \code{ind.train}.
Default is \code{NULL} and corresponds to only adding an intercept to each model.
You can use \code{\link[=covar_from_df]{covar_from_df()}} to convert from a data frame.}

\item{tol}{Relative tolerance to assess convergence of the coefficient.
Default is \code{1e-8}.}

\item{maxiter}{Maximum number of iterations before giving up.
Default is \code{20}. Usually, convergence is reached within 3 or 4 iterations.
If there is not convergence,
\link[stats:glm]{glm} is used instead for the corresponding column.}

\item{ncores}{Number of cores used. Default doesn't use parallelism.
You may use \link{nb_cores}.}
}
\value{
A data.frame with 4 elements:
\enumerate{
\item the slopes of each regression,
\item the standard errors of each slope,
\item the number of iteration for each slope. If is \code{NA}, this means that the
algorithm didn't converge, and \link[stats:glm]{glm} was used instead.
\item the z-scores associated with each slope.
This is also an object of class \code{mhtest}. See \code{methods(class = "mhtest")}.
}
}
\description{
Slopes of column-wise logistic regressions of each column
of a Filebacked Big Matrix, with some other associated statistics.
Covariates can be added to correct for confounders.
}
\details{
If convergence is not reached by the main algorithm for some columns,
the corresponding \code{niter} element is set to \code{NA} and a message is given.
Then, \link[stats:glm]{glm} is used instead for the corresponding column.
If it can't converge either, all corresponding estimations are set to \code{NA}.
}
\examples{
set.seed(1)

X <- big_attachExtdata()
n <- nrow(X)
y01 <- sample(0:1, size = n, replace = TRUE)
covar <- matrix(rnorm(n * 3), n)

X1 <- X[, 1] # only first column of the Filebacked Big Matrix

# Without covar
test <- big_univLogReg(X, y01)
## new class `mhtest`
class(test)
attr(test, "transfo")
attr(test, "predict")
## plot results
plot(test)
plot(test, type = "Volcano")
## To get p-values associated with the test
test$p.value <- predict(test, log10 = FALSE)
str(test)
summary(glm(y01 ~ X1, family = "binomial"))$coefficients[2, ]

# With all data
str(big_univLogReg(X, y01, covar.train = covar))
summary(glm(y01 ~ X1 + covar, family = "binomial"))$coefficients[2, ]

# With only half of the data
ind.train <- sort(sample(n, n/2))
str(big_univLogReg(X, y01[ind.train],
                   covar.train = covar[ind.train, ],
                   ind.train = ind.train))
summary(glm(y01 ~ X1 + covar, family = "binomial",
            subset = ind.train))$coefficients[2, ]
}
\seealso{
\link[stats:glm]{glm}
}
