% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/read_fcs.R
\name{read_fcs}
\alias{read_fcs}
\title{Read a fcs file.}
\usage{
read_fcs(
  .file_name,
  .fsc_ssc = c("FSC-A", "SSC-A"),
  .bead_channels = c("FL6-H", "FL2-H"),
  .filter = list(`FSC-A` = c(200000L, 800000L), `SSC-A` = c(200000L, 1000000L), `FL6-H`
    = c(7.3, Inf)),
  .compensation = "guess",
  ...
)
}
\arguments{
\item{.file_name}{The path and name of the file to be read.}

\item{.fsc_ssc}{The names of the forward and side scatter channels. A
character vector of length of two.}

\item{.bead_channels}{The names of the channels with bead events. A character
vector of length of at least two.}

\item{.filter}{Optional list of upper and lower cutoff for individual
channels. Use \code{.filter = NULL} for no filtering at all.}

\item{.compensation}{A character vector, a compensation matrix, or
\code{NULL}. See 'Details' for extended information of the argument.}

\item{...}{additional arguments passed to \code{flowCore::read.FCS}}
}
\value{
A \code{data.frame} with the columns given in \code{.fsc_ssc} and \code{.bead_channels}.
}
\description{
This is a wrapper around reading a FACS data file using \code{flowCore},
transforming the channels with bead signal events using
\code{arcsinhTransform()} from \code{flowCore}, automatic removing boundary
events of the forward and side scatter channels, subsetting channels (if
needed), and cast to a \code{data.frame}.
}
\details{
The \code{.compensation} argument takes a character vector or an actual
compensation matrix. In case of the latter, it must be a object of class
\code{matrix}. If an object of class \code{character} is given to the
\code{.compensation} argument, it can be the keyword 'guess' or a search
pattern matching the keyword in the fcs file that contains the compensation
matrix. If the keyword 'guess' is passed, the function looks for a matrix at
the keywords "SPILL" and "SPILLOVER", and if none are found it takes the
first matrix which contains the parameter names as column names. Finally, the
values of \code{.compensation} can also be \code{NULL}, in which case, no
compensation is performed.

To summarize, the argument \code{.compensation} can be:

\describe{
\item{A matrix}{The compensation matrix to apply}
\item{A character}{
\describe{
\item{The word 'guess'}{Looks for a matrix at the keywords "SPILL"
and "SPILLOVER". If none found the first matrix with parameter names is
applied}
\item{A search string}{The first matrix found within the keywords
matching the given search string is applied. The search string can be a
regular expression}
}
}
\item{NULL}{Nothing is done}
}
}
\examples{
library(beadplexr)

.file_name <- system.file("extdata", "K2-C07-A7.fcs",
                          package = "beadplexr")

# Load the fcs file, with no filter
.data <- read_fcs(.file_name = .file_name, .filter = NULL)

plot(.data[, c("FSC-A", "SSC-A")])
plot(.data[, c("FL2-H", "FL6-H")])

# Load the fcs file, with default filter
.data <- read_fcs(.file_name = .file_name, .fsc_ssc = c("FSC-H", "SSC-H"))

plot(.data[, c("FSC-H", "SSC-H")])
plot(.data[, c("FL2-H", "FL6-H")])

# Load the fcs file, with custom filter
.data <- read_fcs(.file_name = .file_name,
                  .filter = list("FSC-A" = c(2e5L, 3.3e5L),
                                 "SSC-A" = c(2e5, 1e6L),
                                 "FL2-H" = c(11L, 14L),
                                 "FL6-H" = c(9L, Inf)))

plot(.data[, c("FSC-A", "SSC-A")])
plot(.data[, c("FL2-H", "FL6-H")])

# Specify three bead channels

.data <- read_fcs(.file_name = .file_name,
                  .bead_channels = c("FL6-H", "FL2-H", "FL1-H"))

plot(.data[, c("FSC-A", "SSC-A")])
plot(.data[, c("FL2-H", "FL6-H")])
plot(.data[, c("FL1-H", "FL2-H")])
plot(.data[, c("FL1-H", "FL6-H")])

}
\seealso{
\code{\link[flowCore]{rectangleGate}} for instructions to the
\code{.filter} list argument, \code{\link[flowCore]{boundaryFilter}}
for automatic removal of boundary events, and
\code{\link[flowCore]{arcsinhTransform}} for the transformation.
}
