% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/osdsm.R
\name{osdsm}
\alias{osdsm}
\title{Extract backbone using the Ordinal Stochastic Degree Sequence Model}
\usage{
osdsm(
  B,
  alpha = 0.05,
  trials = NULL,
  signed = FALSE,
  mtc = "none",
  class = "original",
  narrative = FALSE
)
}
\arguments{
\item{B}{An ordinally weighted bipartite graph, as: (1) an incidence matrix in the form of a matrix or sparse \code{\link{Matrix}}; (2) an edgelist in the form of a three-column dataframe; (3) an \code{\link{igraph}} object.
Any rows and columns of the associated bipartite matrix that contain only zeros or only ones are automatically removed before computations.}

\item{alpha}{real: significance level of hypothesis test(s)}

\item{trials}{integer: the number of bipartite graphs generated to approximate the edge weight distribution. If NULL, the number of trials is selected based on \code{alpha} (see details)}

\item{signed}{boolean: TRUE for a signed backbone, FALSE for a binary backbone (see details)}

\item{mtc}{string: type of Multiple Test Correction to be applied; can be any method allowed by \code{\link{p.adjust}}.}

\item{class}{string: the class of the returned backbone graph, one of c("original", "matrix", "Matrix", "igraph", "edgelist").
If "original", the backbone graph returned is of the same class as \code{B}.}

\item{narrative}{boolean: TRUE if suggested text & citations should be displayed.}
}
\value{
If \code{alpha} != NULL: Binary or signed backbone graph of class \code{class}.

If \code{alpha} == NULL: An S3 backbone object containing three matrices (the weighted graph, edges' upper-tail p-values,
edges' lower-tail p-values), and a string indicating the null model used to compute p-values, from which a backbone
can subsequently be extracted using \code{\link[=backbone.extract]{backbone.extract()}}. The \code{signed}, \code{mtc}, \code{class}, and \code{narrative} parameters
are ignored.
}
\description{
\code{osdsm} extracts the backbone of a bipartite projection using the Ordinal Stochastic Degree Sequence Model.
}
\details{
The \code{osdsm} function compares an edge's observed weight in the projection \code{B*t(B)} to the distribution of weights
expected in a projection obtained from a random bipartite network where both the rows and the columns contain
approximately the same number of each value. The edges in \code{B} must be integers, and are assumed to represent an
ordinal-level measure such as a Likert scale that starts at 0.

When \code{signed = FALSE}, a one-tailed test (is the weight stronger) is performed for each edge with a non-zero weight. It
yields a backbone that perserves edges whose weights are significantly \emph{stronger} than expected in the chosen null
model. When \code{signed = TRUE}, a two-tailed test (is the weight stronger or weaker) is performed for each every pair of nodes.
It yields a backbone that contains positive edges for edges whose weights are significantly \emph{stronger}, and
negative edges for edges whose weights are significantly \emph{weaker}, than expected in the chosen null model.
\emph{NOTE: Before v2.0.0, all significance tests were two-tailed and zero-weight edges were evaluated.}

The p-values used to evaluate the statistical significance of each edge are computed using Monte Carlo methods. The number of
\code{trials} performed affects the precision of these p-values, and the confidence that a given p-value is less than the
desired \code{alpha} level. Because these p-values are proportions (i.e., the proportion of times an edge is weaker/stronger
in the projection of a random bipartite graphs), evaluating the statistical significance of an edge is equivalent to
comparing a proportion (the p-value) to a known proportion (alpha). When \code{trials = NULL}, the \code{power.prop.test} function
is used to estimate the required number of trials to make such a comparison with a \code{alpha} type-I error rate, (1-\code{alpha}) power,
and when the riskiest p-value being evaluated is at least 5\% smaller than \code{alpha}. When any \code{mtc} correction is applied,
for simplicity this estimation is based on a conservative Bonferroni correction.
}
\examples{
#A weighted binary bipartite network of 20 agents & 50 artifacts; agents form two communities
B <- rbind(cbind(matrix(sample(0:3, 250, replace = TRUE, prob = ((1:4)^2)),10),
                 matrix(sample(0:3, 250, replace = TRUE, prob = ((4:1)^2)),10)),
           cbind(matrix(sample(0:3, 250, replace = TRUE, prob = ((4:1)^2)),10),
                 matrix(sample(0:3, 250, replace = TRUE, prob = ((1:4)^2)),10)))

P <- B\%*\%t(B) #An ordinary weighted projection...
plot(igraph::graph_from_adjacency_matrix(P, mode = "undirected",
                                         weighted = TRUE, diag = FALSE)) #...is a dense hairball

bb <- osdsm(B, alpha = 0.05, narrative = TRUE,  #An oSDSM backbone...
            class = "igraph", trials = 100)
plot(bb) #...is sparse with clear communities
}
\references{
package: {Neal, Z. P. (2022). backbone: An R Package to Extract Network Backbones. \emph{PLOS ONE, 17}, e0269137. \doi{10.1371/journal.pone.0269137}}

osdsm: {Neal, Z. P. (2017). Well connected compared to what? Rethinking frames of reference in world city network research. \emph{Environment and Planning A, 49}, 2859-2877. \doi{10.1177/0308518X16631339}}
}
