#' Pooled central moment estimates - two-sample
#'
#' Calculate pooled unbiased estimates of central moments and their powers and
#' products.
#'
#' @family pooled estimates (two-sample)
#'
#' @param m2 naive biased variance estimate \eqn{m_2 = 1/(n_x + n_y) \sum_{i =
#'   1}^{n_x} ((X_i - \bar{X})^2 + \sum_{i = 1}^{n_y} ((Y_i - \bar{Y})^2}{m[2] =
#'   mean(c((X - X-bar)^2, (Y - Y-bar)^2))} for vectors \code{X} and \code{Y}.
#' @param m3 naive biased third central moment estimate \eqn{m_3 = 1/(n_x + n_y)
#'   \sum_{i = 1}^{n_x} ((X_i - \bar{X})^3 + \sum_{i = 1}^{n_y} ((Y_i -
#'   \bar{Y})^3}{m[3] = mean(c((X - X-bar)^3, (Y - Y-bar)^3))} for vectors
#'   \code{X} and \code{Y}.
#' @param m4 naive biased fourth central moment estimate \eqn{m_4 = 1/(n_x +
#'   n_y) \sum_{i = 1}^{n_x} ((X_i - \bar{X})^4 + \sum_{i = 1}^{n_y} ((Y_i -
#'   \bar{Y})^4}{m[4] = mean(c((X - X-bar)^4, (Y - Y-bar)^4))} for vectors
#'   \code{X} and \code{Y}.
#' @param m6 naive biased sixth central moment estimate \eqn{m_6 = 1/(n_x + n_y)
#'   \sum_{i = 1}^{n_x} ((X_i - \bar{X})^6 + \sum_{i = 1}^{n_y} ((Y_i -
#'   \bar{Y})^6}{m[6] = mean(c((X - X-bar)^6, (Y - Y-bar)^6))} for vectors
#'   \code{X} and \code{Y}.
#' @param n_x number of observations in the first group.
#' @param n_y number of observations in the second group.
#' @return Unbiased estimate of a sixth central moment.
#' @examples
#' nx <- 10
#' ny <- 8
#' shp <- 3
#' smpx <- rgamma(nx, shape = shp) - shp
#' smpy <- rgamma(ny, shape = shp)
#' mx <- mean(smpx)
#' my <- mean(smpy)
#' m  <- numeric(6)
#' for (j in 2:6) {
#'   m[j] <- mean(c((smpx - mx)^j, (smpy - my)^j))
#' }
#' uM6pool(m[2], m[3], m[4], m[6], nx, ny)
#' @export
uM6pool <- function(m2, m3, m4, m6, n_x, n_y) {
    (m6*n_x^7*n_y^15 + 900*m2^3*n_x^12 + (8*m6*n_x^8 - 3*(10*m2*m4 + 11*m6)*n_x^7 - 5*(9*m2^3 - 4*m3^2 - 15*m2*m4 - 2*m6)*n_x^6 + (210*m2^3 - 80*m3^2 - 165*m2*m4 - 3*m6)*n_x^5 - 5*(63*m2^3 - 22*m3^2 - 36*m2*m4)*n_x^4 + 25*(6*m2^3 - 2*m3^2 - 3*m2*m4)*n_x^3)*n_y^14 - 11700*m2^3*n_x^11 + (28*m6*n_x^9 - 21*(10*m2*m4 + 11*m6)*n_x^8 + 5*(8*m3^2 + 252*m2*m4 + 111*m6)*n_x^7 + 15*(68*m2^3 - 44*m3^2 - 178*m2*m4 - 25*m6)*n_x^6 - 2*(1800*m2^3 - 915*m3^2 - 2040*m2*m4 - 131*m6)*n_x^5 + (4545*m2^3 - 2370*m3^2 - 3540*m2*m4 - 97*m6)*n_x^4 - 5*(369*m2^3 - 320*m3^2 - 324*m2*m4)*n_x^3 - 30*(10*m2^3 + 21*m3^2 + 10*m2*m4)*n_x^2 + 75*(3*m2^3 + 2*m3^2)*n_x)*n_y^13 + 12600*m2^3*n_x^10 + (56*m6*n_x^10 - 63*(10*m2*m4 + 11*m6)*n_x^9 + 10*(90*m2^3 - 16*m3^2 + 606*m2*m4 + 313*m6)*n_x^8 - 3*(1430*m2^3 + 60*m3^2 + 7015*m2*m4 + 2023*m6)*n_x^7 + 5*(828*m2^3 + 1076*m3^2 + 7419*m2*m4 + 1110*m6)*n_x^6 + 2*(1110*m2^3 - 6335*m3^2 - 22425*m2*m4 - 2179*m6)*n_x^5 - (1725*m2^3 - 14410*m3^2 - 32415*m2*m4 - 2318*m6)*n_x^4 - (9825*m2^3 + 7750*m3^2 + 13680*m2*m4 + 684*m6)*n_x^3 + 900*m2^3 + 15*(841*m2^3 + 112*m3^2 + 195*m2*m4 + 8*m6)*n_x^2 - 150*(36*m2^3 + m3^2)*n_x)*n_y^12 - 1800*m2^3*n_x^9 + (70*m6*n_x^11 - 105*(10*m2*m4 + 11*m6)*n_x^10 + 5*(576*m2^3 - 136*m3^2 + 2820*m2*m4 + 1537*m6)*n_x^9 - 15*(1962*m2^3 - 396*m3^2 + 4853*m2*m4 + 1679*m6)*n_x^8 + 10*(10701*m2^3 - 1274*m3^2 + 18657*m2*m4 + 4265*m6)*n_x^7 - (211845*m2^3 - 880*m3^2 + 280020*m2*m4 + 40877*m6)*n_x^6 + (296160*m2^3 + 17320*m3^2 + 284895*m2*m4 + 29256*m6)*n_x^5 - 5*(67131*m2^3 + 4442*m3^2 + 35886*m2*m4 + 2710*m6)*n_x^4 + (306405*m2^3 + 5450*m3^2 + 70110*m2*m4 + 2934*m6)*n_x^3 - 11700*m2^3 - 15*(12901*m2^3 - 262*m3^2 + 1100*m2*m4 + 8*m6)*n_x^2 + 75*(951*m2^3 - 14*m3^2 + 12*m2*m4)*n_x)*n_y^11 + (56*m6*n_x^12 - 105*(10*m2*m4 + 11*m6)*n_x^11 + 50*(81*m2^3 - 20*m3^2 + 369*m2*m4 + 204*m6)*n_x^10 - 3*(20530*m2^3 - 4940*m3^2 + 43455*m2*m4 + 15936*m6)*n_x^9 + (341055*m2^3 - 72850*m3^2 + 473205*m2*m4 + 124842*m6)*n_x^8 - 3*(317855*m2^3 - 51090*m3^2 + 323705*m2*m4 + 61741*m6)*n_x^7 + 2*(820815*m2^3 - 82160*m3^2 + 629655*m2*m4 + 82827*m6)*n_x^6 - 6*(323010*m2^3 - 18495*m3^2 + 186405*m2*m4 + 16976*m6)*n_x^5 + 2*(826230*m2^3 - 22955*m3^2 + 323925*m2*m4 + 19226*m6)*n_x^4 - 3*(340935*m2^3 - 11140*m3^2 + 85320*m2*m4 + 1852*m6)*n_x^3 + 12600*m2^3 + 15*(28922*m2^3 - 1742*m3^2 + 4995*m2*m4 - 32*m6)*n_x^2 - 150*(741*m2^3 - 43*m3^2 + 78*m2*m4)*n_x)*n_y^10 + (28*m6*n_x^13 - 63*(10*m2*m4 + 11*m6)*n_x^12 + 5*(576*m2^3 - 136*m3^2 + 2820*m2*m4 + 1537*m6)*n_x^11 - 3*(20530*m2^3 - 4940*m3^2 + 43455*m2*m4 + 15936*m6)*n_x^10 + 20*(24147*m2^3 - 5645*m3^2 + 31983*m2*m4 + 8748*m6)*n_x^9 - (1825185*m2^3 - 387500*m3^2 + 1796160*m2*m4 + 380917*m6)*n_x^8 + (3919605*m2^3 - 674440*m3^2 + 3029085*m2*m4 + 497096*m6)*n_x^7 - 10*(534498*m2^3 - 68534*m3^2 + 334389*m2*m4 + 40598*m6)*n_x^6 + 3*(1617715*m2^3 - 145750*m3^2 + 836190*m2*m4 + 74124*m6)*n_x^5 - 3*(987530*m2^3 - 53950*m3^2 + 405315*m2*m4 + 24981*m6)*n_x^4 + 3*(398010*m2^3 - 18450*m3^2 + 127455*m2*m4 + 3482*m6)*n_x^3 - 1800*m2^3 - 30*(10101*m2^3 - 945*m3^2 + 2875*m2*m4 - 16*m6)*n_x^2 + 1800*(24*m2^3 - 4*m3^2 + 7*m2*m4)*n_x)*n_y^9 + (8*m6*n_x^14 - 21*(10*m2*m4 + 11*m6)*n_x^13 + 10*(90*m2^3 - 16*m3^2 + 606*m2*m4 + 313*m6)*n_x^12 - 15*(1962*m2^3 - 396*m3^2 + 4853*m2*m4 + 1679*m6)*n_x^11 + (341055*m2^3 - 72850*m3^2 + 473205*m2*m4 + 124842*m6)*n_x^10 - (1825185*m2^3 - 387500*m3^2 + 1796160*m2*m4 + 380917*m6)*n_x^9 + 10*(514803*m2^3 - 101122*m3^2 + 404775*m2*m4 + 71676*m6)*n_x^8 - 2*(4304550*m2^3 - 704260*m3^2 + 2787855*m2*m4 + 415957*m6)*n_x^7 + (9135645*m2^3 - 1177210*m3^2 + 5047065*m2*m4 + 617964*m6)*n_x^6 - 15*(412382*m2^3 - 37856*m3^2 + 199731*m2*m4 + 20213*m6)*n_x^5 + 6*(421025*m2^3 - 18880*m3^2 + 176745*m2*m4 + 14778*m6)*n_x^4 - 30*(17697*m2^3 - 56*m3^2 + 6566*m2*m4 + 322*m6)*n_x^3 + 360*(95*m2^3 - 10*m3^2 + 50*m2*m4 - 2*m6)*n_x^2 + 1800*(m2^3 + m3^2 - m2*m4)*n_x)*n_y^8 + (m6*n_x^15 - 3*(10*m2*m4 + 11*m6)*n_x^14 + 5*(8*m3^2 + 252*m2*m4 + 111*m6)*n_x^13 - 3*(1430*m2^3 + 60*m3^2 + 7015*m2*m4 + 2023*m6)*n_x^12 + 10*(10701*m2^3 - 1274*m3^2 + 18657*m2*m4 + 4265*m6)*n_x^11 - 3*(317855*m2^3 - 51090*m3^2 + 323705*m2*m4 + 61741*m6)*n_x^10 + (3919605*m2^3 - 674440*m3^2 + 3029085*m2*m4 + 497096*m6)*n_x^9 - 2*(4304550*m2^3 - 704260*m3^2 + 2787855*m2*m4 + 415957*m6)*n_x^8 + 10*(1128189*m2^3 - 157958*m3^2 + 623832*m2*m4 + 86246*m6)*n_x^7 - 6*(1536940*m2^3 - 181865*m3^2 + 758565*m2*m4 + 95447*m6)*n_x^6 + 3*(1516295*m2^3 - 137200*m3^2 + 691365*m2*m4 + 81206*m6)*n_x^5 - 6*(191055*m2^3 - 5080*m3^2 + 78370*m2*m4 + 9368*m6)*n_x^4 + 360*(170*m2^3 + 65*m3^2 + 35*m2*m4 + 7*m6)*n_x^3 + 360*(45*m2^3 - 10*m3^2 + 20*m2*m4 + 2*m6)*n_x^2)*n_y^7 - (5*(9*m2^3 - 4*m3^2 - 15*m2*m4 - 2*m6)*n_x^14 - 15*(68*m2^3 - 44*m3^2 - 178*m2*m4 - 25*m6)*n_x^13 - 5*(828*m2^3 + 1076*m3^2 + 7419*m2*m4 + 1110*m6)*n_x^12 + (211845*m2^3 - 880*m3^2 + 280020*m2*m4 + 40877*m6)*n_x^11 - 2*(820815*m2^3 - 82160*m3^2 + 629655*m2*m4 + 82827*m6)*n_x^10 + 10*(534498*m2^3 - 68534*m3^2 + 334389*m2*m4 + 40598*m6)*n_x^9 - (9135645*m2^3 - 1177210*m3^2 + 5047065*m2*m4 + 617964*m6)*n_x^8 + 6*(1536940*m2^3 - 181865*m3^2 + 758565*m2*m4 + 95447*m6)*n_x^7 - 6*(951120*m2^3 - 109530*m3^2 + 440655*m2*m4 + 55292*m6)*n_x^6 + 6*(339000*m2^3 - 37240*m3^2 + 154035*m2*m4 + 19494*m6)*n_x^5 - 720*(435*m2^3 - 25*m3^2 + 185*m2*m4 + 27*m6)*n_x^4 - 1800*(m2^3 - 4*m3^2 + 4*m2*m4)*n_x^3)*n_y^6 + ((210*m2^3 - 80*m3^2 - 165*m2*m4 - 3*m6)*n_x^14 - 2*(1800*m2^3 - 915*m3^2 - 2040*m2*m4 - 131*m6)*n_x^13 + 2*(1110*m2^3 - 6335*m3^2 - 22425*m2*m4 - 2179*m6)*n_x^12 + (296160*m2^3 + 17320*m3^2 + 284895*m2*m4 + 29256*m6)*n_x^11 - 6*(323010*m2^3 - 18495*m3^2 + 186405*m2*m4 + 16976*m6)*n_x^10 + 3*(1617715*m2^3 - 145750*m3^2 + 836190*m2*m4 + 74124*m6)*n_x^9 - 15*(412382*m2^3 - 37856*m3^2 + 199731*m2*m4 + 20213*m6)*n_x^8 + 3*(1516295*m2^3 - 137200*m3^2 + 691365*m2*m4 + 81206*m6)*n_x^7 - 6*(339000*m2^3 - 37240*m3^2 + 154035*m2*m4 + 19494*m6)*n_x^6 + 3600*(142*m2^3 - 21*m3^2 + 70*m2*m4 + 9*m6)*n_x^5 - 1800*(26*m2^3 - 5*m3^2 + 15*m2*m4 + 2*m6)*n_x^4)*n_y^5 - (5*(63*m2^3 - 22*m3^2 - 36*m2*m4)*n_x^14 - (4545*m2^3 - 2370*m3^2 - 3540*m2*m4 - 97*m6)*n_x^13 + (1725*m2^3 - 14410*m3^2 - 32415*m2*m4 - 2318*m6)*n_x^12 + 5*(67131*m2^3 + 4442*m3^2 + 35886*m2*m4 + 2710*m6)*n_x^11 - 2*(826230*m2^3 - 22955*m3^2 + 323925*m2*m4 + 19226*m6)*n_x^10 + 3*(987530*m2^3 - 53950*m3^2 + 405315*m2*m4 + 24981*m6)*n_x^9 - 6*(421025*m2^3 - 18880*m3^2 + 176745*m2*m4 + 14778*m6)*n_x^8 + 6*(191055*m2^3 - 5080*m3^2 + 78370*m2*m4 + 9368*m6)*n_x^7 - 720*(435*m2^3 - 25*m3^2 + 185*m2*m4 + 27*m6)*n_x^6 + 1800*(26*m2^3 - 5*m3^2 + 15*m2*m4 + 2*m6)*n_x^5)*n_y^4 + (25*(6*m2^3 - 2*m3^2 - 3*m2*m4)*n_x^14 - 5*(369*m2^3 - 320*m3^2 - 324*m2*m4)*n_x^13 - (9825*m2^3 + 7750*m3^2 + 13680*m2*m4 + 684*m6)*n_x^12 + (306405*m2^3 + 5450*m3^2 + 70110*m2*m4 + 2934*m6)*n_x^11 - 3*(340935*m2^3 - 11140*m3^2 + 85320*m2*m4 + 1852*m6)*n_x^10 + 3*(398010*m2^3 - 18450*m3^2 + 127455*m2*m4 + 3482*m6)*n_x^9 - 30*(17697*m2^3 - 56*m3^2 + 6566*m2*m4 + 322*m6)*n_x^8 + 360*(170*m2^3 + 65*m3^2 + 35*m2*m4 + 7*m6)*n_x^7 + 1800*(m2^3 - 4*m3^2 + 4*m2*m4)*n_x^6)*n_y^3 - 15*(2*(10*m2^3 + 21*m3^2 + 10*m2*m4)*n_x^13 - (841*m2^3 + 112*m3^2 + 195*m2*m4 + 8*m6)*n_x^12 + (12901*m2^3 - 262*m3^2 + 1100*m2*m4 + 8*m6)*n_x^11 - (28922*m2^3 - 1742*m3^2 + 4995*m2*m4 - 32*m6)*n_x^10 + 2*(10101*m2^3 - 945*m3^2 + 2875*m2*m4 - 16*m6)*n_x^9 - 24*(95*m2^3 - 10*m3^2 + 50*m2*m4 - 2*m6)*n_x^8 - 24*(45*m2^3 - 10*m3^2 + 20*m2*m4 + 2*m6)*n_x^7)*n_y^2 + 75*((3*m2^3 + 2*m3^2)*n_x^13 - 2*(36*m2^3 + m3^2)*n_x^12 + (951*m2^3 - 14*m3^2 + 12*m2*m4)*n_x^11 - 2*(741*m2^3 - 43*m3^2 + 78*m2*m4)*n_x^10 + 24*(24*m2^3 - 4*m3^2 + 7*m2*m4)*n_x^9 + 24*(m2^3 + m3^2 - m2*m4)*n_x^8)*n_y)/(n_x^7*n_y^15 + (8*n_x^8 - 45*n_x^7 + 25*n_x^6 - 23*n_x^5 + 15*n_x^4 - 5*n_x^3)*n_y^14 + (28*n_x^9 - 315*n_x^8 + 1041*n_x^7 - 1035*n_x^6 + 968*n_x^5 - 582*n_x^4 + 250*n_x^3 - 95*n_x^2 + 15*n_x)*n_y^13 + (56*n_x^10 - 945*n_x^9 + 5746*n_x^8 - 15009*n_x^7 + 18194*n_x^6 - 17382*n_x^5 + 11102*n_x^4 - 4984*n_x^3 + 1765*n_x^2 - 360*n_x + 60)*n_y^12 + 60*n_x^12 + (70*n_x^11 - 1575*n_x^10 + 14015*n_x^9 - 61245*n_x^8 + 138114*n_x^7 - 174283*n_x^6 + 163238*n_x^5 - 107571*n_x^4 + 52507*n_x^3 - 20705*n_x^2 + 5460*n_x - 780)*n_y^11 - 780*n_x^11 + (56*n_x^12 - 1575*n_x^11 + 18570*n_x^10 - 115728*n_x^9 + 403823*n_x^8 - 803772*n_x^7 + 969776*n_x^6 - 834367*n_x^5 + 498284*n_x^4 - 209069*n_x^3 + 65785*n_x^2 - 12810*n_x + 780)*n_y^10 + 780*n_x^10 + (28*n_x^13 - 945*n_x^12 + 14015*n_x^11 - 115728*n_x^10 + 565900*n_x^9 - 1656392*n_x^8 + 2913840*n_x^7 - 3198552*n_x^6 + 2405947*n_x^5 - 1196804*n_x^4 + 378693*n_x^3 - 82370*n_x^2 + 10845*n_x + 660)*n_y^9 + 660*n_x^9 + (8*n_x^14 - 315*n_x^13 + 5746*n_x^12 - 61245*n_x^11 + 403823*n_x^10 - 1656392*n_x^9 + 4192900*n_x^8 - 6526559*n_x^7 + 6365599*n_x^6 - 4065108*n_x^5 + 1560742*n_x^4 - 280282*n_x^3 + 8290*n_x^2 + 1290*n_x - 840)*n_y^8 - 840*n_x^8 + (n_x^15 - 45*n_x^14 + 1041*n_x^13 - 15009*n_x^12 + 138114*n_x^11 - 803772*n_x^10 + 2913840*n_x^9 - 6526559*n_x^8 + 9021718*n_x^7 - 7773202*n_x^6 + 4204679*n_x^5 - 1200142*n_x^4 + 38980*n_x^3 + 54690*n_x^2 - 6960*n_x + 120)*n_y^7 + 120*n_x^7 + (25*n_x^14 - 1035*n_x^13 + 18194*n_x^12 - 174283*n_x^11 + 969776*n_x^10 - 3198552*n_x^9 + 6365599*n_x^8 - 7773202*n_x^7 + 5924490*n_x^6 - 2796336*n_x^5 + 645586*n_x^4 + 19470*n_x^3 - 30120*n_x^2 + 2520*n_x)*n_y^6 - (23*n_x^14 - 968*n_x^13 + 17382*n_x^12 - 163238*n_x^11 + 834367*n_x^10 - 2405947*n_x^9 + 4065108*n_x^8 - 4204679*n_x^7 + 2796336*n_x^6 - 1209662*n_x^5 + 272430*n_x^4 - 7920*n_x^3 - 2760*n_x^2)*n_y^5 + (15*n_x^14 - 582*n_x^13 + 11102*n_x^12 - 107571*n_x^11 + 498284*n_x^10 - 1196804*n_x^9 + 1560742*n_x^8 - 1200142*n_x^7 + 645586*n_x^6 - 272430*n_x^5 + 63840*n_x^4 - 3480*n_x^3)*n_y^4 - (5*n_x^14 - 250*n_x^13 + 4984*n_x^12 - 52507*n_x^11 + 209069*n_x^10 - 378693*n_x^9 + 280282*n_x^8 - 38980*n_x^7 - 19470*n_x^6 - 7920*n_x^5 + 3480*n_x^4)*n_y^3 - 5*(19*n_x^13 - 353*n_x^12 + 4141*n_x^11 - 13157*n_x^10 + 16474*n_x^9 - 1658*n_x^8 - 10938*n_x^7 + 6024*n_x^6 - 552*n_x^5)*n_y^2 + 15*(n_x^13 - 24*n_x^12 + 364*n_x^11 - 854*n_x^10 + 723*n_x^9 + 86*n_x^8 - 464*n_x^7 + 168*n_x^6)*n_y)
}
