% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Variance_approximations.R
\name{Var_approx}
\alias{Var_approx}
\title{Approximate the Variance of the Horvitz-Thompson estimator}
\usage{
Var_approx(y, pik, n, method, ...)
}
\arguments{
\item{y}{numeric vector containing the values of the variable of interest for
all population units}

\item{pik}{numeric vector of first-order inclusion probabilities, of length
equal to population size}

\item{n}{a scalar indicating the sample size}

\item{method}{string indicating the approximation that should be used.
One of "Hajek1", "Hajek2", "HartleyRao1", "HartleyRao2", "FixedPoint".}

\item{...}{two optional parameters can be modified to control the iterative
procedure in \code{method="FixedPoint"}: \code{maxIter} sets the maximum number
of iterations and \code{eps} controls the convergence error}
}
\value{
a scalar, the approximated variance.
}
\description{
Approximations of the Horvitz-Thompson variance for High-Entropy sampling designs.
Such methods use only first-order inclusion probabilities.
}
\details{
The variance approximations available in this function are described below,
the notation used is that of Matei and Tillé (2005).

\itemize{
    \item Hájek variance approximation (\code{method="Hajek1"}):

    \deqn{ \tilde{Var} = \sum_{i \in U} \frac{b_i}{\pi_i^2}(y_i - y_i^*)^2  }{
    \sum b (y - y*)^2 / (\pi^2)}
    where
    \deqn{y_i^* = \pi_i \frac{ \sum_{j\in U} b_j y_j/\pi_j }{ \sum_{j \in U} b_j } }{
    y* = \pi (\sum b*y/\pi) / (\sum b)}
    and
    \deqn{ b_i = \frac{ \pi_i(1-\pi_i)N }{ N-1 } }{ b= (N\pi(\-\pi)) / (N-1)}

    \item Starting from Hajék (1964), Brewer (2002) defined the following estimator
    (\code{method="Hajek2"}):

    \deqn{\tilde{Var} = \sum_{i \in U} \pi_i(1-\pi_i) \Bigl( \frac{y_i}{\pi_i} -
    \frac{\tilde{Y}}{n} \Bigr)^2 }{
    \sum \pi(1-\pi) ( y/\pi - Y*/n)^2 }
    where \eqn{\tilde{Y} = \sum_{i \in U} a_i y_i}{ Y* = \sum a*y }
    and \eqn{a_i = n(1-\pi_i)/\sum_{j \in U} \pi_j(1-\pi_j) }{ a = n(1-\pi) / \sum(\pi(1-\pi))}

    \item Hartley and Rao (1962) variance approximation (\code{method="HartleyRao1"}):



    \deqn{ \tilde{Var} =
    \sum_{i \in U} \pi_i \Bigl( 1 - \frac{n-1}{n}\pi_i \Bigr)
    \Biggr( \frac{y_i}{\pi_i} - \frac{Y}{n}  \Biggr)^2}{}

    \deqn{\qquad - \frac{n-1}{n^2} \sum_{i \in U} \Biggl( 2\pi_i^3 -
    \frac{\pi_i^2}{2} \sum_{j \in U} \pi_j^2 \Biggr)
    \Biggr( \frac{y_i}{\pi_i} - \frac{Y}{n}  \Biggr)^2}{}

    \deqn{\quad \qquad + \frac{2(n-1)}{n^3} \Biggl( \sum_{i \in U}\pi_i y_i
    - \frac{Y}{n}\sum_{i\in U} \pi_i^2 \Biggr)^2 }{*see pdf version of documentation*}



    \item Hartley and Rao (1962) provide a simplified version of the
    variance above (\code{method="HartleyRao2"}):

    \deqn{ \tilde{Var} = \sum_{i \in U} \pi_i \Bigl( 1 - \frac{n-1}{n}\pi_i \Bigr) \Biggr( \frac{y_i}{\pi_i} - \frac{Y}{n}  \Biggr)^2 }{
     Var = \sum \pi ( 1 - ( (n-1)/n )\pi )( y/\pi - Y/n )^2 }

     \item \code{method="FixedPoint"} computes the Fixed-Point variance approximation
     proposed by Deville and Tillé (2005).
     The variance can be expressed in the same form as in \code{method="Hajek1"},
     and the coefficients \eqn{b_i}{b} are computed iteratively by the algorithm:
     \enumerate{
         \item \deqn{b_i^{(0)} = \pi_i (1-\pi_i) \frac{N}{N-1}, \,\, \forall i \in U }{ b0 = \pi(1-\pi)(N/(N-1))}
         \item \deqn{ b_i^{(k)} = \frac{(b_i^{(k-1)})^2 }{\sum_{j\in U} b_j^{(k-1)} } + \pi_i(1-\pi_i) }{
         b(k) = [ b(i-1) ]^2 / [ \sum b(i-1) ] + \pi(1-\pi) }
      }
      a necessary condition for convergence is checked and, if not satisfied,
      the function returns an alternative solution that uses only one iteration:

      \deqn{b_i = \pi_i(1-\pi_i)\Biggl( \frac{N\pi_i(1-\pi_i)}{ (N-1)\sum_{j\in U}\pi_j(1-\pi_j) } + 1 \Biggr) }{
      b = \pi(1-\pi)( 1 + (N\pi(1-\pi)) / ( (N-1) \sum \pi(1-\pi) ) ) }

}
}
\examples{

N <- 500; n <- 50

set.seed(0)
x <- rgamma(n=N, scale=10, shape=5)
y <- abs( 2*x + 3.7*sqrt(x) * rnorm(N) )

pik  <- n * x/sum(x)
pikl <- outer(pik, pik, '*'); diag(pikl) <- pik

### Variance approximations ---
Var_approx(y, pik, n, method = "Hajek1")
Var_approx(y, pik, n, method = "Hajek2")
Var_approx(y, pik, n, method = "HartleyRao1")
Var_approx(y, pik, n, method = "HartleyRao2")
Var_approx(y, pik, n, method = "FixedPoint")


}
\references{
Matei, A.; Tillé, Y., 2005. Evaluation of variance approximations and estimators
in maximum entropy sampling with unequal probability and fixed sample size.
Journal of Official Statistics 21 (4), 543-570.
}
