% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tmvnorm.R
\name{tmvnorm}
\alias{tmvnorm}
\title{Multivariate truncated normal distribution}
\arguments{
\item{n}{number of observations}

\item{x, q}{vector of quantiles}

\item{B}{number of replications for the (quasi)-Monte Carlo scheme}

\item{log}{logical; if \code{TRUE}, probabilities and density are given on the log scale.}

\item{mu}{vector of location parameters}

\item{sigma}{covariance matrix}

\item{lb}{vector of lower truncation limits}

\item{ub}{vector of upper truncation limits}

\item{type}{string, either of \code{mc} or \code{qmc} for Monte Carlo and quasi Monte Carlo, respectively}
}
\value{
\code{dtmvnorm} gives the density, \code{ptmvnorm} and \code{pmvnorm} give the distribution function of respectively the truncated and multivariate Gaussian distribution and \code{rtmvnorm} generate random deviates.
}
\description{
Density, distribution function and random generation for the multivariate truncated normal distribution
with mean vector \code{mu}, covariance matrix \code{sigma}, lower truncation limit \code{lb} and upper truncation limit \code{ub}. 
The truncation limits can include infinite values. The Monte Carlo (\code{type = "mc"}) uses a sample of size \code{B}, while the
quasi Monte Carlo (\code{type = "qmc"}) uses a pointset of size \code{ceiling(n/12)} and estimates the relative error using 12 independent randomized QMC estimators.
}
\section{Usage}{
 \preformatted{
dtmvnorm(x, mu, sigma, lb, ub, type = c("mc", "qmc"), log = FALSE, B = 1e4)
ptmvnorm(q, mu, sigma, lb, ub, type = c("mc", "qmc"), log = FALSE, B = 1e4)
rtmvnorm(n, mu, sigma, lb, ub)}
}

\examples{
d <- 4; lb <- rep(0, d)
mu <- runif(d)
sigma <- matrix(0.5, d, d) + diag(0.5, d)
samp <- rtmvnorm(n = 10, mu = mu, sigma = sigma, lb = lb)
loglik <- dtmvnorm(samp, mu = mu, sigma = sigma, lb = lb, log = TRUE)
cdf <- ptmvnorm(samp, mu = mu, sigma = sigma, lb = lb, log = TRUE, type = "q")

# Exact Bayesian Posterior Simulation Example
# Vignette, example 5
\dontrun{
data("lupus"); # load lupus data
Y <- lupus[,1]; # response data
X <- as.matrix(lupus[,-1])  # construct design matrix
n <- nrow(X)
d <- ncol(X)
X <- diag(2*Y-1) \%*\% X; # incorporate response into design matrix
nusq <- 10000; # prior scale parameter
C <- solve(diag(d)/nusq + crossprod(X))
sigma <- diag(n) + nusq*tcrossprod(X) # this is covariance of Z given beta
est <- pmvnorm(sigma = sigma, lb = 0) 
# estimate acceptance probability of crude Monte Carlo
print(attributes(est)$upbnd/est[1])
# reciprocal of acceptance probability
Z <- rtmvnorm(sigma = sigma, n = 1e3, lb = rep(0, n))
# sample exactly from auxiliary distribution
beta <- rtmvnorm(n = nrow(Z), sigma = C) + Z \%*\% X \%*\% C
# simulate beta given Z and plot boxplots of marginals
boxplot(beta, ylab = expression(beta))
# output the posterior means
colMeans(beta)
}
}
\references{
Z. I. Botev (2017), \emph{The normal law under linear restrictions:
simulation and estimation via minimax tilting}, Journal of the Royal
Statistical Society, Series B, \bold{79} (1), pp. 1--24.
}
\author{
Zdravko I. Botev, Leo Belzile (wrappers)
}
