% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Support.R
\name{SplitFrequency}
\alias{SplitFrequency}
\alias{SplitNumber}
\alias{ForestSplits}
\alias{TreeSplits}
\title{Frequency of splits}
\usage{
SplitFrequency(reference, forest)

SplitNumber(tips, tree, tipIndex, powersOf2)

ForestSplits(forest, powersOf2)

TreeSplits(tree)
}
\arguments{
\item{reference}{A tree of class \code{phylo}, a \code{Splits} object.}

\item{forest}{a list of trees of class \code{phylo}, or a \code{multiPhylo} object; or a
\code{Splits} object. See
\href{https://ms609.github.io/TreeTools/articles/load-trees.html}{vignette} for
possible methods of loading trees into R.}

\item{tips}{Integer vector specifying the tips of the tree within the chosen
split.}

\item{tree}{A tree of class \code{\link[ape:read.tree]{phylo}}.}

\item{tipIndex}{Character vector of tip names, in a fixed order.}

\item{powersOf2}{Integer vector of same length as \code{tipIndex}, specifying a
power of 2 to be associated with each tip in turn.}
}
\value{
\code{SplitFrequency()} returns the number of trees in \code{forest} that
contain each split in \code{reference}.
If \code{reference} is a tree of class \code{phylo}, then the sequence will correspond
to the order of nodes (use \code{ape::nodelabels()} to view).
Note that the three nodes at the root of the tree correspond to a single
split; see the example for how these might be plotted on a tree.
}
\description{
\code{SplitFrequency()} provides a simple way to count the number of times that
bipartition splits, as defined by a reference tree, occur in a forest of
trees. May be used to calculate edge ("node") support for majority consensus
or bootstrap trees.
}
\details{
If multiple calculations are required, some time can be saved by using the
constituent functions (see examples)
}
\section{Functions}{
\itemize{
\item \code{SplitNumber()}: Assign a unique integer to each split

\item \code{ForestSplits()}: Frequency of splits in a given forest of trees

\item \code{TreeSplits()}: Deprecated. Listed the splits in a given tree.
Use as.Splits instead.

}}
\examples{
# An example forest of 100 trees, some identical
forest <- as.phylo(c(1, rep(10, 79), rep(100, 15), rep(1000, 5)), nTip = 9)

# Generate an 80\% consensus tree
cons <- ape::consensus(forest, p = 0.8)
plot(cons)

# Calculate split frequencies
splitFreqs <- SplitFrequency(cons, forest)

# Optionally, colour edges by corresponding frequency.
# Note that not all edges are associated with a unique split
# (and two root edges may be associated with one split - not handled here)
edgeSupport <- rep(1, nrow(cons$edge)) # Initialize trivial splits to 1
childNode <- cons$edge[, 2]
edgeSupport[match(names(splitFreqs), childNode)] <- splitFreqs / 100

plot(cons, edge.col = SupportColour(edgeSupport), edge.width = 3)

# Annotate nodes by frequency 
LabelSplits(cons, splitFreqs, unit = "\%",
            col = SupportColor(splitFreqs / 100),
            frame = "none", pos = 3L)

}
\seealso{
Other Splits operations: 
\code{\link{LabelSplits}()},
\code{\link{NSplits}()},
\code{\link{NTip}()},
\code{\link{PolarizeSplits}()},
\code{\link{Splits}},
\code{\link{SplitsInBinaryTree}()},
\code{\link{TipLabels}()},
\code{\link{TipsInSplits}()},
\code{\link{match.Splits}},
\code{\link{xor}()}
}
\author{
\href{https://orcid.org/0000-0001-5660-1727}{Martin R. Smith}
(\href{mailto:martin.smith@durham.ac.uk}{martin.smith@durham.ac.uk})
}
\concept{Splits operations}
