% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/EstStempCens.R
\name{EstStempCens}
\alias{EstStempCens}
\title{ML estimation in spatio-temporal model with censored/missing responses}
\usage{
EstStempCens(
  y,
  x,
  cc,
  time,
  coord,
  LI,
  LS,
  init.phi,
  init.rho,
  init.tau2,
  tau2.fixo = FALSE,
  type.Data = "balanced",
  method = "nlminb",
  kappa = 0,
  type.S = "exponential",
  IMatrix = TRUE,
  lower.lim = c(0.01, -0.99, 0.01),
  upper.lim = c(30, 0.99, 20),
  M = 20,
  perc = 0.25,
  MaxIter = 300,
  pc = 0.2,
  error = 1e-06
)
}
\arguments{
\item{y}{a vector of responses.}

\item{x}{a matrix or vector of covariates.}

\item{cc}{a vector of censoring indicators. For each observation: \code{1} if censored/missing and \code{0} if non-censored/non-missing.}

\item{time}{a vector of time.}

\item{coord}{a matrix of coordinates of the spatial locations.}

\item{LI}{lower limit of detection. For each observation: if non-censored/non-missing \code{=y}, if left-censored/missing \code{=-Inf} or \code{=LOD} if right/interval-censored.}

\item{LS}{upper limit of detection. For each observation: if non-censored/non-missing \code{=y}, if right-censored/missing \code{=Inf} or \code{=LOD} if left/interval-censored.}

\item{init.phi}{initial value of the spatial scaling parameter.}

\item{init.rho}{initial value of the time scaling parameter.}

\item{init.tau2}{initial value of the the nugget effect parameter.}

\item{tau2.fixo}{\code{TRUE} or \code{FALSE}. Indicate if the nugget effect (\eqn{\tau^2}) parameter must be fixed.
By default = \code{FALSE}.}

\item{type.Data}{type of the data: '\code{balanced}' for balanced data and '\code{unbalanced}' for unbalanced data. By default = \code{balanced}.}

\item{method}{optimization method used to estimate (\eqn{\phi}, \eqn{\rho} and \eqn{\tau^2}):
'\code{optim}' for the function \code{\link[stats]{optim}} and '\code{nlminb}' for the function \code{\link[stats]{nlminb}}.
By default = \code{nlminb}.}

\item{kappa}{parameter for all spatial covariance functions. In the case of exponential, gaussian and spherical function \eqn{\kappa} is equal to zero.
For the power exponential function \eqn{\kappa} is a number between 0 and 2. For the matern correlation function is upper than 0.}

\item{type.S}{type of spatial correlation function: '\code{exponential}' for exponential, '\code{gaussian}' for gaussian,
'\code{matern}' for matern, '\code{pow.exp}' for power exponential and '\code{spherical}' for spherical function, respectively.
Default is \code{exponential} function.}

\item{IMatrix}{\code{TRUE} or \code{FALSE}. Indicate if the observed information matrix will be computed. By default = \code{TRUE}.}

\item{lower.lim, upper.lim}{vectors of lower and upper bounds for the optimization method.
If unspecified, the default is \code{c(0.01,-0.99,0.01)} for the lower bound and \code{c(30,0.99,20)} for the upper bound if tau2.fixo=\code{FALSE}.}

\item{M}{number of Monte Carlo samples for stochastic approximation. By default = \code{20}.}

\item{perc}{percentage of burn-in on the Monte Carlo sample. By default = \code{0.25}.}

\item{MaxIter}{the maximum number of iterations of the SAEM algorithm. By default = \code{300}.}

\item{pc}{percentage of iterations of the SAEM algorithm with no memory. By default = \code{0.20}.}

\item{error}{the convergence maximum error. By default = \code{1e-6}.}
}
\value{
The function returns an object of class \code{Est.StempCens} which is a list given by:

\describe{
  \item{\code{m.data}}{Returns a list with all data components given in input.}
  \item{\code{m.results}}{A list given by:}
}
  \item{theta}{final estimation of \eqn{\theta = (\beta, \sigma^2, \tau^2, \phi, \rho)}.}
  \item{Theta}{estimated parameters in all iterations, \eqn{\theta = (\beta, \sigma^2, \tau^2, \phi, \rho)}.}
  \item{beta}{estimated \eqn{\beta}.}
  \item{sigma2}{estimated \eqn{\sigma^2}.}
  \item{tau2}{estimated \eqn{\tau^2}.}
  \item{phi}{estimated \eqn{\phi}.}
  \item{rho}{estimated \eqn{\rho}.}
  \item{Eff.range}{estimated effective range.}
  \item{PsiInv}{estimated \eqn{\Psi^-1}, where \eqn{\Psi=\Sigma/\sigma^2}.}
  \item{Cov}{estimated \eqn{\Sigma}.}
  \item{SAEMy}{stochastic approximation of the first moment for the truncated normal distribution.}
  \item{SAEMyy}{stochastic approximation of the second moment for the truncated normal distribution.}
  \item{Hessian}{Hessian matrix, the negative of the conditional expected second derivative matrix given the observed values.}
  \item{Louis}{the observed information matrix using the Louis' method.}
  \item{loglik}{log likelihood for SAEM method.}
  \item{AIC}{Akaike information criteria.}
  \item{BIC}{Bayesian information criteria.}
  \item{AICcorr}{corrected AIC by the number of parameters.}
  \item{iteration}{number of iterations needed to convergence.}
}
\description{
Return the maximum likelihood estimates of the unknown parameters of spatio-temporal model with censored/missing responses.
The estimates are obtained using SAEM algorithm.
The function also computes the observed information matrix using the method developed by \insertCite{louis1982finding;textual}{StempCens}.
The types of censoring considered are left, right, interval or missing values.
}
\details{
The spatio-temporal Gaussian model is giving by:

\eqn{ Y(s_i,t_j)= \mu(s_i,t_j)+ Z(s_i,t_j) +  \epsilon(s_i,t_j),}

where the deterministic term \eqn{\mu(s_i,t_j)} and the stochastic terms \eqn{Z(s_i,t_j)},
\eqn{\epsilon(s_i,t_j)} can depend on the observed spatio-temporal indexes for \eqn{Y(s_i,t_j)}.
We assume \eqn{Z} is normally distributed with zero-mean and covariance matrix \eqn{\Sigma_z = \sigma^2 \Omega_{\phi\rho}},
where \eqn{\sigma^2} is the partial sill, \eqn{\Omega_{\phi\rho}} is the spatio-temporal correlation matrix,\eqn{\phi}
and \eqn{\rho} are the spatial and time scaling parameters; \eqn{\epsilon(s_i,t_j)} is an independent and
identically distributed measurement error with \eqn{E[\epsilon(s_i,t_j)]=0}, variance
\eqn{Var[\epsilon(s_i,t_j)]=\tau^2} (the nugget effect) and \eqn{Cov[\epsilon(s_i,t_j), \epsilon(s_k,t_l)]=0}
for all \eqn{s_i =! s_k} or \eqn{t_j =! t_l}.

In particular, we define \eqn{\mu(s_i,t_j)}, the mean of the stochastic process as

\eqn{\mu(s_i,t_j)=\sum_{k=1}^{p} x_k(s_i,t_j)\beta_k,}

where \eqn{x_1(s_i,t_j),..., x_p(s_i,t_j)} are known functions of \eqn{(s_i,t_j)}, and \eqn{\beta_1,...,\beta_p}
are unknown parameters to be estimated. Equivalently, in matrix notation, we have the spatio-temporal linear model as follows:

\eqn{Y = X \beta + Z + \epsilon,}

\eqn{Z ~ N(0,\sigma^2 \Omega_{\phi\rho}),}

\eqn{\epsilon ~ N(0,\tau^2 I_m).}

Therefore the spatio-temporal process, \eqn{Y}, has normal distribution with mean \eqn{E[Y]=X\beta} and
variance \eqn{\Sigma=\sigma^2\Omega_{\phi\rho}+\tau^2 I_m}. We assume that \eqn{\Sigma} is non-singular
and \eqn{X} has full rank.

The estimation process was computed via SAEM algorithm initially proposed by \insertCite{delyon1999convergence;textual}{StempCens}.
}
\examples{
\dontrun{
# Simulating data
# Initial parameter values
beta <- c(-1,1.50)
phi <- 5;     rho <- 0.45
tau2 <- 0.80; sigma2 <- 1.5
n1 <- 5    # Number of spatial locations
n2 <- 5    # Number of temporal index
set.seed(1000)
x.coord <- round(runif(n1,0,10),9)    # X coordinate
y.coord <- round(runif(n1,0,10),9)    # Y coordinate
coord  <- cbind(x.coord,y.coord)      # Cartesian coordinates without repetitions
coord2 <- cbind(rep(x.coord,each=n2),rep(y.coord,each=n2)) # Cartesian coordinates with repetitions
time <- as.matrix(seq(1,n2))          # Time index without repetitions
time2 <- as.matrix(rep(time,n1))      # Time index with repetitions
x1 <- rexp(n1*n2,2)
x2 <- rnorm(n1*n2,2,1)
x  <- cbind(x1,x2)
media <- x\%*\%beta
# Covariance matrix
Ms  <- as.matrix(dist(coord))   # Spatial distances
Mt  <- as.matrix(dist(time))    # Temporal distances
Cov <- CovarianceM(phi,rho,tau2,sigma2,Ms,Mt,1.5,"matern")
# Data
require(mvtnorm)
y <- as.vector(rmvnorm(1,mean=as.vector(media),sigma=Cov))
perc <- 0.20
aa <- sort(y); bb <- aa[1:(perc*n1*n2)]; cutof <- bb[perc*n1*n2]
cc <- matrix(1,(n1*n2),1)*(y<=cutof)
y[cc==1] <- cutof
LI <- y; LI[cc==1] <- -Inf    # Left-censored
LS <- y

# Estimation
est_teste <- EstStempCens(y, x, cc, time2, coord2, LI, LS, init.phi=3.5,
                 init.rho=0.5, init.tau2=0.7,tau2.fixo=FALSE, kappa=1.5,
                 type.S="matern", IMatrix=TRUE, M=20, perc=0.25,
                 MaxIter=300, pc=0.2)}

}
\author{
Katherine L. Valeriano, Victor H. Lachos and Larissa A. Matos
}
