# Generated by using Rcpp::compileAttributes() -> do not edit by hand
# Generator token: 10BE3573-1514-4C36-9D1C-5A225CD40393

#' Heuristic Product Within Distance (Spatially Balanced Sampling Design)
#'
#' Selects spatially balanced samples through the use of
#' Heuristic Product Within Distance design (HPWD). To have constant inclusion
#' probabilities \eqn{\pi_{i}=nsamp/N}, where \eqn{nsamp} is sample size
#' and \eqn{N} is population size, the distance matrix has to be standardized
#' with function \code{\link{stprod}}.
#'
#' The HPWD design generates samples approximately with the same
#' probabilities of the \code{\link{pwd}} but with a significantly smaller
#' number of steps. In fact, this algorithm randomly selects a sample of size
#' \eqn{n} exactly with \eqn{n} steps, updating at each step the selection
#' probability of not-selected units, depending on their distance from the
#' units that were already selected in the previous steps.
#'
#' @param dis A distance matrix NxN that specifies how far all the pairs
#' of units in the population are.
#' @param nsamp Sample size.
#' @param beta Parameter \eqn{\beta} for the algorithm. The higher
#' \eqn{\beta} is, the more the sample is going to be spread (default = 10).
#' @param nrepl Number of samples to draw (default = 1).
#' @return Returns a matrix \code{nrepl} x \code{nsamp}, which contains the
#' \code{nrepl} selected samples, each of them stored in a row. In particular,
#' the i-th row contains all labels of units selected in the i-th sample.
#' @references
#' Benedetti R, Piersimoni F (2017). A spatially balanced design with
#' probability function proportional to the within sample distance.
#' \emph{Biometrical Journal}, \strong{59}(5), 1067-1084.
#' \url{https://doi.org/10.1002/bimj.201600194}
#'
#' Benedetti R, Piersimoni F (2017). Fast Selection of Spatially Balanced Samples. \emph{arXiv}.
#' \url{https://arxiv.org/abs/1710.09116}
#' @examples
#' # Example 1
#' # Draw 1 sample of dimension 10 without constant inclusion probabilities
#' dis <- as.matrix(dist(cbind(lucas_abruzzo$x, lucas_abruzzo$y))) # distance matrix
#' s <- hpwd(dis = dis, nsamp = 10) # drawn sample
#' \donttest{
#' # Example 2
#' # Draw 1 sample of dimension 15 with constant inclusion probabilities
#' # equal to nsamp/N, with N = population size
#' dis <- as.matrix(dist(cbind(lucas_abruzzo$x, lucas_abruzzo$y))) # distance matrix
#' con <- rep(1, nrow(dis)) # vector of constraints
#' stand_dist <- stprod(mat = dis, vec = con) # standardized matrix
#' s <- hpwd(dis = stand_dist, nsamp = 15) # drawn sample
#'
#' # Example 3
#' # Draw 2 samples of dimension 15 with constant inclusion probabilities
#' # equal to nsamp/N, with N = population size, and an increased level of spread, beta = 20
#' dis <- as.matrix(dist(cbind(lucas_abruzzo$x, lucas_abruzzo$y))) # distance matrix
#' con <- rep(0, nrow(dis)) # vector of constraints
#' stand_dist <- stprod(mat = dis, vec = con) # standardized matrix
#' s <- hpwd(dis = stand_dist, nsamp = 15, beta = 20, nrepl = 2) # drawn samples
#' }
#' @export
hpwd <- function(dis, nsamp, beta = 10, nrepl = 1L) {
    .Call(`_Spbsampling_hpwd`, dis, nsamp, beta, nrepl)
}

#' Product Within Distance (Spatially Balanced Sampling Design)
#'
#' Selects spatially balanced samples through the use of the
#' Product Within Distance design (PWD). To have constant inclusion
#' probabilities \eqn{\pi_{i}=nsamp/N}, where \eqn{nsamp} is sample size and
#' \eqn{N} is population size, the distance matrix has to be standardized with
#' function \code{\link{stprod}}.
#'
#' @param dis A distance matrix NxN that specifies how far all the pairs
#' of units in the population are.
#' @param nsamp Sample size.
#' @param beta Parameter \eqn{\beta} for the algorithm. The higher
#' \eqn{\beta} is, the more the sample is going to be spread (default = 10).
#' @param nrepl Number of samples to draw (default = 1).
#' @param niter Number of iterations for the algorithm. More iterations are
#' better but require more time. Usually 10 is very efficient (default = 10).
#' @return Returns a matrix \code{nrepl} x \code{nsamp}, which contains the
#' \code{nrepl} selected samples, each of them stored in a row. In particular,
#' the i-th row contains all labels of units selected in the i-th sample.
#' @references
#' Benedetti R, Piersimoni F (2017). A spatially balanced design with
#' probability function proportional to the within sample distance.
#' \emph{Biometrical Journal}, \strong{59}(5), 1067-1084.
#' \url{https://doi.org/10.1002/bimj.201600194}
#' @examples
#' # Example 1
#' # Draw 1 sample of dimension 15 without constant inclusion probabilities
#' dis <- as.matrix(dist(cbind(lucas_abruzzo$x, lucas_abruzzo$y))) # distance matrix
#' s <- pwd(dis = dis, nsamp = 15)  # drawn sample
#' \donttest{
#' # Example 2
#' # Draw 1 sample of dimension 15 with constant inclusion probabilities
#' # equal to nsamp/N, with N = population size
#' dis <- as.matrix(dist(cbind(lucas_abruzzo$x, lucas_abruzzo$y))) # distance matrix
#' con <- rep(0, nrow(dis)) # vector of constraints
#' stand_dist <- stprod(mat = dis, vec = con) # standardized matrix
#' s <- pwd(dis = stand_dist, nsamp = 15)  # drawn sample
#'
#' # Example 3
#' # Draw 2 samples of dimension 15 with constant inclusion probabilities
#' # equal to nsamp/N, with N = population size, and an increased level of spread, beta = 20
#' dis <- as.matrix(dist(cbind(lucas_abruzzo$x, lucas_abruzzo$y))) # distance matrix
#' con <- rep(0, nrow(dis)) # vector of constraints
#' stand_dist <- stprod(mat = dis, vec = con) # standardized matrix
#' s <- pwd(dis = stand_dist, nsamp = 15, beta = 20, nrepl = 2)  # drawn samples
#' }
#' @export
pwd <- function(dis, nsamp, beta = 10, nrepl = 1L, niter = 10L) {
    .Call(`_Spbsampling_pwd`, dis, nsamp, beta, nrepl, niter)
}

#' Spatial Balance Index
#'
#' Computes the Spatial Balance Index (SBI), which is a measure of
#' spatial balance of a sample. The lower it is, the better the spread.
#'
#' The SBI is based on Voronoi polygons. Given a sample s, each unit \eqn{i}
#' in the sample has its own Voronoi polygon, which is composed by all
#' population units closer to \eqn{i} than to any other sample unit \eqn{j}.
#' Then, per each Voronoi polygon, define \eqn{v_{i}} as the sum of the
#' inclusion probabilities of all units in the \eqn{i}-th Voronoi polygon.
#' Finally, the variance of \eqn{v_{i}} is the SBI.
#'
#' @param dis A distance matrix NxN that specifies how far all the pairs
#' of units in the population are.
#' @param pi A vector of first order inclusion probabilities of the units
#' of the population.
#' @param s A vector of labels of the sample.
#' @return Returns the Spatial Balance Index.
#' @references
#' Stevens DL, Olsen AR (2004). Spatially Balanced Sampling of Natural Resources.
#' \emph{Journal of the American Statistical Association}, \strong{99}(465), 262-278.
#' \url{https://doi.org/10.1198/016214504000000250}
#' @examples
#' \dontshow{
#' d <- matrix(runif(200), 100, 2)
#' dis <- as.matrix(dist(d))
#' pi <- rep(10 / 100, 100)
#' s <- sample(1:100,10)
#' sbi(dis = dis, pi = pi, s = s)
#' }
#' \donttest{
#' dis <- as.matrix(dist(cbind(simul1$x, simul1$y))) # distance matrix
#' con <- rep(0, nrow(dis)) # vector of constraints
#' stand_dist <- stprod(mat = dis, vec = con) # standardized matrix
#' pi <- rep(100 / nrow(dis), nrow(dis)) # vector of probabilities inclusion
#' s <- pwd(dis = stand_dist, nsamp = 100) # sample
#' sbi(dis = dis, pi = pi, s = s)
#' }
#' @importFrom stats var
#' @export
sbi <- function(dis, pi, s) {
    .Call(`_Spbsampling_sbi`, dis, pi, s)
}

#' Standardize a symmetric matrix (distances) to fixed row (column) products
#'
#' Standardizes a distance matrix to fixed rows and columns
#' products. The function iteratively constrains a logarithmic transformed
#' matrix to know products, and in order to keep the symmetry of the matrix,
#' at each iteration performs an average with its transpose. When the known
#' products are all equal to a constant (e.g. 0), this method provides a
#' simple and accurate way to scale a distance matrix to a doubly stochastic
#' matrix.
#'
#' The standardized matrix will not be affected by problems arising from units
#' with different inclusion probabilities caused by undesired features of the
#' spatial distribution of the population, as edge effects and/or isolated
#' points.
#'
#' @param mat A distance matrix size NxN.
#' @param con A vector of row (column) constraints.
#' @param differ A scalar with the maximum accepted difference with the constraint (default = 1e-15).
#' @param niter An integer with the maximum number of iterations (default = 1000).
#' @return Returns a standardized distance matrix of size NxN.
#' @references
#' Benedetti R, Piersimoni F (2017). A spatially balanced design with
#' probability function proportional to the within sample distance.
#' \emph{Biometrical Journal}, \strong{59}(5), 1067-1084.
#' \url{https://doi.org/10.1002/bimj.201600194}
#' @examples
#' \dontshow{
#' d <- matrix(runif(200), 100, 2)
#' dis <- as.matrix(dist(d))
#' con <- rep(0, nrow(dis))
#' stand_dist <- stprod(mat = dis, con = con)
#' }
#' \donttest{
#' dis <- as.matrix(dist(cbind(simul1$x, simul1$y))) # distance matrix
#' con <- rep(0, nrow(dis)) # vector of constraints
#' stand_dist <- stprod(mat = dis, con = con) # standardized matrix
#' }
#' @export
stprod <- function(mat, con, differ = 1e-15, niter = 1000L) {
    .Call(`_Spbsampling_stprod`, mat, con, differ, niter)
}

#' Standardize a symmetric matrix (distances) to fixed row (column) totals
#'
#' Standardizes a distance matrix to fixed rows and columns
#' totals. The function iteratively constrains the rows sums of the matrix
#' to know totals, and in order to keep the symmetry of the matrix, at each
#' iteration performs an average with its transpose. When the known totals are
#' all equal to a constant (e.g. 1), this method provides a simple and
#' accurate way to scale a distance matrix to a doubly stochastic matrix.
#'
#' The standardized matrix will not be affected by problems arising from units
#' with different inclusion probabilities caused by undesired features of the
#' spatial distribution of the population, as edge effects and/or isolated
#' points.
#'
#' @param  mat A distance matrix size NxN.
#' @param  con A vector of row (column) constraints.
#' @param differ A scalar with the maximum accepted difference with the constraint (default = 1e-15).
#' @param niter An integer with the maximum number of iterations (default = 1000).
#' @return Returns a standardized distance matrix of size NxN.
#' @references
#' Benedetti R, Piersimoni F (2017). A spatially balanced design with
#' probability function proportional to the within sample distance.
#' \emph{Biometrical Journal}, \strong{59}(5), 1067-1084.
#' \url{https://doi.org/10.1002/bimj.201600194}
#' @examples
#' dis <- as.matrix(dist(cbind(simul2$x, simul2$y))) # distance matrix
#' con <- rep(1, nrow(dis)) # vector of constraints
#' stand_dist <- stsum(mat = dis, con = con) # standardized matrix
#' @export
stsum <- function(mat, con, differ = 1e-15, niter = 1000L) {
    .Call(`_Spbsampling_stsum`, mat, con, differ, niter)
}

#' Sum Within Distance (Spatially Balanced Sampling Design)
#'
#' Selects spatially balanced samples through the use of the
#' Sum Within Distance design (SWD). To have a constant inclusion
#' probabilities \eqn{\pi_{i}=nsamp/N}, where \eqn{nsamp} is sample size and
#' \eqn{N} is population size, the distance matrix has to be standardized with
#' function \code{\link{stsum}}.
#'
#' @param dis A distance matrix NxN that specifies how far all the pairs
#' of units in the population are.
#' @param nsamp Sample size.
#' @param beta Parameter \eqn{\beta} for the algorithm. The higher
#' \eqn{\beta} is, the more the sample is going to be spread.
#' @param nrepl Number of samples to draw (default = 1).
#' @param niter Number of iterations for the algorithm. More iterations are
#' better but require more time. Usually 10 is very efficient (default = 10).
#' @return Returns a matrix \code{nrepl} x \code{nsamp}, which contains the
#' \code{nrepl} selected samples, each of them stored in a row. In particular,
#' the i-th row contains all labels of units selected in the i-th sample.
#' @references
#' Benedetti R, Piersimoni F (2017). A spatially balanced design with
#' probability function proportional to the within sample distance.
#' \emph{Biometrical Journal}, \strong{59}(5), 1067-1084.
#' \url{https://doi.org/10.1002/bimj.201600194}
#' @examples
#' # Example 1
#' # Draw 1 sample of dimension 15 without constant inclusion probabilities
#' dis <- as.matrix(dist(cbind(income_emilia$x_coord, income_emilia$y_coord))) # distance matrix
#' s <- swd(dis = dis, nsamp = 15)  # drawn sample
#' \donttest{
#' # Example 2
#' # Draw 1 sample of dimension 15 with constant inclusion probabilities
#' # equal to nsamp/N, with N = population size
#' dis <- as.matrix(dist(cbind(income_emilia$x_coord,income_emilia$y_coord))) # distance matrix
#' con <- rep(1, nrow(dis)) # vector of constraints
#' stand_dist <- stsum(mat = dis, con = vec) # standardized matrix
#' s <- swd(dis = stand_dist, nsamp = 15)  # drawn sample
#'
#' # Example 3
#' # Draw 2 samples of dimension 15 with constant inclusion probabilities
#' # equal to nsamp/N, with N = population size and an increased level of spread, i.e. beta = 20
#' dis <- as.matrix(dist(cbind(income_emilia$x_coord,income_emilia$y_coord))) # distance matrix
#' con <- rep(1, nrow(dis)) # vector of constraints
#' stand_dist <- stsum(mat = dis, con = vec) # standardized matrix
#' s <- swd(dis = stand_dist, nsamp = 15, beta = 20, nrepl = 2)  # drawn samples
#' }
#' @export
swd <- function(dis, nsamp, beta = 10, nrepl = 1L, niter = 10L) {
    .Call(`_Spbsampling_swd`, dis, nsamp, beta, nrepl, niter)
}

