% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/filter_global.R
\name{globalPeaksFilter}
\alias{globalPeaksFilter}
\title{Reference similarity based peak selection.}
\usage{
globalPeaksFilter(
  msiData,
  referenceImage,
  method = "pearson",
  threshold = NULL,
  verbose = TRUE
)
}
\arguments{
\item{msiData}{\link{msi.dataset-class} object. See \link{msiDataset}.}

\item{referenceImage}{\link{ms.image-class} object. Reference image used
to calculate the similarity values.}

\item{method}{method used to calculate the similariry between the peak
intensities and the reference image. Accepted values are:
\itemize{
   \item \code{pearson}: Pearson's correlation
   \item \code{spearman}: Spearman's correlation
   \item \code{ssim}: structural similarity index measure
   \item \code{nmi}: normalized mutual information.
}}

\item{threshold}{numeric (default = 0, default = 0.001 (SSIM)). The threshold
applied to the similarity values between the peaks images and the reference
image. The default value of 0 guarantees that only the ions with a positive
similarity with the reference image (typically representing the spatial
distribution of the signal source) are retrieved. For consistency, the NMI are
scaled in [-1, 1] to match the same range of correlations.}

\item{verbose}{logical (default = \code{TRUE}). Additional output text.}
}
\value{
\code{peak.filter} object. See link{applyPeaksFilter}.
}
\description{
\code{globalPeaksFilter} returns a list of peaks selected by their similarity
with a reference image.
}
\details{
A filter based on the similarity between the peak signals and a reference
signal. The reference signal, passed as an \code{\link{ms.image-class}} object,
can be calculated using the \code{\link{refAndROIimages}} function. Both continuous
and binary references can be passed. The filter then calculates the similarity
between the peaks signal and the reference image and select those with a similarity
larger than \code{threshold}. Multiple measures are available, correlation,
structural similarity index measure (SSIM), and normalized mutual information (NMI).
Since correlation can assume values in [-1, 1], also NMI are scaled in [-1, 1].
}
\examples{
## Load package
library("SPUTNIK")

## Mass spectrometry intensity matrix
X <- matrix(rnorm(16000), 400, 40)
X[X < 0] <- 0

## Print original dimensions
print(dim(X))

## m/z vector
mzVector <- seq(600, 900, by = (900 - 600) / 39)

## Read the image size
imSize <- c(20, 20)

## Construct the ms.dataset object
msiX <- msiDataset(X, mzVector, imSize[1], imSize[2])

## Generate the reference image and the ROI mask
ref.roi <- refAndROIimages(msiX, refMethod = "sum", roiMethod = "otsu")

## Perform global peaks filter
glob.peaks <- globalPeaksFilter(
  msiData = msiX, referenceImage = ref.roi$Reference,
  method = "pearson", threshold = 0
)

## Apply the filter
msiX <- applyPeaksFilter(msiX, glob.peaks)

## Print the new dimensions
print(dim(getIntensityMat(msiX)))
}
\references{
Wang, Z., Bovik, A. C., Sheikh, H. R., & Simoncelli, E. P. (2004).
Image quality assessment: from error visibility to structural similarity.
IEEE transactions on image processing, 13(4), 600-612.

Meyer, P. E. (2009). Infotheo: information-theoretic measures.
R package. Version, 1(0).
}
\seealso{
\code{\link{countPixelsFilter}} \code{\link{applyPeaksFilter-msi.dataset-method}}
}
\author{
Paolo Inglese \email{p.inglese14@imperial.ac.uk}
}
