% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SNSeg_Multi.R
\name{SNSeg_Multi}
\alias{SNSeg_Multi}
\title{Self-normalization (SN) based change points estimation for multivariate time
series}
\usage{
SNSeg_Multi(
  ts,
  paras_to_test = "mean",
  confidence = 0.9,
  grid_size_scale = 0.05,
  grid_size = NULL,
  plot_SN = FALSE,
  est_cp_loc = TRUE
)
}
\arguments{
\item{ts}{A multivariate time series represented as a matrix with p columns,
where each column is a univariate time series. The dimension p for ts should
be at least 2.}

\item{paras_to_test}{Type of the parameter as a string for which SN
algorithms test. Available choices include \code{mean} and \code{covariance}.}

\item{confidence}{Confidence level of SN tests as a numeric value. Available
choices of confidence levels contain 0.9, 0.95, 0.99, 0.995 and 0.999. The
default is set to 0.9.}

\item{grid_size_scale}{numeric value of the trimming parameter and only in
use if grid_size = NULL. Users are allowed to choose any grid_size_scale
between 0.05 and 0.5. A warning will be given if it is outside the
range.}

\item{grid_size}{Local window size h to compute the critical value for SN
test. Since grid_size = n*grid_size_scale, where n is the length of time
series, this function will compute the grid_size_scale by diving n from
grid_size when it is not NULL.}

\item{plot_SN}{Boolean value to plot the time series or not.
The default setting is FALSE.}

\item{est_cp_loc}{Boolean value to plot a red solid vertical line for
estimated change-point locations if est_cp_loc = TRUE}
}
\value{
\code{SNSeg_Multi} returns an S3 object of class "SNSeg_Multi" including
the time series, the type of parameter to be tested, the local window size to
cover a change point, the estimated change-point locations, the confidence level
and the critical value of the SN test. It also generates time series segmentation
plot when \code{plot_SN = TRUE}.
\describe{
\item{\code{ts}}{A numeric matrix of the input time series.}
\item{\code{paras_to_test}}{the parameter used for the SN test as character.}
\item{\code{grid_size}}{A numeric value of the window size.}
\item{\code{SN_sweep_result}}{A list of n matrices where each matrix
consists of four columns: (1) SN-based test statistic for each change-point
location (2) Change-point location  (3) Lower bound of the window h and
(4) Upper bound of the window h.}
\item{\code{est_cp}}{A vector containing the locations of the estimated
change-points.}
\item{\code{confidence}}{Confidence level of SN test as a numeric value.}
\item{\code{critical_value}}{Critical value of the SN-based test statistic.}
}

Users can apply the functions \code{summary.SN} to compute the parameter estimate
of each segment separated by the detected change-points. An additional function
\code{plot.SN} can be used to plot the time series with estimated change-points.
Users can set the option \code{plot_SN = TRUE} or use the function \code{plot.SN}
to plot the time series.

It deserves to note that some change-points could be missing due to the constraint
on \code{grid_size_scale} or related \code{grid_size} that \code{grid_size_scale}
has a minimum value of 0.05. Therefore, SNCP claims no change-points within the
first n\emph{\code{grid_size_scale} or the last n}\code{grid_size_scale} time points.
This is a limitation of the function \code{SNSeg_Multi}.

For more examples of \code{SNSeg_Multi} see the help vignette:
\code{vignette("SNSeg", package = "SNSeg")}
}
\description{
The function \code{SNSeg_Multi} is a SN-based change-points estimation
procedure for a multivariate time series based on changes in the multivariate
means or covariance matrix.
}
\examples{
\donttest{
# Please run this function before simulation
exchange_cor_matrix <- function(d, rho){
  tmp <- matrix(rho, d, d)
  diag(tmp) <- 1
  return(tmp)
}

# simulation of multivariate time series
library(mvtnorm)
set.seed(10)
d <- 5
n <- 600
nocp <- 5
cp_sets <- round(seq(0, nocp+1 ,1)/(nocp+1)*n)
mean_shift <- rep(c(0,2),100)[1:(length(cp_sets)-1)]/sqrt(d)
rho_sets <- 0.2
sigma_cross <- list(exchange_cor_matrix(d,0))
ts <- MAR_MTS_Covariance(n, 2, rho_sets, cp_sets = c(0,n), sigma_cross)
ts <- ts[1][[1]]

# Test for the change in multivariate means
# grid_size defined
result <- SNSeg_Multi(ts, paras_to_test = "mean", confidence = 0.99,
                      grid_size_scale = 0.05, grid_size = 45)
# Estimated change-point locations
result$est_cp

# For more examples, please run the following command:
# vignette("SNSeg", package = "SNSeg")
}

}
