% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/boot_overlap.R
\name{boot_overlap}
\alias{boot_overlap}
\title{Bootstrap overlaps between Utilisation Distributions (UDs)}
\usage{
boot_overlap(
  data,
  R = 1000,
  method = "PHR",
  percent = 100,
  quantiles = c(0.25, 0.5, 0.75)
)
}
\arguments{
\item{data}{A matrix or list of RasterLayer objects. 
Each row of the matrix or each RasterLayer object contains a utilisation distribution 
(or other statistics that sums to 1 - e.g. proportion of time spent).
\bold{The grid size and geographical extent must be consistent across each row of the matrix or each RasterLayer object.}
The function assumes that each column of the matrix is associated with a unique geographical location or 
that each RasterLayer has exactly the same geographical extent and resolution.}

\item{R}{An integer specifying the number of iterations. A larger \emph{R} is required when the sample size is large.
R > sample size x 100 is recommended (e.g. R > 1000 for a sample size 10).}

\item{method}{The overlap quantification method. 
"HR" is for the proportion of an individual's home range overlapped by the known habitats of other individuals. 
"PHR" is for the probability of an individual to be within the known habitats of other individuals. 
"VI", "BA" and "UDOI" quantify overlap between UDs using the full probabilistic properties as described in Fieberg and Kochanny (2005). 
For the latter three options, the function calculates overlaps between each additional UD and a collective UD. 
To generate a collective UD, each UD is overlaid and averaged at each grid cell so the probability density of the collective UD sums up to 1.}

\item{percent}{An integer specifying the percent volume of each UD to be considered in the analysis.}

\item{quantiles}{A vector or a number to specify the quantiles to be calculated in the summary of the results.}
}
\value{
A list containing two data frames - raw results and summary (mean, sd, sem and quantiles at each sample size).
}
\description{
Function to calculate overlaps between UDs relative to sample size by bootstrapping.
}
\details{
This function calculates and bootstraps overlap between UDs based on the areas ("HR"), 
areas of collective UDs and the probability distribution of each individual ("PHR"), 
or the probability distribution of an individual and an averaged probability distribution of collective individuals ("VI", "BA", "UDOI").
}
\examples{
\dontrun{

#1 Utilisation uistributions of flatback turtles (n = 15).
data(ud_matrix)

#2 Calculate overlap probability from 2000 random permutation.
overlap <- boot_overlap(ud_matrix, R = 2000, method = "PHR")

#3 Find the minimum sample size required to estimate the general distribution.
a <- asymptote(overlap, upper.degree = 10, estimator = 'glm', family = binomial)

#4 Plot the mean probability and rational function fit relative to the sample sizes.
ggplot(data = a$results, aes(x = x))+
  geom_pointrange(aes(y = y, ymin = y_lwr, ymax = y_upr)) + 
  geom_hline(yintercept = a$h.asymptote*0.95, linetype = 2) +
  scale_x_continuous(breaks = seq(0, 15, 3), limits = c(2,15), name = "Animals tracked (n)") +
  scale_y_continuous(limits = c(0.5,1), name = "Overlap probability")
}
}
\references{
Shimada T, Thums M, Hamann M, Limpus CJ, Hays GC, FitzSimmons N, Wildermann NE, Duarte CD, Meekan MG (2021) 
Optimising sample sizes for animal distribution analysis using tracking data. 
\emph{Methods in Ecology and Evolution} 12(2):288-297 \doi{10.1111/2041-210X.13506}

Fieberg J & Kochanny CO (2005) Quantifying home-range overlap: The importance of the utilization distribution. 
\emph{The Journal of Wildlife Management}, 69(4), 1346–1359. \doi{10.2193/0022-541x(2005)69[1346:Qhotio]2.0.Co;2}
}
\seealso{
\code{\link{combn_overlap}}, \code{\link{boot_area}}
}
\author{
Takahiro Shimada
}
