% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/depthfilter.r
\name{depthfilter}
\alias{depthfilter}
\title{Filter locations by water depth}
\usage{
depthfilter(sdata, bathymetry, extract = "bilinear", tide, qi = 4,
  depth = 0, tidal.plane, filter = TRUE)
}
\arguments{
\item{sdata}{A data frame containing columns with the following headers: "id", "DateTime", "lat", "lon", "qi". 
This filter is independently applied to a subset of data grouped by the unique "id". 
"DateTime" is date & time in class POSIXct. "lat" and "lon" are the recorded latitude and longitude in decimal degrees. 
"qi" is the numerical quality index associated with each fix where the greater number represents better quality 
(e.g. number of GPS satellites used for estimation).}

\item{bathymetry}{object of class "RasterLayer" containing bathymetric data in meters. Geographic coordinate system is WGS84.}

\item{extract}{Method to extract cell values from raster layer inherited from extract function of raster package. 
Default is "bilinear". See the raster package for details.}

\item{tide}{A data frame containing columns with the following headers: "tideDT", "reading", "standard.port". 
"tideDT" is date & time in class POSIXct for each observed tidal height. "reading" is the observed tidal height in meters. 
"standard.port" is the identifier of each tidal station.}

\item{qi}{An integer specifying threshold quality index. 
Fixes associated to a quality index higher than the threshold are excluded from the depthfilter. Default is 4}

\item{depth}{An integer denoting vertical difference from a high tide line in meters. 
A positive value indicates above the high tide and a negative value indicates below the high tide. 
The function removes fixes above the given threshold. Default is 0 m (i.e. high tide line).}

\item{tidal.plane}{A data frame containing columns with the following headers: 
"standard.port", "secondary.port", "lat", "lon", "timeDiff", "datumDiff". 
"standard.port" is the identifier for a tidal observation station. 
"secondary.port" is the identifier for a station at which tide is only predicted using tidal records observed at the related standard port. 
"lat" and "lon" are the latitude and longitude of each secondary port in decimal degrees. 
"timeDiff" is the time difference between standard port and its associated secondary port. 
"datumDiff" is the baseline difference in meters between bathymetry and tidal observations/predictions 
if each data uses different datum (e.g. LAT and MSL).}

\item{filter}{Default is TRUE. If FALSE, the function does not filter locations but the depth estimates are returned.}
}
\value{
Input data is returned with two columns added; "depth.exp", "depth.HT". 
"depth.exp" is the estimated water depth at the time of location fixing. 
"depth.HT" is the estimated water depth at the high tide nearest to the time and location of each fix. 
When the "filter" option is enabled, the fixes identified by this filter are removed from the input data.
}
\description{
Function to remove fixes located at a given height from the high tide line.
}
\details{
This function removes fixes located at a given height from estimated high tide line when the "filter" option is enabled. 
The function chooses the closest match between each fix and tidal observations or predictions in temporal and spatial scales 
in order to estimate height of high tide at the time and location of each fix. 
It does not filter data when the "filter" option is disabled but it returns the estimated water depth of each location with 
the tide effect accounted for (bathymetry + tide height). The estimated water depths are returned in the "depth.exp" column.
}
\note{
Input data must not contain temporal or spatial duplicates.
}
\examples{

### Load data sets
# Fastloc GPS data obtained from a green turtle
data(turtle)

# Bathymetry model developed by Beaman (2010)
data(bathymodel)

# A tidal plane for the example site
data(tidalplane)

# Tidal observations and predictions for the example site
data(tidedata)

# Map for the example site
data(basemap)


### Remove temporal and/or spatial duplicates
turtle.dup <- dupfilter(turtle)


### Remove biologically unrealistic fixes 
## Estimate vmax
vmax <- est.vmax(turtle.dup)

## Estimate maxvlp
maxvlp <- est.maxvlp(turtle.dup)

## Apply ddfilter
turtle.dd <- ddfilter(turtle.dup, vmax=vmax, maxvlp=maxvlp)


### Apply depthfilter
turtle.dep <- depthfilter(sdata=turtle.dd, 
                          bathymetry=bathymodel, 
                          tide=tidedata, 
                          tidal.plane=tidalplane)


### Plot data on a map before and after depthfilter is applied
par(mfrow=c(1,2))
par(mar=c(4,5,2,1))
plot(basemap, col="grey", xlim=c(152.8, 153.1), ylim=c(-25.75, -25.24))
axis(1, at=seq(from=152, to=154, by=0.2))
axis(2, at=seq(from=-26, to=-25, by=0.2), las=2)
mtext("Longitude", side=1, line=2.5)
mtext("Latitude", side=2, line=3.5)
box()
title("Not applied")
LatLong <- data.frame(Y=turtle.dup$lat, X=turtle.dup$lon)
coordinates(LatLong) <- ~X+Y
proj4string(LatLong) <- CRS("+proj=longlat +ellps=WGS84 +datum=WGS84")
plot(LatLong, pch=21, bg="yellow", add=TRUE)

par(mar=c(4,4,2,2))
plot(basemap, col="grey", xlim=c(152.8, 153.1), ylim=c(-25.75, -25.24))
axis(1, at=seq(from=152, to=154, by=0.2))
axis(2, at=seq(from=-26, to=-25, by=0.2), las=2)
mtext("Longitude", side=1, line=2.5)
box()
title("Applied")
LatLong <- data.frame(Y=turtle.dep$lat, X=turtle.dep$lon)
coordinates(LatLong) <- ~X+Y
proj4string(LatLong) <- CRS("+proj=longlat +ellps=WGS84 +datum=WGS84")
plot(LatLong, pch=21, bg="yellow", add=TRUE)
}
\references{
Shimada T, Limpus C, Jones R, Hazel J, Groom R, Hamann M (2016) 
Sea turtles return home after intentional displacement from coastal foraging areas. 
Marine Biology 163:1-14 doi:10.1007/s00227-015-2771-0

Beaman, R.J. (2010) Project 3DGBR: A high-resolution depth model for the Great Barrier Reef and Coral Sea. 
Marine and Tropical Sciences Research Facility (MTSRF) Project 2.5i.1a Final Report, MTSRF, Cairns, Australia, pp. 13 plus Appendix 1.
}
\seealso{
dupfilter, ddfilter
}
\author{
Takahiro Shimada
}
