% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cobraInit.R
\name{cobraInit}
\alias{cobraInit}
\title{Initial phase for SACOBRA optimizer}
\usage{
cobraInit(
  xStart,
  fn,
  fName,
  lower,
  upper,
  feval,
  initDesign = "LHS",
  initDesPoints = 2 * length(xStart) + 1,
  initDesOptP = NULL,
  initBias = 0.005,
  skipPhaseI = TRUE,
  seqOptimizer = "COBYLA",
  seqFeval = 1000,
  seqTol = 1e-06,
  ptail = TRUE,
  squares = TRUE,
  conTol = 0,
  DOSAC = 1,
  sac = defaultSAC(DOSAC),
  repairInfeas = FALSE,
  ri = defaultRI(),
  RBFmodel = "cubic",
  RBFwidth = -1,
  GaussRule = "One",
  widthFactor = 1,
  RBFrho = 0,
  MS = defaultMS(),
  equHandle = defaultEquMu(),
  rescale = TRUE,
  newlower = -1,
  newupper = 1,
  XI = DRCL,
  TrustRegion = FALSE,
  TRlist = defaultTR(),
  conditioningAnalysis = defaultCA(),
  penaF = c(3, 1.7, 3e+05),
  sigmaD = c(3, 2, 100),
  constraintHandling = "DEFAULT",
  verbose = 1,
  verboseIter = 10,
  DEBUG_RBF = defaultDebugRBF(),
  DEBUG_TR = FALSE,
  DEBUG_TRU = FALSE,
  DEBUG_RS = FALSE,
  DEBUG_XI = FALSE,
  trueFuncForSurrogates = FALSE,
  saveIntermediate = FALSE,
  saveSurrogates = FALSE,
  epsilonInit = NULL,
  epsilonMax = NULL,
  solu = NULL,
  cobraSeed = 42
)
}
\arguments{
\item{xStart}{a vector of dimension \code{d} containing the starting point for the optimization problem}

\item{fn}{objective and constraint functions: \code{fn} is a function accepting 
a \code{d}-dimensional vector \eqn{\vec{x}} and returning an \eqn{(1+m+r)}-dimensional
vector \code{c(}\eqn{f,g_1,\ldots,g_m,h_1,\ldots,h_r}\code{)}}

\item{fName}{the results of \code{\link{cobraPhaseII}} are saved to \code{<fname>.Rdata}}

\item{lower}{lower bound \eqn{\vec{a}} of search space, same dimension as \code{xStart}}

\item{upper}{upper bound \eqn{\vec{b}} of search space, same dimension as \code{xStart}}

\item{feval}{maximum number of function evaluations}

\item{initDesign}{["LHS"] one out of ["RANDOM","LHS","BIASED","OPTIMIZED","OPTBIASED"]}

\item{initDesPoints}{[\code{2*d+1}] number of initial points, must be smaller than \code{feval}}

\item{initDesOptP}{[NULL] only for initDesign=="OPTBIASED": number of points for the "OPT"
phase. If NULL, take initDesPoints.}

\item{initBias}{[0.005] bias for normal distribution in "OPTBIASED" and "BIASED"}

\item{skipPhaseI}{[TRUE] if TRUE, then skip \code{\link{cobraPhaseI}}}

\item{seqOptimizer}{["COBYLA"] string defining the optimization method for COBRA phases I 
and II, one out of ["COBYLA","ISRES","HJKB","NMKB","ISRESCOBY"]}

\item{seqFeval}{[1000] maximum number of function evaluations on the surrogate model}

\item{seqTol}{[1e-6] convergence tolerance for sequential optimizer, see param \code{tol} 
in \code{\link[dfoptim]{nmkb}} or param \code{control$xtol_rel} 
in \code{\link[nloptr]{cobyla}}}

\item{ptail}{[TRUE] TRUE: with, FALSE: without polynomial tail in \code{trainRBF}}

\item{squares}{[TRUE] set to TRUE for including the second order polynomials in building the fitness and constraint surrogates in \code{trainRBF}}

\item{conTol}{[0.0] constraint violation tolerance}

\item{DOSAC}{[1] set one out of [0|1|2]. \cr
0: COBRA-R settings, \cr 1: SACOBRA settings, \cr 2: SACOBRA settings with fewer parameters. \cr 
The precise settings are documented in \code{\link{defaultSAC}}.}

\item{sac}{[\code{\link{defaultSAC}(DOSAC)}] list with other parameters for SACOBRA.}

\item{repairInfeas}{[FALSE] if TRUE, trigger the repair of appropriate infeasible solutions}

\item{ri}{[\code{\link{defaultRI}()}] list with other parameters for 
\code{\link{repairInfeasRI2}}}

\item{RBFmodel}{["cubic"] a string for the type of the RBF model, "cubic", "Gaussian" or "MQ"}

\item{RBFwidth}{[-1] only relevant for Gaussian RBF model. Determines the width \eqn{\sigma}. 
For more details see parameter \code{width} in \code{\link{trainGaussRBF}} in \code{RBFinter.R}.}

\item{GaussRule}{["One"] only relevant for Gaussian RBF model, see \code{\link{trainGaussRBF}}}

\item{widthFactor}{[1.0] only relevant for Gaussian RBF model. Additional constant 
factor applied to each width \eqn{\sigma}}

\item{RBFrho}{[0.0] experimental: 0: interpolating, > 0, approximating (spline-like) Gaussian RBFs}

\item{MS}{[\code{\link{defaultMS}()}] list of online model selection parameters described in \code{\link{defaultMS}}. 
If \code{MS$active = TRUE} then the type of RBF models for each function will be selected automatically and the \code{RBFmodel} parameter becomes irrelevant.}

\item{equHandle}{[\code{\link{defaultEquMu}()}] list with of parameters for
equality constraint handling described in \code{\link{defaultEquMu}()}. equHandle$active is set to TRUE by default.}

\item{rescale}{[TRUE] if TRUE, transform the input space from \code{[lower,upper]} 
to hypercube \code{[newlower,newupper]^d}}

\item{newlower}{[-1] lower bound of each rescaled input space dimension, if \code{rescale==TRUE}}

\item{newupper}{[+1] upper bound of each rescaled input space dimension, if \code{rescale==TRUE}}

\item{XI}{[DRCL] magic parameters for the distance requirement cycle (DRC)}

\item{TrustRegion}{[FALSE] if TRUE, perform trust region algorithm \code{\link{trustRegion}}.}

\item{TRlist}{[\code{\link{defaultTR}()}] a list of parameters, needed only 
in case \code{TrustRegion==TRUE}.}

\item{conditioningAnalysis}{[\code{\link{defaultCA}()}] A list with setting for the objective function conditioning analysis and online whitening}

\item{penaF}{[c(3,1.7,3e5)] parameters for dynamic penalty factor (fct subProb in 
\code{\link{cobraPhaseII}}): \code{c(start,augment,max)}, only relevant \code{if seqOptimizer==HJKB} or \code{seqOptimizer==NMKB}}

\item{sigmaD}{[c(3,2.0,100)] parameters for dynamic distance factor (fct subProb in 
\code{\link{cobraPhaseII}}): \code{c(start,augment,max)}, , only relevant \code{if seqOptimizer==HJKB} or \code{seqOptimizer==NMKB}}

\item{constraintHandling}{["DEFAULT"] (other choices: "JOINESHOUCK", "SMITHTATE", "COIT", "BAECKKHURI";
experimental, only relevant \code{if seqOptimizer==HJKB} or \code{seqOptimizer==NMKB}
see the code in function \code{subProb} in \code{\link{cobraPhaseII}})}

\item{verbose}{[1] set one out of [0|1|2], how much output to print}

\item{verboseIter}{[10] an interegr value. Printing the summarized results after each \code{verboseIter} iterations.}

\item{DEBUG_RBF}{[\code{defaultDebugRBF()}] list with settings for visualization RBF (only for \code{d==2})}

\item{DEBUG_TR}{[FALSE] prints information about trust region status and visualisation for \code{d==2} (coming soon)}

\item{DEBUG_TRU}{[FALSE] visualize trust-region RBF (only for dimension==2)}

\item{DEBUG_RS}{[FALSE] prints the RS probability in each iteration in the console}

\item{DEBUG_XI}{[FALSE] if TRUE, then print in \code{\link{cobraPhaseII}} extra debug information: 
\code{xStart} in every iteration to console and add some extra debug 
columns to \code{cobra$df}}

\item{trueFuncForSurrogates}{[FALSE] if TRUE, use the true (constraint & fitness) functions
instead of surrogates (only for debug analysis)}

\item{saveIntermediate}{[FALSE] if TRUE, then \code{\link{cobraPhaseII}} saves intermediate results
in dir 'results/' (create it, if necessary)}

\item{saveSurrogates}{[FALSE] if TRUE, then \code{\link{cobraPhaseII}} returns the last surrogate models in
cobra$fitnessSurrogate and cobra$constraintSurrogates}

\item{epsilonInit}{[NULL] initial constant added to each constraint to maintain a certain margin to boundary}

\item{epsilonMax}{[NULL] maximum for constant added to each constraint}

\item{solu}{[NULL] the best-known solution (only for diagnostics). This is normally a 
vector of length d. If there are multiple solutions, it is a matrix with d
columns (each row is a solution). If NULL, then the current best point
will be used in \code{\link{cobraPhaseII}}. 
\code{solu} is given in original input space.}

\item{cobraSeed}{[42] seed for random number generator}
}
\value{
\code{cobra}, an object of class COBRA, this is a (long) list containing most
  of the argument settings (see above) and in addition (among others):
     \item{\code{A}}{ (feval x dim)-matrix containing the initial design points in input .  
           space. If rescale==TRUE, all points are in  \strong{rescaled} input space. }
     \item{\code{Fres}}{ a vector of the objective values of the initial design points }
     \item{\code{Gres}}{ a matrix of the constraint values of the initial design points }
     \item{\code{nConstraints}}{ the total number \eqn{m+r} of constraints } 
     \item{\code{Tfeas}}{ the threshhold parameter for the number of consecutive iterations 
           that yield feasible solutions before margin epsilon is reduced }
     \item{\code{Tinfeas}}{ the threshhold parameter for the number of consecutive iterations 
           that yield infeasible solutions before margin epsilon is increased }
     \item{\code{numViol}}{ number of constraint violations }
     \item{\code{maxViol}}{ maximum constraint violation}
     \item{\code{trueMaxViol}}{ maximum constraint violation}
     \item{\code{trustregX}}{A vector of all refined solutions generated by trust region algorithm (see \code{trustRegion})}
     
  
  Note that \code{cobra$Fres}, \code{cobra$fbest}, \code{cobra$fbestArray} and similar contain 
  always the objective values of the orignial function \code{cobra$fn[1]}. (The surrogate models 
  may be trained on a \code{\link{plog}}-transformed version of this function.)
}
\description{
In this phase the important parameters are set and the initial design population are evaluated on the real function. The problem to solve is: 
\deqn{ \mbox{Minimize}\quad  f(\vec{x}) , \vec{x} \in [\vec{a},\vec{b}] \subset \mathbf{R}^d }
\deqn{ \mbox{subject to}\quad g_i(\vec{x}) \le 0, i=1,\ldots,m    }
\deqn{ \mbox{~~~~~~~~~~}\quad\quad h_j(\vec{x}) = 0, j=1,\ldots,r.   }
}
\details{
If \code{epsilonInit} or \code{epsilonMax} are NULL on input, then \code{cobra$epsilonInit}
and \code{cobra$epsilonMax},  resp., are set to \code{0.005*l} where \code{l} is the smallest 
side of the search box.

Note that the parameters \code{penaF}, \code{sigmaD}, \code{constraintHandling} are only 
relevant for penalty-based internal optimizers \code{\link[dfoptim]{nmkb}} or HJKB. They are NOT relevant  
for default optimizer \code{\link[nloptr]{cobyla}}.

Although the software was originally designed to handle only constrained optimization problems, 
it can also address unconstrained optimization problems  

How to code which constraint is equality constraint? - Function \code{fn} should return 
an \eqn{(1+m+r)}-dimensional vector with named elements. The first element is the objective, the 
other elements are the constraints. All equality constraints should carry the name \code{equ}. 
(Yes, it is possible that multiple elements of a vector have the same name.)
}
\examples{
## Initialize cobra. The problem to solve is the sphere function sum(x^2)    
## with the equality constraint that the solution is on a circle with 
## radius 2 and center at c(1,0).
d=2
fName="onCircle"
cobra <- cobraInit(xStart=rep(5,d), fName=fName,
                   fn=function(x){c(obj=sum(x^2),equ=(x[1]-1)^2+(x[2]-0)^2-4)},  
                   lower=rep(-10,d), upper=rep(10,d), feval=40)
                   
## Run sacobra optimizer
cobra <- cobraPhaseII(cobra)

## The true solution is at solu = c(-1,0) (the point on the circle closest 
## to the origin) where the true optimum is fn(solu)[1] = optim = 1
## The solution found by SACOBRA:
print(getXbest(cobra))
print(getFbest(cobra))

## Plot the resulting error (best-so-far feasible optimizer result - true optimum)
## on a logarithmic scale:
optim = 1
plot(abs(cobra$df$Best-optim),log="y",type="l",ylab="error",xlab="iteration",main=fName)

}
\seealso{
\code{\link{startCobra}}, \code{\link{cobraPhaseI}}, \code{\link{cobraPhaseII}}
}
\author{
Wolfgang Konen, Samineh Bagheri, Patrick Koch, Cologne University of Applied Sciences
}
