% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/assembly.R, R/methods.R
\name{assembly}
\alias{assembly}
\alias{mst}
\alias{mst_route}
\alias{mst_objective}
\alias{mst_constraint}
\alias{mst_stage_length}
\alias{mst_rdp}
\alias{mst_module_info}
\alias{mst_assemble}
\alias{mst_get_items}
\alias{print.mst}
\alias{plot.mst}
\title{Assemble Computerized Adaptive Multistage Testing}
\usage{
mst(pool, design, n_panels = 1, method = c("topdown", "bottomup"),
  test_len = NULL, max_use = NULL, ...)

mst_route(x, route, op = c("+", "-"))

mst_objective(x, coef, mode = c("max", "min"), indices = NULL,
  target = NULL, method = NULL, ...)

mst_constraint(x, coef, min = NA, max = NA, level = NULL,
  indices = NULL, method = NULL)

mst_stage_length(x, stages, min = NA, max = NA)

mst_rdp(x, theta, indices, tol = 0.5)

mst_module_info(x, theta, min = NA, max = NA, indices)

mst_assemble(x, solver = c("lpsolve", "glpk"), silent = FALSE,
  time_limit = 30, message = FALSE, ...)

mst_get_items(x, panel_ix = NULL, stage_ix = NULL, module_ix = NULL,
  route_ix = NULL)

\method{print}{mst}(x, ...)

\method{plot}{mst}(x, ...)
}
\arguments{
\item{pool}{the item pool (a list of '3pl', 'gpcm', and 'grm' items)}

\item{design}{the MST design (string): e.g., "1-3", "1-2-2", "1-2-3"}

\item{n_panels}{the number of panels (integer)}

\item{method}{the design method (string): 'topdown' or 'bottomup'}

\item{test_len}{the module/route length (integer)}

\item{max_use}{the maximum selection of items (integer)}

\item{...}{additional arguments}

\item{x}{the MST object}

\item{route}{a MST route (a vector of module index)}

\item{op}{\code{"+"} to add a route and \code{"-"} to remove a route}

\item{coef}{the coefficients (numeric vector or string)}

\item{mode}{the optimization direction: \code{"max"} or \code{"min"}}

\item{indices}{the indices of the route (topdown) or the module (bottomup) where the objective is added}

\item{target}{the target values of the absolute objectives, \code{NULL} for the relative objective}

\item{min}{the lower bound of the constraint}

\item{max}{the upper bound of the constraint}

\item{level}{the constrained level of categorical item attribute,
\code{NULL} for continuous item attributes}

\item{stages}{the stage indices}

\item{theta}{the theta point where TIF is controlled}

\item{tol}{tolerance parameter (numeric)}

\item{solver}{the MIP solver: \code{"lpsolve"} or \code{"glpk"}}

\item{silent}{\code{TRUE} to mute solving status}

\item{time_limit}{the time limit for solving the model in seconds}

\item{message}{\code{TRUE} to print messages from the solver}

\item{panel_ix}{the panel index (int vector)}

\item{stage_ix}{the stage index (int vector)}

\item{module_ix}{the module index (int vector)}

\item{route_ix}{the route index (int vector)}
}
\value{
\code{mst} returns a \code{mst} object.

\code{mst_get_items} returns the assembled forms in a list of 3pl, gpcm, and grm items
}
\description{
\code{mst} creates a multistage (MST) assembly model

\code{mst_route} adds/removes a route to/from the assembly model

\code{mst_objective} adds an objective to the assembly model

\code{mst_constraint} adds constraints to the assembly model

\code{mst_stage_length} sets length limits on stages

\code{mst_rdp} anchors the routing decision point (rdp) between adjacent modules

\code{mst_module_info} sets the information requirements for modules

\code{mst_assemble} tries to solve the assembly model

\code{mst_get_items} retrieves items from the assembly results
}
\details{
A \code{mst} object stores the definitions of the MST. When \code{mst_assemble}
is called, definitions are converted to a real mixed integer programming model for
assembly. If the model is solved, assembled items are appended to the original object. \cr
The bottom-up approach adds objectives and constraints on individual modules, and the top-down
approach adds objectives and constraints on routes.

\code{coef} in \code{mst_objective} can be a vector of theta points where TIFs are optimized, or
a continuous variable in the pool where the item attribute is optimized, or a numeric value with
the same length of the pool at either item or group level.

\code{plot.mst} draws module information functions when \code{byroute=FALSE}
and route information functions when \code{byroute=TRUE}. Use \code{label=TRUE}
to put labels on routes and modules.
}
\examples{
## generate item pool
set.seed(123456)
items <- Rirt::model_mixed_gendata(1, n_3pl=200)$items

## Ex. 1: 1-2-2 MST, 2 panels, 20 items, topdown
## maximize info. at -1 and 1 for easy and hard routes
x <- mst(items, "1-2-2", n_panels=2, method='topdown', test_len=10, max_use=1)
x <- mst_objective(x, -1, indices=1:2)
x <- mst_objective(x, 1, indices=3:4)
x <- mst_assemble(x, 'lpsolve', time_limit=2)
plot(x, byroute=TRUE, label=TRUE)

\donttest{
## Ex. 2: 1-2-3 MST, 2 panels, bottomup,
## remove two routes with large theta change: 1-2-6, 1-3-4
## 10 items in each module, content= and 3 items in content area 1 in each module
## maximize info. at -1, 0 and 1 for easy, medium, and hard modules
x <- mst(items, "1-2-3", 1, 'bottomup', len=10, max_use=1)
x <- mst_route(x, c(1, 2, 6), "-")
x <- mst_route(x, c(1, 3, 4), "-")
x <- mst_objective(x,  0, indices=c(1, 5))
x <- mst_objective(x, -1, indices=c(2, 4))
x <- mst_objective(x,  1, indices=c(3, 6))
x <- mst_assemble(x, timeout=30)
plot(x, byroute=FALSE)
plot(x, byroute=TRUE)
}
}
