% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/medde.R
\name{medde}
\alias{medde}
\title{Maximum Entropy [De]Regularized Density Estimation}
\usage{
medde(x, v = 300, lambda = 0.5, alpha = 1, Dorder = 1, rtol = 1e-06,
  verb = 0, control = NULL)
}
\arguments{
\item{x}{Data: either univariate or bivariate (not yet implemented in
Rmosek)}

\item{v}{Undata: either univariate or bivariate, by default there is an
equally spaced grid of 300 values}

\item{lambda}{total variation penalty parameter, if lambda is in [-1,0], a
concavity constraint is imposed. If lambda is in (-oo, -1) a convexity
constraint on .5 x^2 + log f is imposed. See Koenker and Mizera (2013) for
further details on this last option, and Koenker and Mizera (2010) for
further details on the concavity constrained options.}

\item{alpha}{Renyi entropy parameter characterizing fidelity criterion}

\item{Dorder}{Order of the derivative operator for the penalty}

\item{rtol}{Convergence tolerance for Mosek algorithm}

\item{verb}{Parameter controlling verbosity of solution, 0 for silent, 5
gives rather detailed iteration log.}

\item{control}{Mosek control list see KWDual documentation}
}
\value{
An object of class "Medde" with components \item{x}{points of
evaluation on the domain of the density} \item{y}{estimated function values
at the evaluation points x} \item{phi}{n by 2 matrix of (sorted) original
data and estimated log density values these data points} \item{logLik}{log
likelihood value} \item{status}{exit status from Mosek}
}
\description{
Density estimation based on maximum entropy methods
}
\details{
See the references for further details. And also Mosek "Manuals". The
acronym has a nice connection to

http://www.urbandictionary.com/define.php?term=medder

Term used in Bahamian dialect, mostly on the Family Islands like Eleuthera
and Cat Island meaning "mess with" "get involved" "get entangled" "fool
around" "bother"

"I don't like to medder up with all kinda people"

"Don't medder with people (chirren)"

"Why you think she medderin up in their business."

This version implements a class of penalized density estimators solving:
\deqn{\min_x \phi(x_1) | A_1 x_1 - A_2 x_2 = b,  0 \leq x_1, -\lambda \leq x_2 \leq \lambda }
where \eqn{x}x is a vector with two component subvectors: \eqn{x_1} is a 
vector of function values of the density \eqn{x_2} is a vector of dual values,
\eqn{\lambda} is typically positive, if negative then the \eqn{x_2} constraint
is replaced by \eqn{x_2 \geq 0}, which for \eqn{\alpha = 1}, 
constrains the fitted density to be log-concave, for
negative lambda fitting for other alphas and Dorders 
proceed at your own risk. See also below...
\eqn{\phi} is an additive convex function in the coordinates of \eqn{x_1},
interpretable as a negative Renyi entropy.
This formulation purports to solve a class of dual penalized maximum (Renyi) entropy
problems in which regularization is achieved by constraining the sup norm of the
dual vector.  In the primal representation of the problems this corresponds to
a roughness penalty on the total variation of the Dorder derivative of some
transformation of the fitted density.
}
\examples{

#Maximum Likelihood Estimation of a Log-Concave Density
set.seed(1968)
x <- rgamma(50,10)
m <- medde(x, v = 50, lambda = -.5, verb = 5)
plot(m, type = "l")
lines(m$x,dgamma(m$x,10),col = 2)
points(x,m$g,cex = 0.5)
rug(x)
title(paste("log likelihood = ", round(m$logLik,2)))
legend(14,.12,c("ghat","true"),lty = 1, col = 1:2)

#Maximum Likelihood Estimation of a Gamma Density with TV constraint
set.seed(1968)
x <- rgamma(50,5)
f <- medde(x, v = 50, lambda = 0.005, verb = 5)
plot(f, type = "l")
lines(f$x,dgamma(f$x,5),col = 2)
legend(10,.15,c("ghat","true"),lty = 1, col = 1:2)


}
\author{
Roger Koenker and Ivan Mizera
}
\references{
Chen, Y. and R.J. Samworth, (2013) "Smoothed log-concave
maximum likelihood estimation with applications", \emph{Statistica Sinica},
23, 1373--1398.

Koenker, R and I. Mizera, (2007) ``Density Estimation by Total Variation
Regularization,'' \emph{Advances in Statistical Modeling and Inference:
Essays in Honor of Kjell Doksum}, V.N. Nair (ed.), 613-634.

Koenker, R and I. Mizera, (2006) ``The alter egos of the regularized maximum
likelihood density estimators: deregularized maximum-entropy, Shannon,
Renyi, Simpson, Gini, and stretched strings,'' \emph{ Proceedings of the 7th
Prague Symposium on Asymptotic Statistics}.

Koenker, R and I. Mizera, (2010) ``Quasi-Concave Density Estimation''
\emph{Annals of Statistics}, 38, 2998-3027.

Koenker, R and I. Mizera, (2013) ``Convex Optimization, Shape Constraints,
Compound Decisions, and Empirical Bayes Rules,'' JASA, 109, 674--685.

Koenker, R and I. Mizera, (2014) ``Convex Optimization in R.'',
\emph{Journal of Statistical Software}, 60, 1-23.
}
\seealso{
This function is based on an earlier function of the same name in
the deprecated package MeddeR that was based on an R-Matlab interface.
}
\keyword{nonparametric}

