% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/data_simulation.R
\name{generatedata_adjpin}
\alias{generatedata_adjpin}
\title{Simulation of AdjPIN model data.}
\usage{
generatedata_adjpin(series=1, days = 60, parameters = NULL, ranges = list(),
restricted = list(), verbose = TRUE)
}
\arguments{
\item{series}{The number of datasets to generate.}

\item{days}{The number of trading days, for which aggregated
buys and sells are generated. The default value is \code{60}.}

\item{parameters}{A vector of model parameters of size \code{10} and it has
the following form \{\eqn{\alpha}, \eqn{\delta}, \eqn{\theta}, \eqn{\theta'},
\eb, \es, \mub, \mus, \Db, \Ds\}.}

\item{ranges}{A list of ranges for the different simulation
parameters having named elements \code{alpha} \eqn{(\alpha)},
\code{delta} \eqn{(\delta)}, \code{theta} \eqn{(\theta)}, \code{thetap} \eqn{(\theta')},
\code{eps.b} (\eb), \code{eps.s} (\es), \code{mu.b} (\mub), \code{mu.s} (\mus), \code{d.b} (\Db),
\code{d.s} (\Ds).
The value of each element is a vector of two numbers: the first one is the
minimal value \code{min_v} and the second one is the maximal value \code{max_v}.
If the element corresponding to a given parameter is missing, the default
range for that parameter is used, otherwise, the simulation parameters are
uniformly drawn from the interval (\code{min_v}, \code{max_v}). The default value
is \code{list()}.}

\item{restricted}{A binary list that allows estimating restricted
AdjPIN models by specifying which model parameters are assumed to be equal.
It contains one or multiple of the following four elements
\verb{\{theta, mu, eps, d\}}. For instance, If \code{theta} is set to \code{TRUE},
then the probability of liquidity shock in no-information days, and in
information days is assumed to be the same (\thetaB\code{=}\thetaS). If any of
the remaining rate elements \verb{\{mu, eps, d\}} is set to \code{TRUE},
(say  \code{mu=TRUE}), then the rate is assumed to be the same on the buy side,
and on the sell side (\mub\code{=}\mus). If more than one element is set to
\code{TRUE}, then the restrictions are combined. For instance, if the argument
\code{restricted} is set to \code{list(theta=TRUE, eps=TRUE, d=TRUE)}, then the
restricted AdjPIN model is estimated, where \thetaB\code{=}\thetaS, \eb\code{=}\es,
and \Db\code{=}\Ds. If the value of the argument \code{restricted} is the empty list
(\code{list()}), then all parameters of the model are assumed to be independent,
and the unrestricted model is estimated. The default value is the empty
list \code{list()}.}

\item{verbose}{A binary variable that determines whether detailed
information about the progress of the data generation is displayed.
No output is produced when \code{verbose} is set to \code{FALSE}. The default
value is \code{TRUE}.}
}
\value{
Returns an object of class \code{dataset} if \code{series=1}, and an
object of class \code{data.series} if \code{series>1}.
}
\description{
Generates a \code{dataset} object or a \code{data.series} object (a list
of \code{dataset} objects) storing simulation parameters as well as aggregate
daily buys and sells simulated following the assumption of the \code{AdjPIN} model
of \insertCite{Duarte09;textual}{PINstimation}.
}
\details{
If the argument \code{parameters} is missing, then the parameters are
generated using the ranges specified in the argument \code{ranges}.
If the argument \code{ranges} is set to \code{list()}, default ranges are used. Using
the default ranges, the simulation parameters are obtained using the
following procedure:
\itemize{
\item \eqn{\alpha}, \eqn{\delta}: \code{(alpha, delta)} uniformly
distributed on \code{(0, 1)}.
\item \eqn{\theta}, \eqn{\theta'}: \code{(theta,thetap)} uniformly
distributed on \code{(0, 1)}.
\item \eb: \code{(eps.b)} an integer uniformly drawn from the interval
\code{(100, 10000)} with step \code{50}.
\item \es: \code{(eps.s)} an integer uniformly drawn from (\code{(4/5)}\eb,
\code{(6/5)}\eb) with step \code{50}.
\item \Db: \code{(d.b)} an integer uniformly drawn from (\code{(1/2)}\eb,
\code{2}\eb).
\item \Ds: \code{(d.s)} an integer uniformly drawn from (\code{(4/5)}\Db,
\code{(6/5)}\Db).
\item \mub: \code{(mu.b)} uniformly distributed on the interval
\verb{((1/2) max}(\eb, \es)\verb{, 5 max}(\eb, \es)\verb{)}.
\item \mus: \code{(mu.s)} uniformly distributed on the interval
(\code{(4/5)}\mub, \code{(6/5)}\mub)..
}

Based on the simulation parameters \code{parameters}, daily buys and sells are
generated by the assumption that buys and sells follow Poisson
distributions with mean parameters:
\itemize{
\item (\eb, \es) in a day with no information and no liquidity shock;
\item (\eb+\Db, \es+\Ds) in a day with no information and with liquidity
shock;
\item (\eb+\mub, \es) in a day with good information and no liquidity
shock;
\item (\eb+\mub+\Db, \es+\Ds) in a day with good information and
liquidity shock;
\item (\eb, \es+\mus) in a day with bad information and no liquidity
shock;
\item (\eb+\Ds, \es+\mus+\Ds) in a day with bad information and
liquidity shock;
}
}
\examples{
# ------------------------------------------------------------------------ #
# Generate data following the AdjPIN model using generatedata_adjpin()     #
# ------------------------------------------------------------------------ #

# With no arguments, the function generates one dataset object spanning
# 60 days, and where the parameters are chosen as described in the section
# 'Details'.

sdata <- generatedata_adjpin()

# Alternatively, simulation parameters can be provided. Recall the order of
# parameters (alpha, delta, theta, theta', eps.b, eps.s, mub, mus, db, ds).

givenpoint <- c(0.4, 0.1, 0.5, 0.6, 800, 1000, 2300, 4000, 500, 500)
sdata <- generatedata_adjpin(parameters = givenpoint)

# Data can be generated following restricted AdjPIN models, for example, with
# restrictions 'eps.b = eps.s', and 'mu.b = mu.s'.

sdata <- generatedata_adjpin(restricted = list(eps = TRUE, mu = TRUE))

# Data can be generated using provided ranges of simulation parameters as fed
# to the function using the argument 'ranges', where thetap corresponds to
# theta'.

sdata <- generatedata_adjpin(ranges = list(
  alpha = c(0.1, 0.15), delta = c(0.2, 0.2),
  theta = c(0.2, 0.6), thetap = c(0.2, 0.4)
))

# The value of a given simulation parameter can be set to a specific value by
# setting the range of the desired parameter takes a unique value, instead of
# a pair of values.

sdata <- generatedata_adjpin(ranges = list(
  alpha = 0.4, delta = c(0.2, 0.7),
  eps.b = c(100, 7000), mu.b = 8000
))

# Display the details of the generated simulation data

show(sdata)

# ------------------------------------------------------------------------ #
# Use generatedata_adjpin() to check the accuracy of adjpin()              #
# ------------------------------------------------------------------------ #

model <- adjpin(sdata@data, verbose = FALSE)

summary <- cbind(
  c(sdata@emp.pin['adjpin'], model@adjpin, abs(model@adjpin -
  sdata@emp.pin['adjpin'])),
  c(sdata@emp.pin['psos'], model@psos, abs(model@psos -
  sdata@emp.pin['psos']))
)
colnames(summary) <- c('adjpin', 'psos')
rownames(summary) <- c('Data', 'Model', 'Difference')

show(knitr::kable(summary, 'simple'))

}
\references{
\insertAllCited
}
