#' NNS Dependence
#'
#' Returns the dependence and nonlinear correlation between two variables based on higher order partial moment matrices measured by frequency or area.
#'
#' @param x a numeric vector, matrix or data frame.
#' @param y \code{NULL} (default) or a numeric vector with compatible dimensions to \code{x}.
#' @param asym logical; \code{FALSE} (default) Allows for asymmetrical dependencies.
#' @param print.map logical; \code{FALSE} (default) Plots quadrant means.
#' @return Returns the bi-variate \code{"Correlation"} and \code{"Dependence"} or correlation / dependence matrix for matrix input.
#' @note p-values and confidence intervals can be obtained from sampling random permutations of \code{y_p} and running \code{NNS.dep(x,y_p)} to compare against a null hypothesis of 0 correlation or independence between \code{x,y}.
#'
#' \code{NNS.cor} has been deprecated \code{(NNS >= 0.5.4)} and can be called via \code{NNS.dep}.
#' @author Fred Viole, OVVO Financial Systems
#' @references Viole, F. and Nawrocki, D. (2013) "Nonlinear Nonparametric Statistics: Using Partial Moments"
#' \url{ https://www.amazon.com/dp/1490523995/ref=cm_sw_su_dp}
#' @examples
#' \dontrun{
#' set.seed(123)
#' x <- rnorm(100) ; y <- rnorm(100)
#' NNS.dep(x, y)
#'
#' ## Correlation / Dependence Matrix
#' x <- rnorm(100) ; y <- rnorm(100) ; z <- rnorm(100)
#' B <- cbind(x, y, z)
#' NNS.dep(B)
#'
#'
#' ## p-values for [NNS.dep]
#' x <- seq(-5, 5, .1); y <- x^2 + rnorm(length(x))
#'
#'
#' nns_cor_dep <- NNS.dep(x, y, print.map = TRUE)
#' nns_cor_dep
#'
#' ## Create permutations of y
#' y_p <- replicate(1000, sample.int(length(y)))
#'
#' ## Generate new correlation and dependence measures on each new permutation of y
#' nns.mc <- apply(y_p, 2, function(g) NNS.dep(x, y[g]))
#'
#' ## Store results
#' cors <- unlist(lapply(nns.mc, "[[", 1))
#' deps <- unlist(lapply(nns.mc, "[[", 2))
#'
#' ## View results
#' hist(cors)
#' hist(deps)
#'
#' ## Left tailed correlation p-value
#' cor_p_value <- LPM(0, nns_cor_dep$Correlation, cors)
#' cor_p_value
#'
#' ## Right tailed correlation p-value
#' cor_p_value <- UPM(0, nns_cor_dep$Correlation, cors)
#' cor_p_value
#'
#' ## Confidence Intervals
#' ## For 95th percentile VaR (both-tails) see [LPM.VaR] and [UPM.VaR]
#' ## Lower CI
#' LPM.VaR(.025, 0, cors)
#' ## Upper CI
#' UPM.VaR(.025, 0, cors)
#'
#' ## Left tailed dependence p-value
#' dep_p_value <- LPM(0, nns_cor_dep$Dependence, deps)
#' dep_p_value
#'
#' ## Right tailed dependence p-value
#' dep_p_value <- UPM(0, nns_cor_dep$Dependence, deps)
#' dep_p_value
#' }
#' @export

NNS.dep = function(x,
                   y = NULL,
                   asym = FALSE,
                   print.map = FALSE){

  oldw <- getOption("warn")
  options(warn = -1)

  l <- length(x)

  if(!is.null(y)){

    obs <- max(10, l/5)

    # Define segments
    if(print.map) PART <- NNS.part(x, y, order = NULL, obs.req = obs, min.obs.stop = TRUE, type = "XONLY", Voronoi = TRUE)$dt else PART <- NNS.part(x, y, order = NULL, obs.req = obs, min.obs.stop = TRUE, type = "XONLY", Voronoi = FALSE)$dt

    PART[, weights := .N/l, by = prior.quadrant]
    weights <- PART[, weights[1], by = prior.quadrant]$V1

    ll <- expression(max(min(100, .N), 8))

    res <- PART[,  sign(cor(x[1:eval(ll)],y[1:eval(ll)]))*summary(lm(y[1:eval(ll)]~poly(x[1:eval(ll)], min(10, as.integer(sqrt(.N))), raw = TRUE)))$r.squared, by = prior.quadrant]
    res[is.na(res)] <- 0

    # Compare each asymmetry
    res_xy <- PART[,  sign(cor(x[1:eval(ll)],(y[1:eval(ll)])))*summary(lm(abs(y[1:eval(ll)])~poly(x[1:eval(ll)], min(10, as.integer(sqrt(.N))), raw = TRUE)))$r.squared, by = prior.quadrant]
    res_yx <- PART[,  sign(cor(y[1:eval(ll)],(x[1:eval(ll)])))*summary(lm(abs(x[1:eval(ll)])~poly(y[1:eval(ll)], min(10, as.integer(sqrt(.N))), raw = TRUE)))$r.squared, by = prior.quadrant]

    res_xy[is.na(res_xy)] <- 0
    res_yx[is.na(res_yx)] <- 0

    options(warn = oldw)

    if(asym) dependence <- sum(abs(res_xy$V1) * weights) else dependence <- max(sum(abs(res$V1) * weights),
                                                                                sum(abs(res_xy$V1) * weights),
                                                                                sum(abs(res_yx$V1) * weights))

    corr <- mean(c(sum(res$V1 * weights),
                   sum(res_xy$V1 * weights),
                   sum(res_yx$V1 * weights)))

    return(list("Correlation" = corr,
                "Dependence" = dependence))

  } else {
    return(NNS.dep.matrix(x, asym = asym))
  }

}
