% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/meta2dMainF.R
\name{meta2d}
\alias{meta2d}
\title{Detect rhythmic signals from time-series datasets with multiple methods}
\usage{
meta2d(infile, outdir = "metaout", filestyle, timepoints, minper = 20,
  maxper = 28, cycMethod = c("ARS", "JTK", "LS"),
  analysisStrategy = "auto", outputFile = TRUE, outIntegration = "both",
  adjustPhase = "predictedPer", combinePvalue = "fisher",
  weightedPerPha = FALSE, ARSmle = "auto", ARSdefaultPer = 24,
  outRawData = FALSE, releaseNote = TRUE, outSymbol = "")
}
\arguments{
\item{infile}{a character string. The name of input file containing
time-series data.}

\item{outdir}{a character string. The name of directory used to store
output files.}

\item{filestyle}{a character vector(length 1 or 3). The data format of
input file, must be \code{"txt"}, or \code{"csv"}, or a character vector
containing field separator character(\code{sep}), quoting character
(\code{quote}), and the character used for decimal points(\code{dec},
for details see \code{\link[utils]{read.table}}).}

\item{timepoints}{a numeric vector corresponding to sampling time points
of input time-series data; if sampling time points are in the first line
of input file, it could be set as a character sting-"Line1" or "line1".}

\item{minper}{a numeric value. The minimum period length of interested
rhythms. The default is \code{20} for circadian rhythms.}

\item{maxper}{a numeric value. The maximum period length of interested
rhythms. The default is \code{28} for circadian rhythms.}

\item{cycMethod}{a character vector(length 1 or 2 or 3). User-defined
methods for detecting rhythmic signals, must be selected as any one, any
two or all three methods(default) from \code{"ARS"}(ARSER),
\code{"JTK"}(JTK_CYCLE) and \code{"LS"}(Lomb-Scargle).}

\item{analysisStrategy}{a character string. The strategy used to select
proper methods from \code{cycMethod} for analyzing input time-series
data, must be \code{"auto"}(default), or \code{"selfUSE"}. See
\code{Details} part for more information.}

\item{outputFile}{logical. If \code{TRUE}, analysis results will be wrote
in the output files. If \code{FALSE}, analysis results will be returned
as an R list.}

\item{outIntegration}{a character string. This parameter controls what
kinds of analysis results will be outputted, must be one of \code{"both"}
(default), \code{"onlyIntegration"}(only output integration file), or
\code{"noIntegration"}(not output integration file).}

\item{adjustPhase}{a character string. The method used to adjust original
phase calculated by each method in integration file, must be one of
\code{"predictedPer"}(adjust phase with predicted period length) or
\code{"notAdjusted"}(not adjust phase).}

\item{combinePvalue}{a character string. The method used to integrate
multiple p-values, must be one of \code{"bonferroni"}(Bonferroni
correction), or \code{"fisher"}(Fisher's method).}

\item{weightedPerPha}{logical. If \code{TRUE}, weighted scores based on
p-value given by each method will be used to calculate the integrated
period length and phase.}

\item{ARSmle}{a character string. The strategy of using MLE method in
\code{\link[stats]{ar}} fit of \code{"ARS"}, must be one of
\code{"auto"}(use MLE depending the number of time points), \code{"mle"}
(always use MLE), or \code{"nomle"}(never use MLE).}

\item{ARSdefaultPer}{a numeric value. The expected period length of
interested rhythm, which is a necessary parameter for \code{ARS}. The
default is \code{24}(for circadian rhythms). Set it to another proper
numeric value for other rhythms.}

\item{outRawData}{logical. If \code{TRUE}, raw time-series data will be
added in the output files.}

\item{releaseNote}{logical. If \code{TRUE}, reminding or warning notes
during the analysis will be released on the screen.}

\item{outSymbol}{a character string. A common prefix exists in the names of
output files.}
}
\value{
\code{meta2d} will write analysis results in different files under
  \code{outdir} if set \code{outputFile = TRUE}. Files named with
  "ARSresult", "JTKresult" and "LSreult" store analysis results from
  \code{ARS}, \code{JTK} and \code{LS} respectively. The file named with
  "meta2d" is the integration file, and it stores integrated values in
  columns with a common name tag-"meta2d". The integration file also
  contains p-value, FDR value, period, phase(adjusted phase if
  \code{adjustedPhase = "predictedPer"}) and amplitude values calculated
  by each method.
  If \code{outputFile = FALSE} is selected, \code{meta2d} will return a
  list containing the following components:
  \tabular{rl}{
       ARS   \tab  analysis results from \code{ARS} method\cr
       JTK   \tab  analysis results from \code{JTK} method\cr
       LS    \tab  analysis results from \code{LS} method\cr
       meta  \tab  the integrated analysis results as mentioned above
  }
}
\description{
This is a function that incorporates ARSER, JTK_CYCLE and Lomb-Scargle to
  detect rhythmic signals from time-series datasets.
}
\details{
\href{https://github.com/cauyrd/ARSER}{ARSER}(Yang, 2010),
  \href{http://openwetware.org/wiki/HughesLab:JTK_Cycle}{JTK_CYCLE}(
  Hughes, 2010), and
  \href{http://research.stowers-institute.org/efg/2005/LombScargle/}{
  Lomb-Scargle}(Glynn, 2006) are three popular methods of detecting
  rhythmic signals. \code{ARS} can not analyze unevenly sampled datasets,
  or evenly sampled datasets but with missing values, or with replicate
  samples, or with non-integer sampling interval. \code{JTK} is not
  suitable to analyze unevenly sampled datasets or evenly sampled datasets
  but with non-integer sampling interval. If set \code{analysisStrategy}
  as \code{"auto"}(default), \code{meta2d} will automatically select
  proper method from \code{cycMethod} for each input dataset. If the user
  clearly know that the dataset could be analyzed by each method defined
  by \code{cycMethod} and do not hope to output integrated values,
  \code{analysisStrategy} can be set as \code{"selfUSE"}.

\code{ARS} used here is translated from its python version which always
  uses \code{"yule-walker"}, \code{"burg"}, and \code{"mle"} methods(see
  \code{\link[stats]{ar}}) to fit autoregressive models to time-series
  data. Fitting by \code{"mle"} will be very slow for datasets
  with many time points. If \code{ARSmle = "auto"} is used,
  \code{meta2d} will only include \code{"mle"} when number of time points
  is smaller than 24. In addition, one evaluation work(Wu, 2014) indicates
  that \code{ARS} shows relative high false positive rate in analyzing
  high-resolution datasets (1h/2days and 2h/2days). \code{JTK}(version 3)
  used here is the latest version, which improves its p-value calculation
  in analyzing datasets with missing values.

The power of detecting rhythmic signals for an algorithm is associated
  with the nature of data and interested periodic pattern(Deckard, 2013),
  which indicates that integrating analysis results from multiple methods
  may be helpful to rhythmic detection. For integrating p-values,
  Bonferroni correction(\code{"bonferroni"}) and Fisher's method(
  \code{"fisher"}) (Fisher, 1925; implementation code from \pkg{MADAM})
  could be selected, and \code{"bonferroni"} is usually more conservative
  than \code{"fisher"}. The integrated period is arithmetic mean of
  multiple periods. For integrating phase, \code{meta2d} takes use of
  \href{https://en.wikipedia.org/wiki/Mean_of_circular_quantities}{
  mean of circular quantities}. Integrated period and phase is further
  used to calculate the baseline value and amplitude through fitting a
  constructed periodic model.

Phases given by \code{JTK} and \code{LS} need to be adjusted with their
  predicted period (\code{adjustedPhase = "predictedPer"}) before
  integration. If \code{adjustedPhas = "notAdjusted"} is selected, no
  integrated phase will be calculated. If set \code{weightedPerPha} as
  \code{TRUE}, weighted scores will be used in averaging periods and
  phases. Weighted scores for one method are based on all its reported
  p-values, which means a weighted score assigned to any one profile will
  be affected by all other profiles. It is always a problem of averaging
  phases with quite different period lengths(eg. averaging two phases
  with 16-hours' and 30-hours' period length). Currently, setting
  \code{minper}, \code{maxper} and \code{ARSdefaultPer} to a same value
  may be the only way of completely eliminating such problem.

This function is originally aimed to analyze large scale periodic data(
  eg. circadian transcriptome data) without individual information.
  Please pay attention to data format of input file(see \code{Examples}
  part). Except the first column and first row, others are time-series
  experimental values(setting missing values as \code{NA}).
}
\examples{
# write 'cycMouseLiverProtein' into a 'txt' file
write.table(cycMouseLiverProtein, file="cycMouseLiverProtein.txt",
  sep="\\t", quote=FALSE, row.names=FALSE)
# write 'cycSimu4h2d' and 'cycYeastCycle' into two 'csv' files
write.csv(cycSimu4h2d, file="cycSimu4h2d.csv", row.names=FALSE)
write.csv(cycYeastCycle, file="cycYeastCycle.csv", row.names=FALSE)

# analyze 'cycMouseLiverProtein.txt' with JTK_CYCLE and Lomb-Scargle
meta2d(infile="cycMouseLiverProtein.txt", filestyle="txt",
  outdir="example", timepoints=rep(seq(0, 45, by=3), each=3),
  cycMethod=c("JTK","LS"), outIntegration="noIntegration")

# analyze 'cycSimu4h2d.csv' with ARSER, JTK_CYCLE and Lomb-Scargle and
# output integration file with analysis results from each method
meta2d(infile="cycSimu4h2d.csv", filestyle="csv", outdir="example",
  timepoints="Line1")

# analyze 'cycYeastCycle.csv' with ARSER, JTK_CYCLE and Lomb-Scargle to
# detect transcripts associated with cell cycle, and return analysis
# results instead of output them into files
cyc <- meta2d(infile="cycYeastCycle.csv",filestyle="csv",
  minper=80, maxper=96, timepoints=seq(2, 162, by=16),
  outputFile=FALSE, ARSdefaultPer=85, outRawData=TRUE)
head(cyc$ARS)
head(cyc$JTK)
head(cyc$LS)
head(cyc$meta)
}
\references{
Yang R. and  Su Z. (2010). Analyzing circadian expression data by
  harmonic regression based on autoregressive spectral estimation.
  \emph{Bioinformatics}, \bold{26(12)}, i168--i174.

Hughes M. E., Hogenesch J. B. and Kornacker K. (2010). JTK_CYCLE: an
  efficient nonparametric algorithm for detecting rhythmic components in
  genome-scale data sets. \emph{Journal of Biological Rhythms},
  \bold{25(5)}, 372--380.

Glynn E. F., Chen J. and Mushegian A. R. (2006). Detecting periodic
  patterns in unevenly spaced gene expression time series using
  Lomb-Scargle periodograms. \emph{Bioinformatics}, \bold{22(3)},
  310--316.

Wu G., Zhu J., Yu J., Zhou L., Huang J. Z. and  Zhang Z. (2014). Evaluation
  of five methods for genome-wide circadian gene identification.
  \emph{Journal of Biological Rhythms}, \bold{29(4)}, 231--242.

Deckard A., Anafi R. C., Hogenesch J. B., Haase S.B. and Harer J. (2013).
  Design and analysis of large-scale biological rhythm studies:
  a comparison of algorithms for detecting periodic signals in biological
  data. \emph{Bioinformatics}, \bold{29(24)}, 3174--3180.

Fisher, R.A. (1925). \emph{Statistical methods for research workers}.
  Oliver and Boyd (Edinburgh).

Kugler K. G., Mueller L.A. and Graber A. (2010). MADAM - an open source
  toolbox for meta-analysis. \emph{Source Code for Biology and Medicine},
  \bold{5}, 3.
}

