% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mvn.R
\name{mvn}
\alias{mvn}
\title{Comprehensive Multivariate Normality and Diagnostic Function}
\usage{
mvn(
  data,
  subset = NULL,
  mvn_test = "hz",
  use_population = TRUE,
  tol = 1e-25,
  alpha = 0.05,
  scale = FALSE,
  descriptives = TRUE,
  transform = "none",
  R = 1000,
  univariate_test = "AD",
  multivariate_outlier_method = "none",
  box_cox_transform = FALSE,
  box_cox_transform_type = "optimal",
  show_new_data = FALSE,
  tidy = TRUE
)
}
\arguments{
\item{data}{A numeric matrix or data frame (rows = observations, columns = variables).}

\item{subset}{Optional character; name of a grouping variable in \code{data} for subset analyses.}

\item{mvn_test}{Character; one of \code{"mardia"}, \code{"hz"}, \code{"royston"}, 
\code{"doornik_hansen"}, or \code{"energy"}. Default: \code{"hz"}.}

\item{use_population}{Logical; if \code{TRUE}, uses the population covariance estimator \eqn{\frac{n-1}{n} \times \Sigma}; otherwise uses the sample covariance. Default is \code{TRUE}.}

\item{tol}{Numeric; tolerance for matrix inversion via \code{solve()}. Default: \code{1e-25}.}

\item{alpha}{Numeric; significance level for ARW outlier cutoff when 
\code{multivariate_outlier_method = "adj"}. Default: \code{0.05}.}

\item{scale}{Logical; if \code{TRUE}, standardizes the data before analysis. Default: \code{FALSE}.}

\item{descriptives}{Logical; if \code{TRUE}, compute descriptive statistics. Default: \code{TRUE}.}

\item{transform}{Character; one of \code{"none"}, \code{"log"}, \code{"sqrt"}, \code{"square"}.
Applies marginal transformations before analysis. Default: \code{"none"}.}

\item{R}{Integer; number of bootstrap replicates for \code{"energy"} test. Default: \code{1000}.}

\item{univariate_test}{Character; one of \code{"SW"}, \code{"CVM"}, \code{"Lillie"}, 
\code{"SF"}, \code{"AD"}. Default: \code{"AD"}.}

\item{multivariate_outlier_method}{Character; \code{"none"}, \code{"quan"}, or \code{"adj"}.
Default: \code{"none"}.}

\item{box_cox_transform}{Logical; if \code{TRUE}, applies Box-Cox transformation to all variables. Default: \code{FALSE}.}

\item{box_cox_transform_type}{Character; either \code{"optimal"} or \code{"rounded"} lambda for Box-Cox. Default: \code{"optimal"}.}

\item{show_new_data}{Logical; if \code{TRUE}, include cleaned data (non-outliers). Default: \code{FALSE}.}

\item{tidy}{Logical; if \code{TRUE}, returns the results as a tidy data frame with an added \code{Group} column. Default is \code{TRUE}.}
}
\value{
A named list containing:
\describe{
  \item{multivariate_normality}{A data frame of the selected multivariate normality (MVN) test results.}
  \item{univariate_normality}{A data frame of univariate normality test results.}

  \item{descriptives}{(Optional) A data frame of descriptive statistics if \code{descriptives = TRUE}.}
  \item{multivariate_outliers}{(Optional) A data frame of flagged multivariate outliers if \code{multivariate_outlier_method != "none"}.}
  \item{new_data}{(Optional) Original data with multivariate outliers removed if \code{show_new_data = TRUE}.}
  \item{box_cox_lambda}{(Optional) Estimated Box-Cox lambda values if \code{box_cox_transform = TRUE}.}
  \item{data}{The processed data matrix used in the analysis (transformed and/or cleaned).}
  \item{subset}{(Optional) The grouping variable used for subset analysis, if applicable.}
}
}
\description{
Conduct multivariate normality tests, outlier detection, univariate normality tests,
descriptive statistics, and Box-Cox transformation in one wrapper.
}
\details{
If \code{mvn_test = "mardia"}, it calculates the Mardia's multivariate skewness and kurtosis coefficients as well as their corresponding statistical significance.
It can also calculate corrected version of skewness coefficient for small sample size (n< 20).
For multivariate normality, both p-values of skewness and kurtosis statistics should be greater than 0.05.
If sample size less than 20 then p.value.small should be used as significance value of skewness instead of p.value.skew.
If there are missing values in the data, a listwise deletion will be applied and a complete-case analysis will be performed.

If \code{mvn_test = "hz"}, it calculates the Henze-Zirkler's multivariate normality test. The Henze-Zirkler test is based on a non-negative functional distance that measures the distance between two distribution functions. If the data is multivariate normal, the test statistic HZ is approximately lognormally distributed. It proceeds to calculate the mean, variance and smoothness parameter. Then, mean and variance are lognormalized and the p-value is estimated.
If there are missing values in the data, a listwise deletion will be applied and a complete-case analysis will be performed.

If \code{mvn_test = "royston"}, it calculates the Royston's multivariate normality test. A function to generate the Shapiro-Wilk's W statistic needed to feed the Royston's H test for multivariate normality However, if kurtosis of the data greater than 3 then Shapiro-Francia test is used for leptokurtic samples else Shapiro-Wilk test is used for platykurtic samples.
If there are missing values in the data, a listwise deletion will be applied and a complete-case analysis will be performed. Do not apply Royston's test, if dataset includes more than 5000 cases or less than 3 cases, since it depends on Shapiro-Wilk's test.

If \code{mvn_test = "doornik_hansen"}, it calculates the Doornik-Hansen's multivariate normality test. The code is adapted from asbio package (Aho, 2017).

If \code{mvn_test = "energy"}, it calculates the Energy multivariate normality test. The code is adapted from energy package (Rizzo and Szekely, 2017).
}
\examples{
result = mvn(data = iris[-4], subset = "Species", mvn_test = "hz",
             univariate_test = "AD", 
             multivariate_outlier_method = "adj",
             show_new_data = TRUE)

### Multivariate Normality Result
summary(result, select = "mvn")

### Univariate Normality Result
summary(result, select = "univariate")

### Descriptives
summary(result, select = "descriptives")

### Multivariate Outliers
summary(result, select = "outliers")

### New data without multivariate outliers
summary(result, select = "new_data")


}
\references{
Korkmaz S, Goksuluk D, Zararsiz G. MVN: An R Package for Assessing Multivariate Normality. The R Journal. 2014 6(2):151-162. URL \url{https://journal.r-project.org/archive/2014-2/korkmaz-goksuluk-zararsiz.pdf}

Mardia, K. V. (1970), Measures of multivariate skewnees and kurtosis with applications. Biometrika, 57(3):519-530.

Mardia, K. V. (1974), Applications of some measures of multivariate skewness and kurtosis for testing normality and robustness studies. Sankhy A, 36:115-128.

Henze, N. and Zirkler, B. (1990), A Class of Invariant Consistent Tests for Multivariate Normality. Commun. Statist.-Theor. Meth., 19(10): 35953618.

Henze, N. and Wagner, Th. (1997), A New Approach to the BHEP tests for multivariate normality. Journal of Multivariate Analysis, 62:1-23.

Royston, J.P. (1982). An Extension of Shapiro and Wilks W Test for Normality to Large Samples. Applied Statistics, 31(2):115124.

Royston, J.P. (1983). Some Techniques for Assessing Multivariate Normality Based on the Shapiro-Wilk W. Applied Statistics, 32(2).

Royston, J.P. (1992). Approximating the Shapiro-Wilk W-Test for non-normality. Statistics and Computing, 2:117-119.121133.

Royston, J.P. (1995). Remark AS R94: A remark on Algorithm AS 181: The W test for normality. Applied Statistics, 44:547-551.

Shapiro, S. and Wilk, M. (1965). An analysis of variance test for normality. Biometrika, 52:591611.

Doornik, J.A. and Hansen, H. (2008). An Omnibus test for univariate and multivariate normality. Oxford Bulletin of Economics and Statistics 70, 927-939.

G. J. Szekely and M. L. Rizzo (2013). Energy statistics: A class of statistics based on distances, Journal of Statistical Planning and Inference, http://dx.doi.org/10.1016/j.jspi.2013.03.018

M. L. Rizzo and G. J. Szekely (2016). Energy Distance, WIRES Computational Statistics, Wiley, Volume 8 Issue 1, 27-38. Available online Dec., 2015, http://dx.doi.org/10.1002/wics.1375.

G. J. Szekely and M. L. Rizzo (2017). The Energy of Data. The Annual Review of Statistics and Its Application 4:447-79. 10.1146/annurev-statistics-060116-054026
}
\author{
Selcuk Korkmaz, \email{selcukorkmaz@gmail.com}
}
