
#' @describeIn multiMSE Simulate historical dynamics for multi-OM
#'
#' @export
#'
SimulateMOM <- function(MOM=MSEtool::Albacore_TwoFleet, parallel=TRUE, silent=FALSE) {
  # ---- Initial Checks and Setup ----
  if (class(MOM) == 'MOM') {
    if (MOM@nsim <=1) stop("MOM@nsim must be > 1", call.=FALSE)

  } else {
    stop("You must specify an operating model of class `MOM`")
  }

  # ---- Set up parallel processing ----
  ncpus <- set_parallel(parallel)

  set.seed(MOM@seed) # set seed for reproducibility
  nsim <- MOM@nsim
  nyears <- MOM@Fleets[[1]][[1]]@nyears  # number of historical years
  proyears <- MOM@proyears
  allyears <- nyears+proyears
  Stocks <- MOM@Stocks
  Fleets <- MOM@Fleets
  Obs <- MOM@Obs
  Imps <- MOM@Imps
  Rel <- MOM@Rel
  SexPars <- MOM@SexPars
  Complexes <- MOM@Complexes
  CatchFrac <- MOM@CatchFrac

  np <- length(Stocks)
  nf <- length(Fleets[[1]])

  if(np==1 & nf==1){
    message("You have specified only a single stock and fleet. ",
            "You should really be using the function MSEtool::runMSE()")
  } else if(np>1 & length(MOM@Rel)==0 & length(MOM@SexPars)==0) {
    message("You have specified more than one stock but no MICE relationships ",
            "(slot MOM@Rel) or sex-specific relationships (slot MOM@SexPars) among these. ",
            "As they are independent, consider doing MSE for one stock at a time for ",
            "computational efficiency.")
  }

  maxF <- MOM@maxF
  Snames <- SIL(Stocks,"Name")
  Fnames <- matrix(make.unique(SIL(Fleets,"Name")),nrow=nf)
  cpars <- MOM@cpars

  # ---- Custom Parameters (cpars) Options ----
  control <- cpars$control; cpars$control <- NULL

  # Option to optimize depletion for vulnerable biomass instead of spawning biomass
  optVB <- FALSE
  if (!is.null(control$D) && control$D == "VB") optVB <- TRUE

  # Allocation
  if(length(MOM@Allocation)==0){
    MOM@Allocation <- CatchFrac
    message("Slot @Allocation of MOM object not specified. Setting slot ",
            "@Allocation equal to slot @CatchFrac - current catch fractions")
  }

  if(length(MOM@Efactor)==0){
    MOM@Efactor <- list()
    for(p in 1:np) MOM@Efactor[[p]]<- array(1,c(nsim,nf))
    message("Slot @Efactor of MOM object not specified. Setting slot @Efactor ",
            "to current effort for all fleets")
  }

  # All stocks and sampled parameters must have compatible array sizes (maxage)
  maxage_s <- unique(SIL(MOM@Stocks,"maxage"))
  if (length(maxage_s)>1)
    message(paste("Stocks of varying maximum ages have been specified,",
                  "all simulations will run to",max(maxage_s),"ages"))
  maxage <- max(maxage_s)
  for(p in 1:np) MOM@Stocks[[p]]@maxage<-maxage

  if(!silent) message("Loading operating model")
  StockPars<-FleetPars<-ObsPars<-ImpPars<- SampCpars<-new('list')

  plusgroup <- rep(1, np)

  for(p in 1:np){
    SampCpars[[p]]<-list()
    if(!silent) message(Stocks[[p]]@Name)
    for(f in 1:nf){
      # --- Sample custom parameters ----

      if(length(cpars)>0 && length(cpars[[p]][[f]])>0){
        if(!silent)
          message('Sampling custom parameters for ', Fleets[[p]][[f]]@Name)
        SampCpars[[p]][[f]] <- SampleCpars(cpars[[p]][[f]], nsim,
                                           silent=silent)
      }else{
        SampCpars[[p]][[f]] <-list()
      }
    }

    set.seed(MOM@seed) # set seed again after cpars has been sampled
    # --- Sample Stock Parameters ----
    if(!is.null(SampCpars[[p]][[1]]$plusgroup) & all(SampCpars[[p]][[1]]$plusgroup==0))
      plusgroup[p] <- 0

    StockPars[[p]] <- SampleStockPars(MOM@Stocks[[p]], nsim, nyears,
                                      proyears, SampCpars[[p]][[1]],
                                      msg=!silent)
    StockPars[[p]]$plusgroup <- plusgroup[p]
    StockPars[[p]]$maxF <- MOM@maxF

    # --- Sample Fleet Parameters ----
    FleetPars[[p]]<-ObsPars[[p]]<-ImpPars[[p]]<-list()
    for(f in 1:nf){
      FleetPars[[p]][[f]] <- SampleFleetPars(Fleet=MOM@Fleets[[p]][[f]],
                                             Stock=StockPars[[p]],
                                             nsim, nyears, proyears,
                                             cpars=SampCpars[[p]][[f]])

    }

    # --- Sample Obs Parameters ----
    for(f in 1:nf) {
      ObsPars[[p]][[f]] <- SampleObsPars(MOM@Obs[[p]][[f]], nsim,
                                         cpars=SampCpars[[p]][[f]],
                                         Stock=StockPars[[p]],
                                         nyears, proyears)
    }

    # --- Sample Imp Parameters ----
    for(f in 1:nf) {
      ImpPars[[p]][[f]] <- SampleImpPars(MOM@Imps[[p]][[f]], nsim,
                                         cpars=SampCpars[[p]][[f]],
                                         nyears, proyears)
    }
  }

  # --- Update Parameters for two-sex stocks ----
  # Depletion, stock-recruit parameters, recdevs, Fleet, Obs, and Imp copied
  # from females to males
  if(length(SexPars)>0){
    sexmatches <- sapply(1:nrow(SexPars$SSBfrom), function(x,mat)
      paste(mat[x,],collapse="_"), mat=SexPars$SSBfrom)
    parcopy<-match(sexmatches,sexmatches)
    StockPars_t<-StockPars # need to store a temporary object for copying to/from
    FleetPars_t <- FleetPars

    for(p in 1:np){
      # copied parameters
      StockPars[[p]]$D<-StockPars_t[[parcopy[p]]]$D
      StockPars[[p]]$hs<-StockPars_t[[parcopy[p]]]$hs
      StockPars[[p]]$AC<-StockPars_t[[parcopy[p]]]$AC
      StockPars[[p]]$R0<-StockPars_t[[parcopy[p]]]$R0
      StockPars[[p]]$R0a<-StockPars_t[[parcopy[p]]]$R0a
      StockPars[[p]]$Perr_y<-StockPars_t[[parcopy[p]]]$Perr_y
      for (f in 1:nf) {
        # copy over Fleet, Obs and Imps pars
        FleetPars[[p]][[f]]$Esd <- FleetPars_t[[parcopy[p]]][[f]]$Esd
        FleetPars[[p]][[f]]$Find <- FleetPars_t[[parcopy[p]]][[f]]$Find
        FleetPars[[p]][[f]]$dFFinal <- FleetPars_t[[parcopy[p]]][[f]]$dFFinal
        FleetPars[[p]][[f]]$Spat_targ <- FleetPars_t[[parcopy[p]]][[f]]$Spat_targ
        FleetPars[[p]][[f]]$qinc <- FleetPars_t[[parcopy[p]]][[f]]$qinc
        FleetPars[[p]][[f]]$qcv <- FleetPars_t[[parcopy[p]]][[f]]$qcv
        FleetPars[[p]][[f]]$qvar <- FleetPars_t[[parcopy[p]]][[f]]$qvar
        FleetPars[[p]][[f]]$FinF <- FleetPars_t[[parcopy[p]]][[f]]$FinF
        ObsPars[[p]][[f]] <- ObsPars[[parcopy[p]]][[f]]
        ImpPars[[p]][[f]] <- ImpPars[[parcopy[p]]][[f]]
      }
    }
  } # end of sexpars

  nareas_s <- NIL(StockPars,"nareas",lev1=T)
  if(length(unique(nareas_s))!=1)
    stop("Stocks must have the same specified number of areas - check cpars$mov",
         " for each stock object")
  nareas <- as.numeric(nareas_s[1])

  # ---- Bio-Economic Parameters ----
  # TODO

  # ---- Initialize arrays ----
  n_age <- maxage + 1 # number of age classes (starting at age-0)
  N <- Biomass <- Z<- VBiomass<- SSN <- SSB <- array(NA,
                                                     dim = c(nsim, np, n_age,
                                                             nyears, nareas))
  VF <- FretA <- array(NA, dim = c(nsim, np, nf, n_age, allyears))
  VBF <- FM <- FMret <- array(NA, dim = c(nsim, np, nf, n_age, nyears, nareas))
  SPR <- array(NA, dim = c(nsim, np, n_age, nyears)) # store the Spawning Potential Ratio
  MPA <- array(1,c(np,nf, nyears+proyears,nareas))
  Agearray <- array(rep(1:n_age, each = nsim), dim = c(nsim, n_age))  # Age array

  # ---- Hermaphroditism -----
  # (this is the fraction to be kept (after sex change))
  # E.g. protygynous (Female to male) is H_1_2 where 1 is female 2 is male
  # [sim, stock, maxage] Defaults to all 1s if length(SexPars$Herm)==0
  HermFrac <- expandHerm(SexPars$Herm,maxage=n_age,np=np,nsim=nsim)

  Unfished_Equilibrium <- list()
  for(p in 1:np){ # loop over stocks
    #  --- Pre Equilibrium calcs ----
    surv <- matrix(1, nsim, n_age)
    surv[, 2:n_age] <- t(exp(-apply(StockPars[[p]]$M_ageArray[,,1], 1, cumsum)))[, 1:(n_age-1)]

    if (plusgroup[p]) {
      surv[,n_age] <- surv[,n_age]+surv[,n_age]*
        exp(-StockPars[[p]]$M_ageArray[,n_age,1])/(1-exp(-StockPars[[p]]$M_ageArray[,n_age,1]))
    }

    # predicted Numbers of mature ages in first year
    Nfrac <- surv * StockPars[[p]]$Mat_age[,,1] * HermFrac[,p,]

    # Set up some array indexes sim (S) age (A) year (Y) region/area (R)
    SPAYR <- as.matrix(expand.grid(1:nareas, 1, 1:n_age, p, 1:nsim)[5:1])
    SPA <- SPAYR[,1:3]
    SAY <- SPAYR[, c(1,3,4)]
    SAR <- SPAYR[, c(1,3,5)]
    SA <- Sa <- SPAYR[, c(1,3)]
    SR <- SPAYR[, c(1,5)]
    S <- SPAYR[, 1]
    SY <- SPAYR[, c(1, 4)]
    Sa[,2] <- n_age-Sa[,2]+1 # This is the process error index for initial year

    # Calculate initial distribution if mov provided in cpars
    # ---- Calculate initial distribution if mov provided in cpars ----
    if (is.null(StockPars[[p]]$initdist)) {
      # mov has been passed in cpars - initdist hasn't been defined
      StockPars[[p]]$initdist <- CalcDistribution(StockPars[[p]],
                                                  FleetPars[[p]][[1]],
                                                  SampCpars[[p]][[1]],
                                                  nyears, maxF,
                                                  plusgroup[p], checks=FALSE)
    }

    #*HermFrac[,p,1]  # !!!! INITDIST OF AGE 1. Unfished recruitment by area
    R0a <- matrix(StockPars[[p]]$R0, nrow=nsim, ncol=nareas, byrow=FALSE) *
      StockPars[[p]]$initdist[,1,]

    # ---- Unfished Equilibrium calcs ----
    # unfished survival for every year
    # Survival array
    surv <- array(1, dim=c(nsim, n_age, nyears+proyears))
    surv[, 2:n_age, ] <- aperm(exp(-apply(StockPars[[p]]$M_ageArray, c(1,3), cumsum))[1:(n_age-1), ,],
                               c(2,1,3))
    if (plusgroup[p]) {
      surv[,n_age, ] <- surv[,n_age,]+surv[,n_age,]*
        apply(-StockPars[[p]]$M_ageArray[,n_age,], 2, exp)/(1-apply(-StockPars[[p]]$M_ageArray[,n_age,],
                                                                    2, exp))
    }
    Nfrac <- surv * StockPars[[p]]$Mat_age  # predicted numbers of mature ages in all years

    # indices for all years
    SAYR_a <- as.matrix(expand.grid(1:nareas, 1:(nyears+proyears), 1:n_age, 1:nsim)[4:1])
    SAY_a <- SAYR_a[, 1:3]
    SAR_a <- SAYR_a[, c(1,2,4)]
    SA_a <- SAYR_a[, 1:2]
    SR_a <- SAYR_a[, c(1, 4)]
    S_a <- SAYR_a[, 1]
    SY_a <- SAYR_a[, c(1, 3)]

    # arrays for unfished biomass for all years
    SSN_a <- array(NA, dim = c(nsim, n_age, nyears+proyears, nareas))
    N_a <- array(NA, dim = c(nsim, n_age, nyears+proyears, nareas))
    Biomass_a <- array(NA, dim = c(nsim, n_age, nyears+proyears, nareas))
    SSB_a <- array(NA, dim = c(nsim, n_age, nyears+proyears, nareas))

    # Calculate initial spawning stock numbers for all years
    SSN_a[SAYR_a] <- Nfrac[SAY_a] * StockPars[[p]]$R0[S_a] * StockPars[[p]]$initdist[SAR_a]
    N_a[SAYR_a] <- StockPars[[p]]$R0[S_a] * surv[SAY_a] * StockPars[[p]]$initdist[SAR_a]
    Biomass_a[SAYR_a] <- N_a[SAYR_a] * StockPars[[p]]$Wt_age[SAY_a]  # Calculate initial stock biomass
    SSB_a[SAYR_a] <- SSN_a[SAYR_a] * StockPars[[p]]$Wt_age[SAY_a]    # Calculate spawning stock biomass

    SSN0_a <- apply(SSN_a, c(1,3), sum) # unfished spawning numbers for each year
    N0_a <- apply(N_a, c(1,3), sum) # unfished numbers for each year)
    SSB0_a <- apply(SSB_a, c(1,3), sum) # unfished spawning biomass for each year
    SSB0a_a <- apply(SSB_a, c(1, 3,4), sum)  # Calculate unfished spawning stock biomass by area for each year
    B0_a <- apply(Biomass_a, c(1,3), sum) # unfished biomass for each year

    Vraw <- array(NIL(listy=FleetPars[[p]],namey="V_real"),c(nsim,n_age,allyears,nf))
    Vind <- as.matrix(expand.grid(1:nsim,p,1:nf,1:n_age,1:allyears))
    VF[Vind] <- Vraw[Vind[,c(1,4,5,3)]]

    Fretraw <- array(NIL(listy=FleetPars[[p]],namey="retA_real"),c(nsim,n_age,allyears,nf))
    FretA[Vind] <- Fretraw[Vind[,c(1,4,5,3)]]
    
    if(nf==1){
      V <- VF[,p,1,,] #<-SOL(FleetPars[[p]],"V")
    }else{
      #Weight by catch fraction
      V <- array(0,c(nsim,n_age,allyears))
      for(f in 1:nf){
        V <- V+VF[,p,f,,]*CatchFrac[[p]][,f]
      }
      #V<-nlz(V,c(1,3),"max") # currently assume unfished vulnerability is equally weighted among fleets
      # V includes discards
    }
    # unfished vulnerable biomass for each year
    VB0_a <- apply(apply(Biomass_a, c(1,2,3), sum) * V, c(1,3), sum)

    # ---- Unfished Reference Points ----
    SSBpRa <- array(SSB0_a/matrix(StockPars[[p]]$R0, nrow=nsim, ncol=nyears+proyears),
                    dim = c(nsim, nyears+proyears))

    UnfishedRefs <- sapply(1:nsim, CalcUnfishedRefs, ageM=StockPars[[p]]$ageM, N0_a=N0_a, SSN0_a=SSN0_a,
                           SSB0_a=SSB0_a, B0_a=B0_a, VB0_a=VB0_a, SSBpRa=SSBpRa, SSB0a_a=SSB0a_a)

    N0 <- UnfishedRefs[1,] %>% unlist() # average unfished numbers
    SSN0 <- UnfishedRefs[2,] %>% unlist() # average unfished spawning numbers
    SSB0 <- UnfishedRefs[3,] %>% unlist() # average unfished spawning biomass
    B0 <- UnfishedRefs[4,] %>% unlist() # average unfished biomass
    VB0 <- UnfishedRefs[5,] %>% unlist() # average unfished vulnerable biomass

    Unfished_Equilibrium[[p]] <- list(
      N_at_age=N_a,
      B_at_age=Biomass_a,
      SSB_at_age=SSB_a,
      SSN_at_age=SSN_a,
      VB_at_age=Biomass_a * replicate(nareas,V)
    )

    # average spawning stock biomass per recruit
    SSBpR <- matrix(UnfishedRefs[6,] %>% unlist(), nrow=nsim, ncol=nareas)

    # average unfished biomass
    SSB0a <- UnfishedRefs[7,] %>% unlist() %>% matrix(nrow=nsim, ncol=nareas, byrow = TRUE)
    bR <- matrix(log(5 * StockPars[[p]]$hs)/(0.8 * SSB0a), nrow=nsim)  # Ricker SR params
    aR <- matrix(exp(bR * SSB0a)/SSBpR, nrow=nsim)  # Ricker SR params

    # --- Optimize for Initial Depletion ----
    # currently done in SS2MOM

    # initD <- SampCpars[[p]][[1]]$initD #
    # if (!is.null(initD)) { # initial depletion is not unfished
    #   if (!silent)
    #     message("Optimizing for user-specified depletion in first historical year for ", Snames[p])
    #   Perrmulti <- sapply(1:nsim, optDfunwrap,
    #                       initD=initD,
    #                       Nfrac=Nfrac,
    #                       R0=R0,
    #                       Perr_y=StockPars[[p]]$Perr_y,
    #                       surv=surv,
    #                       Wt_age=StockPars[[p]]$Wt_age,
    #                       SSB0=SSB0,
    #                       n_age=n_age)
    #
    #   StockPars[[p]]$Perr_y[,1:n_age] <- StockPars[[p]]$Perr_y[, 1:n_age] * Perrmulti
    # }
    #

    #  --- Non-equilibrium Initial Year ----
    SSN[SPAYR] <- Nfrac[SAY] * StockPars[[p]]$R0[S] * StockPars[[p]]$initdist[SAR] *
      StockPars[[p]]$Perr_y[Sa]
    # Calculate initial stock numbers
    N[SPAYR] <- StockPars[[p]]$R0[S] * surv[SAY] * HermFrac[SPA] *
      StockPars[[p]]$initdist[SAR] * StockPars[[p]]$Perr_y[Sa]

    # Calculate initial stock biomass
    Biomass[SPAYR] <- N[SPAYR] * StockPars[[p]]$Wt_age[SAY]
    # Calculate spawning stock biomass
    SSB[SPAYR] <- SSN[SPAYR] * StockPars[[p]]$Wt_age[SAY]
    # Calculate vunerable biomass
    VBiomass[SPAYR] <- Biomass[SPAYR] * V[SAY]

    # Assign stock parameters to StockPars object
    StockPars[[p]]$SSBpR <- SSBpR
    StockPars[[p]]$aR <- aR
    StockPars[[p]]$bR <- bR
    StockPars[[p]]$SSB0 <- SSB0
    StockPars[[p]]$SSN0 <- SSN0
    StockPars[[p]]$VB0 <- VB0
    StockPars[[p]]$R0a <- R0a
    StockPars[[p]]$surv <- surv
    StockPars[[p]]$B0 <- B0
    StockPars[[p]]$N0 <- N0

    # loop over fleets
    for(f in 1:nf) {
      FleetPars[[p]][[f]]$V_real<-VF[,p,f,,] # update fleet vulnerability for this stock

      # --- Historical Spatial closures ----
      if (!is.null(SampCpars[[p]][[f]]$MPA)) {
        MPA <- SampCpars[[p]][[f]]$MPA
        if (any(dim(MPA) != c(nyears+proyears, nareas))) {
          stop('cpars$MPA must be a matrix with `nyears+proyears` rows and `nareas` columns', .call=FALSE)
        }
        if (any(MPA !=1 & MPA!=0))
          stop('values in cpars$MPA must be either 0 (closed) or open (1)', .call=FALSE)
        if (any(MPA!=1)) {
          for (a in 1:nareas) {
            yrs <- which(MPA[,a] == 0)
            if (length(yrs)>0) {
              if (!silent)
                message('Spatial closure detected in area ', a, ' in years ',
                        paste(findIntRuns(yrs), collapse=", "))
            }
          }
        }
      } else {
        MPA <- matrix(1, nrow=nyears+proyears, ncol=nareas)
        if (!is.na(FleetPars[[p]][[f]]$MPA) && all(FleetPars[[p]][[f]]$MPA==TRUE)) {
          MPA[,1] <- 0
          if (!silent) message('Historical MPA in Area 1 for all years')

        }

      }
      FleetPars[[p]][[f]]$MPA <- MPA
      if (any(MPA!=1))
        message('NOTE: MPA detected for Fleet ', f, ' but currently NOT implemented in multiMSE')

    } # end of loop over fleets
  } # end of loop over stocks

  # ---- SexPars - Update SSB0 and SRR parameters for male stock ----
  if(length(SexPars)>0){
    message("You have specified sex-specific dynamics, unfished spawning biomass",
            " and specified stock depletion will be mirrored across sex types according ",
            "to SexPars$SSBfrom")

    SSB0s<-matrix(NIL(StockPars,"SSB0"),nrow=nsim) # sim, p
    sexmatches<-sapply(1:nrow(SexPars$SSBfrom),function(x,mat)paste(mat[x,],collapse="_"), mat=SexPars$SSBfrom)
    parcopy<-match(sexmatches,sexmatches)
    StockPars_t<-StockPars # need to store a temporary object for copying to/from

    for(p in 1:np){

      SSB0<-apply(matrix(rep(SexPars$SSBfrom[p,],each=nsim),nrow=nsim)*SSB0s,1,sum)
      StockPars[[p]]$SSB0 <- SSB0
      # !!!!!!!!!!! SSBpR hardwired to be the same among areas !!!!
      StockPars[[p]]$SSBpR <- array(SSB0/StockPars[[p]]$R0,c(nsim,nareas))

      idist<-StockPars[[p]]$R0a/apply(StockPars[[p]]$R0a,1,sum)
      SSB0a<-SSB0*idist

      # Ricker SR params
      StockPars[[p]]$bR <- matrix(log(5 * StockPars[[p]]$hs)/(0.8 * SSB0a),
                                  nrow=nsim)
      StockPars[[p]]$aR <- matrix(exp(StockPars[[p]]$bR * SSB0a)/StockPars[[p]]$SSBpR,
                                  nrow=nsim)

    }
    if(length(SexPars$Herm)>0){
      message("You have specified sequential hermaphroditism (SexPars$Herm).",
              "Unfished stock numbers will be calculated from this vector of fractions ",
              "at age. Population dynamics will move individuals from one sex to another.")
    }
  } # end of SexPars loop

  # --- Optimize catchability (q) to fit depletion ----
  optD <- TRUE

  # skip optimization if qs are provided in cpars
  qs <- matrix(NA, nrow=nsim, ncol=np)
  qfrac <- array(1, dim=c(nsim, np, nf))
  for (p in 1:np) {
    for (f in 1:nf) {
      if(!is.null(SampCpars[[p]][[f]]$qs)) {
        optD <- FALSE
        qs[,p] <- SampCpars[[p]][[f]]$qs
        FleetPars[[p]][[f]]$qs<-qs[,p]*qfrac[,p,f]
      }
    }
  }

  bounds <- c(0.0001, 15) # q bounds for optimizer
  if (optD) {
    if(snowfall::sfIsRunning() & parallel){
      exp.time <- (np * nf)/(9*ncpus) * nsim
      exp.time <- round(exp.time,2)
#
#       message("Optimizing for user-specified depletion ",
#               'using parallel processing',
#               "(takes approximately [(nstocks x nfleets)/(9 x number of cores in cluster)]",
#               " minutes per simulation): about", exp.time, 'minutes')
      if(!silent)
        message("Optimizing for user-specified depletion ",
              'using parallel processing',
              'for ', nsim, 'simulations,', np, ' stocks, and ', nf, 'fleets',
              "(could take a while!)")

      out<-snowfall::sfLapply(1:nsim, getq_multi_MICE, StockPars, FleetPars,
                              np,nf, nareas, maxage, nyears, N, VF, FretA,
                              maxF=MOM@maxF, MPA, CatchFrac, bounds=bounds,
                              tol=1E-6,Rel,SexPars, plusgroup=plusgroup, optVB=optVB)

    }else{
      exp.time <- (np * nf)/(9) * nsim
      exp.time <- round(exp.time,2)

      if(!silent)
        message("Optimizing for user-specified depletion ",
                'using a single core',
                'for ', nsim, 'simulations,', np, ' stocks, and ', nf, 'fleets',
                "(could take a while!)")

      out<-lapply(1:nsim,getq_multi_MICE, StockPars, FleetPars, np, nf, nareas,
                  maxage, nyears, N, VF, FretA, maxF=MOM@maxF,
                  MPA,CatchFrac, bounds=bounds,tol=1E-6,Rel,SexPars,
                  plusgroup=plusgroup, optVB=optVB)

    }

    qs <- t(matrix(NIL(out,"qtot"),nrow=np))
    qfrac <- aperm(array(NIL(out,"qfrac"),c(np,nf,nsim)),c(3,1,2))

    for(p in 1:np){
      for(f in 1:nf){
        FleetPars[[p]][[f]]$qs<-qs[,p]*qfrac[,p,f]
      }
    }
  }


  # --- Check that q optimizer has converged ----
  # bounds for q (catchability). Flag if bounded optimizer hits the bounds
  LimBound <- c(1.1, 0.9)*range(bounds)
  probQ <- which(apply(qs > max(LimBound) | qs < min(LimBound),1,sum)>0)
  Nprob <- length(probQ)

  # If q has hit bound, re-sample depletion and try again.
  # Tries 'ntrials' times and then alerts user
  fracD <- 0.05; ntrials <- 50
  if (!is.null(control$ntrials)) ntrials <- control$ntrials
  if (!is.null(control$ntrials)) fracD <- control$fracD

  if (length(probQ) > 0) {
    Err <- TRUE
    if(!silent) message(Nprob,
                        ' simulations have final biomass that is not ',
                        'close to sampled depletion')
    if(!silent) message('Re-sampling depletion, recruitment error and fishing effort')

    count <- 0
    MOM2 <- MOM
    while (Err & count < ntrials) {
      Nprob <- length(probQ)
      SampCpars2 <- vector("list", nf)
      for(p in 1:np){
        for(f in 1:nf){
          if(length(cpars)>0 && length(cpars[[p]][[f]])>0){
            # check each list object has the same length and if not stop and error report
            ncparsim <- cparscheck(cpars[[p]][[f]])
            SampCpars2[[f]] <- SampleCpars(cpars[[p]][[f]], Nprob, silent=silent)
          }
        }

        ResampStockPars <- SampleStockPars(MOM2@Stocks[[p]],
                                           nsim=Nprob,nyears=nyears,
                                           proyears=proyears,
                                           cpars=SampCpars2[[1]],
                                           msg=FALSE)

        ResampStockPars$CAL_bins <- StockPars[[p]]$CAL_bins
        ResampStockPars$CAL_binsmid <- StockPars[[p]]$CAL_binsmid
        ResampStockPars$nCALbins <- length(StockPars[[p]]$CAL_binsmid )

        # Re-sample depletion
        StockPars[[p]]$D[probQ] <- ResampStockPars$D

        # Re-sample recruitment deviations
        StockPars[[p]]$procsd[probQ] <- ResampStockPars$procsd
        StockPars[[p]]$AC[probQ] <- ResampStockPars$AC
        StockPars[[p]]$Perr_y[probQ,] <- ResampStockPars$Perr_y
        StockPars[[p]]$hs[probQ] <- ResampStockPars$hs
      } # end of P
      # Re-sample historical fishing effort
      ResampFleetPars<- vector("list", nf)
      for(p in 1:np){
        for(f in 1:nf){
          ResampFleetPars <- SampleFleetPars(MOM2@Fleets[[p]][[f]],
                                             Stock=ResampStockPars,
                                             nsim=Nprob,
                                             nyears=nyears,
                                             proyears=proyears,
                                             cpars=SampCpars2[[f]])
          FleetPars[[p]][[f]]$Esd[probQ] <- ResampFleetPars$Esd
          FleetPars[[p]][[f]]$Find[probQ, ] <- ResampFleetPars$Find
          FleetPars[[p]][[f]]$dFfinal[probQ] <- ResampFleetPars$dFfinal
        }
      }

      if(snowfall::sfIsRunning() & parallel){
        out2<-snowfall::sfLapply(probQ,getq_multi_MICE,StockPars, FleetPars,
                                 np,nf, nareas, maxage, nyears, N, VF, FretA,
                                 maxF=MOM@maxF, MPA,CatchFrac, bounds=bounds,
                                 tol=1E-6,Rel,SexPars,
                                 plusgroup=plusgroup, optVB=optVB)
      }else{
        out2<-lapply(probQ,getq_multi_MICE,StockPars, FleetPars, np,nf, nareas,
                     maxage, nyears, N, VF, FretA, maxF=MOM@maxF,
                     MPA,CatchFrac, bounds= bounds,tol=1E-6,Rel,SexPars,
                     plusgroup=plusgroup, optVB=optVB)
      }

      qs2<-t(matrix(NIL(out2,"qtot"),nrow=np))
      qout2<-array(NIL(out2,"qfrac"),c(np,nf,nsim))
      qfrac2<-array(NA,c(Nprob,np,nf))
      qind2<-TEG(dim(qfrac2))
      qfrac2[qind2]<-qout2[qind2[,c(2,3,1)]]
      qfrac[probQ,,]<-qfrac2
      qs[probQ,]<-qs2

      probQ <- which(apply(qs > max(LimBound) | qs < min(LimBound),1,sum)>0)
      count <- count + 1
      if (length(probQ) == 0) Err <- FALSE

    } # end of while loop
    if (Err) { # still a problem

      tooLow <- length(which(qs > max(LimBound)))
      tooHigh <- length(which(qs < min(LimBound)))
      prErr <- length(probQ)/nsim
      if (prErr > fracD & length(probQ) >= 1) {
        if (length(tooLow) > 0)
          message(tooLow, " sims can't get down to the lower bound on depletion")
        if (length(tooHigh) > 0)
          message(tooHigh, " sims can't get to the upper bound on depletion")
        if(!silent)
          message("More than ", fracD*100, "% of simulations can't get to the ",
                  "specified level of depletion with these Operating Model parameters")
        stop("Change OM@seed and try again for a complete new sample, modify the ",
             "input parameters, or increase ntrials")
      } else {
        if (length(tooLow) > 0)
          message(tooLow, " sims can't get down to the lower bound on depletion")
        if (length(tooHigh) > 0)
          message(tooHigh, " sims can't get to the upper bound on depletion")
        if(!silent)
          message("More than ", 100-fracD*100, "% simulations can get to the ",
                  "sampled depletion.\nContinuing")
      }
    }
    for(p in 1:np)for(f in 1:nf) FleetPars[[p]][[f]]$qs<-qs[,p]*qfrac[,p,f]
  } # end of re-optimization conditional

  if(!silent)
    message("Calculating historical stock and fishing dynamics")

  # ---- Run Historical Simulations ----
  histYrs <- sapply(1:nsim, HistMICE, StockPars=StockPars,
                    FleetPars=FleetPars,np=np,nf=nf,nareas=nareas,
                    maxage=maxage,nyears=nyears,N=N,VF=VF,FretA=FretA,
                    maxF=MOM@maxF,MPA=MPA,Rel=Rel,SexPars=SexPars,qs=qs,
                    qfrac=qfrac,
                    plusgroup=plusgroup)

  N <- aperm(array(as.numeric(unlist(histYrs[1,], use.names=FALSE)),
                   dim=c(np,n_age, nyears, nareas, nsim)), c(5,1,2,3,4))

  Biomass <- aperm(array(as.numeric(unlist(histYrs[2,], use.names=FALSE)),
                         dim=c(np ,n_age, nyears, nareas, nsim)), c(5,1,2,3,4))

  SSN <- aperm(array(as.numeric(unlist(histYrs[3,], use.names=FALSE)),
                     dim=c(np,n_age, nyears, nareas, nsim)), c(5,1,2,3,4))
  SSB <- aperm(array(as.numeric(unlist(histYrs[4,], use.names=FALSE)),
                     dim=c(np,n_age, nyears, nareas, nsim)), c(5,1,2,3,4))
  VBiomass <- aperm(array(as.numeric(unlist(histYrs[5,], use.names=FALSE)),
                          dim=c(np, n_age, nyears, nareas, nsim)), c(5,1,2,3,4))
  FM <- aperm(array(as.numeric(unlist(histYrs[6,], use.names=FALSE)),
                    dim=c(np,nf,n_age, nyears, nareas, nsim)), c(6,1,2,3,4,5))
  FMret <- aperm(array(as.numeric(unlist(histYrs[7,], use.names=FALSE)),
                       dim=c(np,nf,n_age, nyears, nareas, nsim)), c(6,1,2,3,4,5))

  Linfarray <- aperm(array(as.numeric(unlist(histYrs[8,], use.names=FALSE)),
                           dim=c(np, nyears, nsim)), c(3,1,2))

  Karray <- aperm(array(as.numeric(unlist(histYrs[9,], use.names=FALSE)),
                        dim=c(np, nyears, nsim)), c(3,1,2))

  t0array <- aperm(array(as.numeric(unlist(histYrs[10,], use.names=FALSE)),
                         dim=c(np, nyears, nsim)), c(3,1,2))

  Len_age <- aperm(array(as.numeric(unlist(histYrs[11,], use.names=FALSE)),
                         dim=c(np, n_age, nyears, nsim)), c(4,1,2,3))

  Wt_age <- aperm(array(as.numeric(unlist(histYrs[12,], use.names=FALSE)),
                        dim=c(np, n_age, nyears, nsim)), c(4,1,2,3))

  VBF <- aperm(array(as.numeric(unlist(histYrs[17,], use.names=FALSE)),
                     dim=c(np,nf,n_age, nyears, nareas, nsim)), c(6,1,2,3,4,5))
  Z <- aperm(array(as.numeric(unlist(histYrs[18,], use.names=FALSE)),
                   dim=c(np,n_age, nyears, nareas, nsim)), c(5,1,2,3,4))
  FMt<-aperm(array(as.numeric(unlist(histYrs[19,], use.names=FALSE)),
                   dim=c(np,n_age, nyears, nareas, nsim)), c(5,1,2,3,4))
  M_ageArray <- aperm(array(as.numeric(unlist(histYrs[20,], use.names=FALSE)),
                            dim=c(np, n_age, nyears, nsim)), c(4,1,2,3))

  # update StockPars
  for (p in 1:np) {
    StockPars[[p]]$Linfarray[,1:nyears] <- Linfarray[,p, ]
    StockPars[[p]]$Karray[,1:nyears] <- Karray[,p, ]
    StockPars[[p]]$t0array[,1:nyears] <- t0array[,p, ]
    StockPars[[p]]$Len_age[,,1:nyears] <- Len_age[,p,, ]
    StockPars[[p]]$Wt_age[,,1:nyears] <- Wt_age[,p,, ]
    StockPars[[p]]$M_ageArray[,,1:nyears] <- M_ageArray[,p,, ]
  }

  # TODO - selectivity-at-age should update if growth changes
  # Depletion check
  SSB0_specified <- array(NIL(StockPars,'SSB0'),c(nsim,np))
  D_specified <- array(NIL(StockPars,'D'),c(nsim,np))
  if (optVB) {
    VB0_specified <- array(NIL(StockPars,'VB0'),c(nsim,np))
    Depletion <- apply(VBiomass[,,,nyears,,drop=F],1:2,sum)/ VB0_specified
  } else {
    Depletion <- apply(SSB[,,,nyears,,drop=F],1:2,sum)/ SSB0_specified
  }

  if(length(SexPars)>0){ # need to copy over depletion for a sex-specific model
    sexmatches<-sapply(1:nrow(SexPars$SSBfrom), function(x,mat)
      paste(mat[x,],collapse="_"), mat=SexPars$SSBfrom)
    parcopy<-match(sexmatches,sexmatches)
    StockPars_t<-StockPars # need to store a temporary object for copying to/from
    Depletion[,1:np]<-Depletion[,parcopy]
  }

  for(p in 1:np)
    StockPars[[p]]$Depletion<-Depletion[,p]  # add actual Depletion to StockPars

  if(!is.null(control$checks)){
    Cpred<-array(NA,c(nsim,np,nf,maxage,nareas))
    Cind<-as.matrix(expand.grid(1:nsim,1:np,1:nf,1:maxage,nyears,1:nareas))
    Cpred[Cind[,c(1:4,6)]]<-Biomass[Cind[,c(1,2,4,5,6)]]*(1-exp(-FM[Cind]))
    Cpred<-apply(Cpred,1:3,sum,na.rm=T)

    for(p in 1:np){
      Cp<-array(Cpred[,p,],c(nsim,nf))/apply(Cpred[,p,],1,sum)

      if(prod(round(CatchFrac[[p]],4)/round(Cp,4))!=1){
        print(Snames[p])
        print(cbind(CatchFrac[[p]],rep(NaN,nsim),round(Cp,4)))
        warning("Possible problem in catch fraction calculations")
      }

    }
  }
  if (!is.null(control$checks)) {
    if (prod(round(Depletion,2)/ round(D_specified,2)) != 1) {
      print(cbind(round(Depletion,4),rep(NaN,nsim), round(D_specified,4)))
      warning("Possible problem in depletion calculations")
    }
  }

  # --- Calculate MSY statistics for each year ----
  # ignores spatial closures
  # assumes all vulnerable fish are caught - ie no discarding
  if(!silent) message("Calculating MSY reference points for each year")
  # average life-history parameters over ageM years
  for(p in 1:np){
    MSY_y <- array(0, dim=c(nsim, nyears+proyears)) # store MSY for each sim and year
    StockPars[[p]]$MSY_y <- MSY_y # store MSY for each sim and year
    StockPars[[p]]$FMSY_y <- MSY_y # store FMSY for each sim, and year
    StockPars[[p]]$SSBMSY_y <- MSY_y # store SSBMSY for each sim, and year
    StockPars[[p]]$BMSY_y <- MSY_y # store BMSY for each sim, and year
    StockPars[[p]]$VBMSY_y <- MSY_y # store VBMSY for each sim, and year

    FMt_future <- aperm(replicate(proyears, FMt[,,,nyears,, drop=FALSE]), c(1,2,3,4,6,5))
    FMt_all <- abind::abind(FMt[,p,,,], FMt_future[,p,,1,,], along=3)

    V <- apply(FMt_all,1:3,sum)
    V[V<=0] <- tiny
    V <- nlz(V,c(1,3),"max")

    for (y in 1:(nyears+proyears)) {
      MSYrefsYr <- sapply(1:nsim, optMSY_eq,
                          StockPars[[p]]$M_ageArray,
                          StockPars[[p]]$Wt_age,
                          StockPars[[p]]$Mat_age,
                          V,
                          StockPars[[p]]$maxage,
                          StockPars[[p]]$R0,
                          StockPars[[p]]$SRrel,
                          StockPars[[p]]$hs,
                          yr.ind=y,
                          plusgroup=plusgroup[p])

      StockPars[[p]]$MSY_y[,y] <- MSYrefsYr[1, ]
      StockPars[[p]]$FMSY_y[,y] <- MSYrefsYr[2,]
      StockPars[[p]]$SSBMSY_y[,y] <- MSYrefsYr[3,]
      StockPars[[p]]$BMSY_y[,y] <- MSYrefsYr[6,]
      StockPars[[p]]$VBMSY_y[,y] <- MSYrefsYr[7,]
    }

    # --- MSY reference points ----
    MSYRefPoints <- sapply(1:nsim, CalcMSYRefs,
                           MSY_y=StockPars[[p]]$MSY_y,
                           FMSY_y= StockPars[[p]]$FMSY_y,
                           SSBMSY_y=StockPars[[p]]$SSBMSY_y,
                           BMSY_y=StockPars[[p]]$BMSY_y,
                           VBMSY_y=StockPars[[p]]$VBMSY_y,
                           ageM=StockPars[[p]]$ageM,
                           nyears=nyears)

    MSY <- MSYRefPoints[1,] %>% unlist() # record the MSY results (Vulnerable)
    FMSY <- MSYRefPoints[2,] %>% unlist()  # instantaneous FMSY (Vulnerable)
    SSBMSY <- MSYRefPoints[3,] %>% unlist()  # Spawning Stock Biomass at MSY
    BMSY <- MSYRefPoints[4,] %>% unlist() # total biomass at MSY
    VBMSY <- MSYRefPoints[5,] %>% unlist() # Biomass at MSY (Vulnerable)
    UMSY <- MSY/VBMSY  # exploitation rate
    FMSY_M <- FMSY/StockPars$M  # ratio of true FMSY to natural mortality rate M
    SSBMSY_SSB0 <- SSBMSY/SSB0 # SSBMSY relative to unfished (SSB)
    BMSY_B0 <- BMSY/B0 # Biomass relative to unfished (B0)
    VBMSY_VB0 <- VBMSY/StockPars[[p]]$VB0 # VBiomass relative to unfished (VB0)

    StockPars[[p]]$MSY <- MSY
    StockPars[[p]]$FMSY <- FMSY
    StockPars[[p]]$SSBMSY <- SSBMSY
    StockPars[[p]]$BMSY <- BMSY
    StockPars[[p]]$VBMSY <- VBMSY
    StockPars[[p]]$UMSY <- UMSY
    StockPars[[p]]$FMSY_M <- FMSY_M
    StockPars[[p]]$SSBMSY_SSB0 <- SSBMSY_SSB0
    StockPars[[p]]$FMSY_M <-  StockPars[[p]]$FMSY/StockPars[[p]]$M
    StockPars[[p]]$BMSY_B0 <- BMSY_B0
    StockPars[[p]]$VBMSY_VB0 <- VBMSY_VB0


    # --- Dynamic Unfished Reference Points ----
    Unfished <- sapply(1:nsim, function(x)
      popdynCPP(nareas, StockPars[[p]]$maxage,
                Ncurr=N[x,p,,1,],
                nyears+proyears,
                M_age=StockPars[[p]]$M_ageArray[x,,],
                Asize_c=StockPars[[p]]$Asize[x,],
                MatAge=StockPars[[p]]$Mat_age[x,,],
                WtAge=StockPars[[p]]$Wt_age[x,,],
                Vuln=FleetPars[[p]][[1]]$V_real[x,,],
                Retc=FleetPars[[p]][[1]]$retA_real[x,,],
                Prec=StockPars[[p]]$Perr_y[x,],
                movc=split.along.dim(StockPars[[p]]$mov[x,,,,],4),
                SRrelc=StockPars[[p]]$SRrel[x],
                Effind=rep(0, nyears+proyears),
                Spat_targc=FleetPars[[p]][[1]]$Spat_targ[x],
                hc=StockPars[[p]]$hs[x],
                R0c=StockPars[[p]]$R0a[x,],
                SSBpRc=StockPars[[p]]$SSBpR[x,],
                aRc=StockPars[[p]]$aR[x,],
                bRc=StockPars[[p]]$bR[x,],
                Qc=0,
                Fapic=0,
                MPA=FleetPars[[p]][[1]]$MPA,
                maxF=maxF,
                control=1,
                SSB0c=StockPars[[p]]$SSB0[x],
                plusgroup=StockPars[[p]]$plusgroup))

    N_unfished <- aperm(array(as.numeric(unlist(Unfished[1,], use.names=FALSE)),
                              dim=c(n_age, nyears+proyears, nareas, nsim)), c(4,1,2,3))

    Biomass_unfished <- aperm(array(as.numeric(unlist(Unfished[2,], use.names=FALSE)),
                                    dim=c(n_age, nyears+proyears, nareas, nsim)), c(4,1,2,3))

    SSN_unfished <- aperm(array(as.numeric(unlist(Unfished[3,], use.names=FALSE)),
                                dim=c(n_age, nyears+proyears, nareas, nsim)), c(4,1,2,3))

    SSB_unfished <- aperm(array(as.numeric(unlist(Unfished[4,], use.names=FALSE)),
                                dim=c(n_age, nyears+proyears, nareas, nsim)), c(4,1,2,3))

    VBiomass_unfished <- aperm(array(as.numeric(unlist(Unfished[5,], use.names=FALSE)),
                                     dim=c(n_age, nyears+proyears, nareas, nsim)), c(4,1,2,3))

    # ---- Calculate Mean Generation Time ----
    MarrayArea <- replicate(nareas, StockPars[[p]]$M_ageArray[,,1:nyears])
    Mnow<-apply(MarrayArea[,,nyears,]*N[,p,,nyears,],1:2,sum)/apply(N[,p,,nyears,],1:2,sum)
    MGTsurv<-t(exp(-apply(Mnow,1,cumsum)))
    StockPars[[p]]$MGT<-apply(Agearray*(StockPars[[p]]$Mat_age[,,nyears]*MGTsurv),1,sum)/apply(StockPars[[p]]$Mat_age[,,nyears]*MGTsurv,1,sum)

    # ---- Calculate Reference Yield ----
    # if(!silent) message("Calculating reference yield - best fixed F strategy")
    ## TODO - add RefY calcs
    StockPars[[p]]$RefY <-StockPars[[p]]$MSY

    # ---- Store Reference Points ----
    StockPars[[p]]$ReferencePoints <- list(
      ByYear=list(
        MSY=StockPars[[p]]$MSY_y,
        FMSY=StockPars[[p]]$FMSY_y,
        SSBMSY=StockPars[[p]]$SSBMSY_y,
        BMSY=StockPars[[p]]$BMSY_y,
        VBMSY=StockPars[[p]]$VBMSY_y
      ),
      Dynamic_Unfished=list(
        N0=apply(N_unfished, c(1,3), sum),
        B0=apply(Biomass_unfished, c(1,3), sum),
        SN0=apply(SSN_unfished, c(1,3), sum),
        SSB0=apply(SSB_unfished, c(1,3), sum),
        VB0=apply(VBiomass_unfished, c(1,3), sum),
        Rec=apply(N_unfished[,1,,], c(1,2), sum)
      ),
      ReferencePoints=data.frame(
        N0=StockPars[[p]]$N0,
        B0=StockPars[[p]]$B0,
        SSB0=StockPars[[p]]$SSB0,
        SSN0=StockPars[[p]]$SSN0,
        VB0=StockPars[[p]]$VB0,
        MSY=StockPars[[p]]$MSY,
        FMSY=StockPars[[p]]$FMSY,
        SSBMSY=StockPars[[p]]$SSBMSY,
        BMSY=StockPars[[p]]$BMSY,
        VBMSY=StockPars[[p]]$VBMSY,
        UMSY=StockPars[[p]]$UMSY,
        FMSY_M=StockPars[[p]]$FMSY_M,
        SSBMSY_SSB0=StockPars[[p]]$SSBMSY_SSB0,
        BMSY_B0=StockPars[[p]]$BMSY_B0,
        VBMSY_VB0=StockPars[[p]]$VBMSY_VB0,
        RefY=StockPars[[p]]$RefY,
        MGT=StockPars[[p]]$MGT
      )
    )
  }

  # --- Calculate Historical Catch ----
  # Calculate catch-at-age
  # empirical weight-at-age for the catch
  Wt_age_C <- array(NA,c(nsim,np,nf,n_age,nyears,nareas))
  for (p in 1:np) {
    for (f in 1:nf){
      Wt_age_C[,p,f,,,] <- replicate(nareas, FleetPars[[p]][[f]]$Wt_age_C[,,1:nyears])
    }
  }
  Ctemp <- array(NA,c(nsim,np,nf,n_age,nyears,nareas))

  CNind <- TEG(dim(Ctemp))
  Nind<-CNind[,c(1,2,4,5,6)]  # sim, stock, n_age, nyears, nareas

  Biomass_C <- array(0, dim=dim(FMret))
  Biomass_C[CNind] <- N[Nind] * Wt_age_C[CNind]

  Ctemp[CNind] <- Biomass_C[CNind]*(1-exp(-Z[Nind]))*(FM[CNind]/Z[Nind])
  CB <- Ctemp

  # Calculate retained-at-age
  Ctemp[CNind] <- N[Nind] * (1-exp(-Z[Nind])) * (FMret[CNind]/Z[Nind])
  Cret <- Ctemp # apply(Ctemp,1:5,sum)
  Cret[is.na(Cret)] <- 0

  Ctemp[CNind] <- Biomass_C[CNind] * (1-exp(-Z[Nind])) * (FMret[CNind]/Z[Nind])
  CBret <- Ctemp

  # Add to FleetPars
  for (p in 1:np) {
    for (f in 1:nf){
      FleetPars[[p]][[f]]$CBret <- CBret[,p,f,,,]
      FleetPars[[p]][[f]]$CB <- CB[,p,f,,,]
    }
  }

  # --- Sampling by area ----
  valNames <- c("Catch", 'BInd', 'SBInd', 'VInd', 'RecInd',
                'CAA', 'CAL')
  Sample_Area_array <- array(1, dim=c(nsim, nyears+proyears, nareas))
  Sample_Area <- rep(list(Sample_Area_array), length(valNames))
  names(Sample_Area) <- valNames

  # Not currently working
  # if (!is.null(OM@cpars$Sample_Area)) {
  #   Sample_Area_in <- OM@cpars$Sample_Area
  #   inval <- names(Sample_Area_in)[!names(Sample_Area_in) %in% valNames]
  #   if (length(inval)>0)
  #     stop("Invalid names in OM@cpars$Sample_Area.\nValid names are:\n", paste(valNames, collapse="\n"))
  #
  #   for (nm in names(Sample_Area_in)) {
  #     dd <- dim(Sample_Area_in[[nm]])
  #     if (length(dd)!=4) { # Sample_area from Hist
  #       Sample_Area[[nm]] <- Sample_Area_in[[nm]]
  #       if (any(dim(Sample_Area_in[[nm]]) != c(nsim, nyears+proyears, nareas)))
  #         stop("OM@cpars$Sample_Area$", nm, " must be dimensions: nsim, nareas, nyears+proyears", call. = FALSE)
  #     }
  #   }
  # }

  nms <- c("Catch", "BInd", "SBInd", "VInd", "CAA", "CAL")
  for (nm in nms) {
    dd <- dim(Sample_Area[[nm]])
    if (length(dd)!=4) { # Sample_area from Hist
      temp <- replicate(n_age, Sample_Area[[nm]])
      Sample_Area[[nm]] <- aperm(temp, c(1,4,2,3))
    }
  }


  # --- Populate Data object with Historical Data ----
  CurrentYr <- MOM@Fleets[[1]][[1]]@CurrentYr
  DataList <- new('list')
  for (p in 1:np) {
    DataList[[p]] <- vector('list', nf)
    message('Generating historical data for ', Snames[p])
    for (f in 1:nf) {
      ObsPars[[p]][[f]]$Sample_Area <- Sample_Area # add to Obs Pars
      Data <- makeData(Biomass=Biomass[,p,,,],
                       CBret=CBret[,p,f,,,],
                       Cret=Cret[,p,f,,,],
                       N=N[,p,,,],
                       SSB=SSB[,p,,,],
                       VBiomass=VBiomass[,p,,,],
                       StockPars=StockPars[[p]],
                       FleetPars=FleetPars[[p]][[f]],
                       ObsPars=ObsPars[[p]][[f]],
                       ImpPars=ImpPars[[p]][[f]],
                       RefPoints=StockPars[[p]]$ReferencePoints$ReferencePoints,
                       SampCpars=SampCpars[[p]][[f]],
                       StockPars[[p]]$initD,
                       Sample_Area,
                       Name=MOM@Name,
                       nyears,
                       proyears,
                       nsim,
                       nareas,
                       MOM@reps,
                       CurrentYr,
                       silent=TRUE,
                       control)

      # ---- Add Stock & Fleet Dynamics to Data ----
      Data@Misc$StockPars <- StockPars[[p]]
      Data@Misc$FleetPars <- FleetPars[[p]][[f]]
      Data@Misc$ReferencePoints <- StockPars[[p]]$ReferencePoints

      DataList[[p]][[f]] <- Data

    }
  }
  

  # ---- Condition Simulated Data on input Data object (if it exists) & calculate error stats ----
  for (p in 1:np) {
    for (f in 1:nf) {
      if (class(SampCpars[[p]][[f]]$Data)=="Data") {
        StockPars2 <- StockPars[[p]]
        StockPars2$Biomass <- Biomass[,p,,,]
        StockPars2$SSB <- SSB[,p,,,]
        StockPars2$VBiomass <- VBiomass[,p,,,]
        StockPars2$N <- N[,p,,,]
        StockPars2$CBret <- CBret[,p,f,,,]

        # real data has been passed in cpars
        updatedData <- AddRealData(SimData= DataList[[p]][[f]],
                                   RealData=SampCpars[[p]][[f]]$Data,
                                   ObsPars=ObsPars[[p]][[f]],
                                   StockPars=StockPars2,
                                   FleetPars=FleetPars[[p]][[f]],
                                   nsim,
                                   nyears,
                                   proyears,
                                   SampCpars=SampCpars[[p]][[f]],
                                   msg=!silent,
                                   control,
                                   Sample_Area)
        DataList[[p]][[f]] <- updatedData$Data
        ObsPars[[p]][[f]] <- updatedData$ObsPars

      }
    }
  }

  multiHist <- vector('list', np)

  for (p in 1:np) {
    multiHist[[p]] <- vector('list', nf)
    for (f in 1:nf) {
      Hist <- new("Hist")
      Data@Misc <- list()
      Hist@Data <-  DataList[[p]][[f]]
      Hist@Data@Obs <- data.frame() # remove

      ind <- which(lapply(ObsPars[[p]][[f]], length) == nsim)
      obs <- data.frame(ObsPars[[p]][[f]][ind])
      ind <- which(lapply(ImpPars[[p]][[f]], length) == nsim)
      imp <- data.frame(ImpPars[[p]][[f]][ind])
      OMPars <- DataList[[p]][[f]]@OM
      OMPars <- data.frame(OMPars, obs, imp)
      Hist@OMPars <- OMPars
      Hist@AtAge <- list(Length=StockPars[[p]]$Len_age,
                         Weight=StockPars[[p]]$Wt_age,
                         Select=FleetPars[[p]][[f]]$V_real,
                         Retention=FleetPars[[p]][[f]]$retA_real,
                         Maturity=StockPars[[p]]$Mat_age,
                         N.Mortality=StockPars[[p]]$M_ageArray,
                         Z.Mortality=Z[,p,,,],
                         F.Mortality=FM[,p,f,,,],
                         Fret.Mortality=FMret[,p,f,,,],
                         Number=N[,p,,,],
                         Biomass=Biomass[,p,,,],
                         VBiomass=VBiomass[,p,,,],
                         SBiomass=SSB[,p,,,],
                         Removals=CB[,p,f,,,],
                         Landings=CBret[,p,f,,,],
                         Discards=CB[,p,f,,,]-CBret[,p,f,,,]
      )

      Hist@TSdata <- list(
        Number=apply(N[,p,,,],c(1,3,4), sum),
        Biomass=apply(Biomass[,p,,,],c(1,3,4), sum),
        VBiomass=apply(VBiomass[,p,,,],c(1,3,4), sum),
        SBiomass=apply(SSB[,p,,,],c(1,3,4), sum),
        Removals=apply(CB[,p,f,,,], c(1,3,4), sum),
        Landings=apply(CBret[,p,f,,,],c(1,3,4), sum),
        Discards=apply(CB[,p,f,,,]-CBret[,p,f,,,],c(1,3,4), sum),
        Find=FleetPars[[p]][[f]]$Find,
        RecDev=StockPars[[p]]$Perr_y,
        Unfished_Equilibrium=Unfished_Equilibrium[[p]]
      )

      Hist@Ref <- StockPars[[p]]$ReferencePoints

      Hist@SampPars <- list(
        Stock=StockPars[[p]],
        Fleet=FleetPars[[p]][[f]],
        Obs=ObsPars[[p]][[f]],
        Imp=ImpPars[[p]][[f]]
      )

      temp <- MOM@cpars$Data
      MOM@cpars <- list()
      MOM@cpars$control <- control
      MOM@cpars$Data <- temp

      Hist@Misc <- list(
        MOM=MOM
      )

      multiHist[[p]][[f]] <- Hist
    }
  }

  class(multiHist) <- c('list', 'multiHist')

  attr(multiHist, "version") <- packageVersion("MSEtool")
  attr(multiHist, "date") <- date()
  attr(multiHist, "R.version") <- R.version

  multiHist

}



#' @describeIn multiMSE Run Forward Projections for a `MOM` object
#' @param multiHist An Historical Simulation object (class `multiHist`)
#' @export
ProjectMOM <- function (multiHist=NULL, MPs=NA, parallel=FALSE, silent=FALSE,
                        checkMPs=TRUE) {

  # ---- Setup ----
  if (! 'multiHist' %in% class(multiHist))
    stop('Must provide an object of class `multiHist`')

  # Unpack historical simulation data
  MOM <- multiHist[[1]][[1]]@Misc$MOM
  set.seed(MOM@seed) # set seed for reproducibility
  nsim <- MOM@nsim # number of simulations
  nyears <- MOM@Fleets[[1]][[1]]@nyears # number of historical years
  proyears <- MOM@proyears # number of projection years
  interval <- MOM@interval # management interval (annual)
  maxF <- MOM@maxF # maximum apical F
  pstar <- MOM@pstar # Percentile of the sample of the TAC for each MP
  reps <- MOM@reps # Number of samples of the management recommendation for each MP

  allyears <- nyears+proyears
  Stocks <- MOM@Stocks
  Fleets <- MOM@Fleets
  Obs <- MOM@Obs
  Imps <- MOM@Imps
  Rel <- MOM@Rel
  SexPars <- MOM@SexPars
  Complexes <- MOM@Complexes
  CatchFrac <- MOM@CatchFrac

  control <- MOM@cpars$control; MOM@cpars$control <- NULL

  np <- length(Stocks)
  nf <- length(Fleets[[1]])

  ncpus <- set_parallel(parallel)

  # ---- Detect MP Specification ----
  MPcond <- "unknown"
  if(np==1&nf==1){
    if (!silent)
      message("runMSE checking: you have specified a single stock and fleet. ",
              "For analysis you should be using runMSE(). Use this only for debugging ",
              "against runMSE.")
    if (class(MPs) !="character") {
      stop('`MPs` must be specified as a vector of character names if there is only',
           ' 1 stock and 1 fleet. `MPs` is currently a ', class(MPs))
    }
    MPcond <- "complex"
    nMP <- length(MPs)
    MPrefs <- array(NA,c(nMP,nf,np))
    MPrefs[] <- unlist(MPs)

    # check class of MPs
    tt <- try(sapply(MPs, get), silent=TRUE)
    if (class(tt) == 'try-error')
      stop('Error in the MPs -', strsplit(tt,':')[[1]][2])
    MP_classes <- sapply(tt, class)
    if (!all(MP_classes == MP_classes[1]))
      stop('All MPs must be same class (`MP`)')
  }

  if(class(MPs) == 'character' & MPcond=='unknown') {
    # check class of MPs
    tt <- try(sapply(MPs, get), silent=TRUE)
    if (class(tt) == 'try-error')
      stop('Error in the MPs -', strsplit(tt,':')[[1]][2])
    MP_classes <- sapply(tt, class)
    if (!all(MP_classes == MP_classes[1]))
      stop('All MPs must be same class (`MP` or `MMP`)')
    MP_class <- unique(MP_classes)

    if(MP_class=="MMP"){
      message("MMP mode: you have specified multi-fleet, multi-stock MPs of ",
              "class MMP. This class of MP accepts all data objects (stocks x fleets) ",
              "to simultaneously make a recommendation specific to each stock and fleet")
      MPcond <- "MMP"
      nMP <- length(MPs)
      MPrefs <- array(NA,c(nMP,nf,np))
      MPrefs[] <- MPs
    }
    if(MP_class=="MP"){
      message("Complex mode: you have specified a vector of MPs rather than a ",
              "list of MPs, one list position for MP type. The same MP will ",
              "be applied to the aggregate data for all stocks and fleets. ",
              "The MP will, for example, be used to set a single TAC for all ",
              "stocks and fleets combined. This will be allocated among fleets ",
              "according to recent catches and among stocks according to ",
              "available, vulnerable biomass")
      MPcond <- "complex"
      MPtemp <- MPs
      nMP <- length(MPs)
      MPrefs <- array(NA,c(nMP,nf,np))
      MPrefs[] <- unlist(MPs)
    }
  }

  if (class(MPs) == 'list' & MPcond=='unknown') {

    if(identical(ldim(MPs), ldim(Fleets))){
      message("Byfleet mode: you have specified an MP for each stock and fleet. ",
              "Only fleet-specific data (e.g. catches and indices) will be used to set ",
              "advice for each fleet for each stock")
      MPcond <- "byfleet"
      nMP <- length(MPs[[1]][[1]])
      MPrefs <- array(NA,c(nMP,nf,np))
      MPrefs[]<-unlist(MPs)
    } else if (ldim(MPs)==ldim(Fleets)[1]){ # not a two-tier list
      message("Bystock mode: you have specified a vector of MPs for each stock, ",
              "but not a vector of MPs for each stock and fleet. The catch data for these",
              " fleets will be combined, a single MP will be used to set a single TAC ",
              "for all fleets combined that will be allocated between the fleets ",
              "according to recent catches")
      MPcond<-"bystock"
      checkN <- unlist(lapply(MPs, length))
      if (!all(checkN == checkN[1]))
        stop('Must have the same number of MPs for each stock')
      nMP<-length(MPs[[1]])
      MPrefs<-array(NA,c(nMP,nf,np))
      for(p in 1:np)MPrefs[,,p]<-MPs[[p]]
    }
  }

  if (MPcond == 'unknown')
    stop('`MPs` is not a vector or list with correct dimensions. See `?multiMSE`')

  if(class(MPs)=="list"){
    allMPs<-unlist(MPs)
  }else{
    allMPs<-MPs
  }
  if (nMP < 1) stop("No valid MPs found", call.=FALSE)

  # ---- Check MPs ----
  if (checkMPs & class(MPs)!='MMP')
    CheckMPs(MPs=allMPs, silent=silent)

  # ---- Set Management Interval for each MP ----
  # TODO - make same structure as MPs argument
  if (length(interval) != nMP) interval <- rep(interval, nMP)[1:nMP]
  if (!all(interval == interval[1])) {
    if (!silent) message("Variable management intervals:")
    df <- data.frame(MP=MPs,interval=interval)
    for (i in 1:nrow(df)) {
      message(df$MP[i], 'has management interval:', df$interval[i])
    }
  }

  # --- Store MSY statistics for each projection year ----
  MSY_y <- FMSY_y <- SSBMSY_y <- BMSY_y <- VBMSY_y <- array(NA, dim=c(nsim,np, nMP, nyears+proyears))
  # MSY stats from historical simulations
  for (p in 1:np) {
    MSY_y[,p,,] <- aperm(replicate(nMP, multiHist[[p]][[1]]@Ref$ByYear$MSY), c(1,3,2))
    FMSY_y[,p,,] <- aperm(replicate(nMP, multiHist[[p]][[1]]@Ref$ByYear$FMSY), c(1,3,2))
    SSBMSY_y[,p,,] <- aperm(replicate(nMP, multiHist[[p]][[1]]@Ref$ByYear$SSBMSY), c(1,3,2))
    BMSY_y[,p,,] <- aperm(replicate(nMP, multiHist[[p]][[1]]@Ref$ByYear$BMSY), c(1,3,2))
    VBMSY_y[,p,,] <- aperm(replicate(nMP, multiHist[[p]][[1]]@Ref$ByYear$VBMSY), c(1,3,2))
  }

  # ---- Set-up arrays and objects for projections ----
  # create a data object for each method
  # (they have identical historical data and branch in projected years)
  MSElist <- list('list')
  for(p in 1:np){
    MSElist[[p]] <- new('list')
    for(f in 1:nf){
      Data <- multiHist[[p]][[f]]@Data # Historical data for this stock and fleet
      MSElist[[p]][[f]] <- list(Data)[rep(1, nMP)]
    }
  }
  # TODO - update names of stored values
  SB_SBMSYa <- array(NA, dim = c(nsim, np, nMP, proyears))  # store the projected SB_SBMSY
  Ba <- array(NA, dim = c(nsim, np, nMP, proyears))  # store the projected Biomass
  SSBa <- array(NA, dim = c(nsim, np, nMP, proyears))  # store the projected SSB
  VBa <- array(NA, dim = c(nsim, np, nMP, proyears))  # store the projected vulnerable biomass

  FMa <- array(NA, dim = c(nsim, np, nf, nMP, proyears))  # store the projected fishing mortality rate
  F_FMSYa <- array(NA, dim = c(nsim, np, nf, nMP, proyears))  # store the projected F_FMSY
  Ca <- array(NA, dim = c(nsim, np, nf, nMP, proyears))  # store the projected removed catch
  CaRet <- array(NA, dim = c(nsim, np, nf, nMP, proyears))  # store the projected retained catch
  TACa <- array(NA, dim = c(nsim, np, nf, nMP, proyears))  # store the projected TAC recommendation
  TAE_out <- array(NA, dim = c(nsim, np, nf, nMP, proyears))  # store the projected TAE recommendation
  Effort <- array(NA, dim = c(nsim, np, nf, nMP, proyears))  # store the Effort

  # ---- Grab Stock, Fleet, Obs and Imp values from Hist ----
  StockPars <- FleetPars <- ObsPars <- ImpPars <- list()
  for(p in 1:np){
    FleetPars[[p]] <- ObsPars[[p]] <- ImpPars[[p]] <- new('list')
    for(f in 1:nf){
      StockPars[[p]] <- multiHist[[p]][[1]]@SampPars$Stock
      FleetPars[[p]][[f]] <- multiHist[[p]][[f]]@SampPars$Fleet
      ObsPars[[p]][[f]] <- multiHist[[p]][[f]]@SampPars$Obs
      ImpPars[[p]][[f]] <- multiHist[[p]][[f]]@SampPars$Imp
    }
  }

  nareas <- StockPars[[1]]$nareas
  maxage <- StockPars[[1]]$maxage
  n_age <- maxage + 1
  plusgroup <- multiHist[[1]][[1]]@SampPars$Stock$plusgroup

  # projection arrays for storing all info (by simulation, stock, age, MP, years, areas)
  N_P_mp <- array(NA, dim = c(nsim, np, n_age, nMP, proyears, nareas))

  # ---- Grab Historical N-at-age etc ----
  N <- array(NA, dim=c(nsim, np, n_age, nyears, nareas))
  Biomass <- SSB <- VBiomass <- N

  FM <- FMret <- array(NA, dim=c(nsim, np, nf, n_age, nyears, nareas))
  VF<- array(NA, dim=c(nsim, np, nf, n_age, nyears+proyears))
  CB <- CB_ret <- FM
  MPA <- array(NA, dim=c(np, nf, nyears+proyears, nareas))

  for (p in 1:np) {
    N[,p,,,] <- multiHist[[p]][[1]]@AtAge$Number
    Biomass[,p,,,] <- multiHist[[p]][[1]]@AtAge$Biomass
    SSB[,p,,,] <- multiHist[[p]][[1]]@AtAge$SBiomass
    VBiomass[,p,,,] <- multiHist[[p]][[1]]@AtAge$VBiomass

    for (f in 1:nf) {
      FM[,p,f,,,] <- multiHist[[p]][[f]]@AtAge$F.Mortality
      FMret[,p,f,,,] <- multiHist[[p]][[f]]@AtAge$Fret.Mortality
      VF[,p,f,,] <- FleetPars[[p]][[f]]$V_real
      MPA[p,f,,] <- FleetPars[[p]][[f]]$MPA
      CB[,p,f,,,] <- multiHist[[p]][[f]]@AtAge$Removals
      CB_ret[,p,f,,,] <- multiHist[[p]][[f]]@AtAge$Landings
    }
  }
  # need to make a copy because R is doing weird things with elements with similar names
  HistFleetPars <- FleetPars
  Snames <- SIL(Stocks,"Name")
  Fnames <- matrix(make.unique(SIL(Fleets,"Name")),nrow=nf)

  # ---- Begin loop over MPs ----
  mm <- 1 # for debugging

  TAC_A <- array(NA,c(nsim,np,nf)) # Temporary store of the TAC
  TAE_A <- array(NA,c(nsim,np,nf)) # Temporary store of the TAE
  MPrecs_A_blank<-list() # Temporary Hierarchical list of MPrec objects
  for(p in 1:np) MPrecs_A_blank[[p]]<-list()
  LastTAE <- histTAE <- Effort_pot <-LastAllocat <-LastCatch <-TACused <-
    array(NA,c(nsim,np,nf))
  LastSpatial <- array(NA,c(nareas,np,nf,nsim))
  # temporary vulnerability for MSY calcs combined over fleets
  V_Pt <- array(NA,c(nsim,nf,n_age,nyears+proyears))

  for (mm in 1:nMP) {
    if(!silent){
      message(" ----- ", mm, "/", nMP, " MPs, Running MSE for: ")  # print a progress report
      for(p in 1:np){
        MPrep<-data.frame(MPrefs[mm,,p])
        row.names(MPrep)<-Fnames[,p]
        names(MPrep)=Snames[p]
        print(MPrep)
      }
      message(" --------------------------------- ")
    }

    checkNA <- array(0,c(np,nf,proyears)) # save number of NAs

    # reset selectivity & retention parameters for projections
    for(p in 1:np){
      for(f in 1:nf){
        # reset selectivity parameters for projections
        FleetPars[[p]][[f]]$L5_P <- HistFleetPars[[p]][[f]]$L5
        FleetPars[[p]][[f]]$LFS_P <- HistFleetPars[[p]][[f]]$LFS
        FleetPars[[p]][[f]]$Vmaxlen_P <- HistFleetPars[[p]][[f]]$Vmaxlen
        # selectivity at length array - projections
        FleetPars[[p]][[f]]$SLarray_P <- HistFleetPars[[p]][[f]]$SLarray_real
        #  selectivity at age array - projections
        FleetPars[[p]][[f]]$V_P <- HistFleetPars[[p]][[f]]$V_real

        # reset retention parameters for projections
        FleetPars[[p]][[f]]$LR5_P <- HistFleetPars[[p]][[f]]$LR5
        FleetPars[[p]][[f]]$LFR_P <- HistFleetPars[[p]][[f]]$LFR
        FleetPars[[p]][[f]]$Rmaxlen_P <- HistFleetPars[[p]][[f]]$Rmaxlen
        # retention at age array - projections
        FleetPars[[p]][[f]]$retA_P <- HistFleetPars[[p]][[f]]$retA_real
        # retention at length array - projections
        FleetPars[[p]][[f]]$retL_P <- HistFleetPars[[p]][[f]]$retL_real
        # Discard ratio for projections
        FleetPars[[p]][[f]]$DR_P <- HistFleetPars[[p]][[f]]$DR

        FleetPars[[p]][[f]]$FM_P <- array(NA,
                                          dim = c(nsim, n_age, proyears, nareas))
        FleetPars[[p]][[f]]$FM_Pret <- array(NA,
                                             dim = c(nsim, n_age, proyears, nareas))
        # stores prospective F before reallocation to new areas
        FleetPars[[p]][[f]]$FM_nospace <- array(NA,
                                                dim = c(nsim, n_age,
                                                        proyears, nareas))
        # last apical F
        FleetPars[[p]][[f]]$FML <- array(NA, dim = c(nsim, nareas))
        FleetPars[[p]][[f]]$Z_P <- array(NA,
                                         dim = c(nsim, n_age, proyears, nareas))
        FleetPars[[p]][[f]]$CB_P <- array(NA,
                                          dim = c(nsim,n_age, proyears, nareas))
        # retained catch
        FleetPars[[p]][[f]]$CB_Pret <- array(NA,
                                             dim = c(nsim,n_age, proyears, nareas))

      }
      # Discard mortality for projections
      StockPars[[p]]$Fdisc_P <- StockPars[[p]]$Fdisc
      StockPars[[p]]$N_P <- array(NA, dim = c(nsim, n_age, proyears, nareas))
      StockPars[[p]]$Biomass_P <- array(NA, dim = c(nsim, n_age, proyears, nareas))
      StockPars[[p]]$VBiomass_P <- array(NA, dim = c(nsim, n_age, proyears, nareas))
      StockPars[[p]]$SSN_P <-array(NA, dim = c(nsim,n_age, proyears, nareas))
      StockPars[[p]]$SSB_P <- array(NA, dim = c(nsim, n_age, proyears, nareas))
    }

    # projection arrays
    N_P <- array(NA, dim = c(nsim, np, n_age, proyears, nareas))
    Biomass_P <- array(NA, dim = c(nsim,np, n_age, proyears, nareas))
    VBiomass_P <- array(NA, dim = c(nsim,np, n_age, proyears, nareas))
    SSN_P <-array(NA, dim = c(nsim,np, n_age, proyears, nareas))
    SSB_P <- array(NA, dim = c(nsim,np, n_age, proyears, nareas))
    FMt_P <- array(NA, dim = c(nsim, np, n_age, proyears, nareas))
    Z_P <- array(NA, dim = c(nsim, np, n_age, proyears, nareas))
    FM_P <- array(NA, dim = c(nsim, np,nf,n_age, proyears, nareas))
    FMret_P <- array(NA, dim = c(nsim,np,nf, n_age, proyears, nareas))
    VBF_P<-array(NA, dim = c(nsim,np,nf, n_age, proyears, nareas))

    # indexes
    SAYRL <- as.matrix(expand.grid(1:nsim, 1:n_age, nyears, 1:nareas))  # Final historical year
    SAYRt <- as.matrix(expand.grid(1:nsim, 1:n_age, 1 + nyears, 1:nareas))  # Trajectory year
    SAYR <- as.matrix(expand.grid(1:nsim, 1:n_age, 1, 1:nareas))
    SYt <- SAYRt[, c(1, 3)]
    SAYt <- SAYRt[, 1:3]
    SR <- SAYR[, c(1, 4)]
    SA1 <- SAYR[, 1:2]
    S1 <- SAYR[, 1]
    SY1 <- SAYR[, c(1, 3)]
    SAY1 <- SAYRt[, 1:3]
    SYA <- as.matrix(expand.grid(1:nsim, 1, 1:n_age))  # Projection year
    SY <- SYA[, 1:2]
    SA <- SYA[, c(1, 3)]
    SAY <- SYA[, c(1, 3, 2)]
    S <- SYA[, 1]

    # -- First projection year ----
    y <- 1
    if(!silent) {
      cat("."); flush.console()
    }

    Perr<-hs<-R0<-SRrel<-K<-Linf<-t0<-M<-array(NA,c(nsim,np))
    aR<-bR<-R0a<-SSBpR<-Asize<-array(NA,c(nsim,np,nareas))
    mov<-array(NA,c(nsim,np,n_age,nareas,nareas,nyears+proyears))
    Spat_targ_y<-array(NA,c(nsim,np,nf))
    M_agecur_y<-Mat_agecur_y<-array(NA,c(nsim,np,n_age))
    a_y<-b_y<-rep(NA,np)
    WatAge <- Len_age <- array(NA,c(nsim, np,n_age))
    SSB0array <- array(NA, c(nsim, np))
    for(p in 1:np){
      Perr[,p]<-StockPars[[p]]$Perr_y[,nyears+n_age]
      hs[,p]<-StockPars[[p]]$hs
      aR[,p,]<-StockPars[[p]]$aR
      bR[,p,]<-StockPars[[p]]$bR
      mov[,p,,,,]<-StockPars[[p]]$mov
      for(f in 1:nf)Spat_targ_y[,p,f]<-FleetPars[[p]][[f]]$Spat_targ
      SRrel[,p]<-StockPars[[p]]$SRrel
      M_agecur_y[,p,]<-StockPars[[p]]$M_ageArray[,,nyears]
      Mat_agecur_y[,p,]<-StockPars[[p]]$Mat_age[,,nyears]
      K[,p]<-StockPars[[p]]$Karray[,nyears]
      Linf[,p]<-StockPars[[p]]$Linfarray[,nyears]
      t0[,p]<-StockPars[[p]]$t0array[,nyears]
      M[,p]<-StockPars[[p]]$M
      R0[,p]<-StockPars[[p]]$R0
      R0a[,p,]<-StockPars[[p]]$R0a
      SSBpR[,p,]<-StockPars[[p]]$SSBpR
      a_y[p]<-StockPars[[p]]$a
      b_y[p]<-StockPars[[p]]$b
      Asize[,p,]<-StockPars[[p]]$Asize
      Len_age[,p,]<-StockPars[[p]]$Len_age[,,nyears]
      WatAge[,p,]<-StockPars[[p]]$Wt_age[,,nyears]
      SSB0array[,np] <-StockPars[[p]]$SSB0
    }

    # note that Fcur is apical F but, in popdynOneMICE it is DIVIDED in future
    # years between the two areas depending on vulnerable biomass.
    # So to get Fcur you need to sum over areas (a bit weird)
    NextYrN <- sapply(1:nsim, function(x)
      popdynOneMICE(np,nf,nareas, maxage,
                    Ncur=array(N[x,,,nyears,],
                               c(np,n_age,nareas)),
                    Vcur=array(VF[x,,,,nyears],c(np,nf,n_age)),
                    FMretx=array(FMret[x,,,,nyears,],c(np,nf,n_age,nareas)),
                    FMx=array(FM[x,,,,nyears,],c(np,nf,n_age,nareas)),
                    PerrYrp=Perr[x,], hsx=hs[x,], aRx=matrix(aR[x,,],nrow=np),
                    bRx=matrix(bR[x,,],nrow=np),
                    movy=array(mov[x,,,,,nyears],c(np,n_age,nareas,nareas)),
                    Spat_targ=array(Spat_targ_y[x,,],c(np,nf)), SRrelx=SRrel[x,],
                    M_agecur=matrix(M_agecur_y[x,,],nrow=np),
                    Mat_agecur=matrix(Mat_agecur_y[x,,],nrow=np),
                    Asizex=matrix(Asize[x,,],ncol=nareas),Kx =K[x,],
                    Linfx=Linf[x,],t0x=t0[x,],Mx=M[x,],
                    R0x=R0[x,],R0ax=matrix(R0a[x,,],nrow=np),
                    SSBpRx=matrix(SSBpR[x,,],nrow=np),ax=a_y,
                    bx=b_y,Rel=Rel,SexPars=SexPars,x=x,
                    plusgroup=plusgroup, SSB0x =SSB0array[x,],
                    Len_age=Len_age[x,,], Wt_age=WatAge[x,,]))

    N_P[,,,1,] <- aperm(array(as.numeric(unlist(NextYrN[1,], use.names=FALSE)),
                              dim=c(np,n_age, nareas, nsim)), c(4,1,2,3))

    Biomass_P[,,,1,] <- aperm(array(as.numeric(unlist(NextYrN[23,],
                                                      use.names=FALSE)),
                                    dim=c(np,n_age, nareas, nsim)), c(4,1,2,3))
    SSN_P[,,,1,] <- aperm(array(as.numeric(unlist(NextYrN[24,],
                                                  use.names=FALSE)),
                                dim=c(np,n_age, nareas, nsim)), c(4,1,2,3))
    SSB_P[,,,1,] <- aperm(array(as.numeric(unlist(NextYrN[25,], use.names=FALSE)),
                                dim=c(np,n_age, nareas, nsim)), c(4,1,2,3))
    VBiomass_P[,,,1,] <- aperm(array(as.numeric(unlist(NextYrN[19,],
                                                       use.names=FALSE)),
                                     dim=c(np,n_age, nareas, nsim)), c(4,1,2,3))
    FML <- apply(array(FM[, ,,, nyears, ],c(nsim,np,nf,n_age,nareas)),
                 c(1, 3), max)

    Len_age <- aperm(array(as.numeric(unlist(NextYrN[14,], use.names=FALSE)),
                           dim=c(np, n_age, nsim)), c(3,1,2))

    Wt_age <- aperm(array(as.numeric(unlist(NextYrN[15,], use.names=FALSE)),
                          dim=c(np, n_age, nsim)), c(3,1,2))

    for(p in 1:np) {
      StockPars[[p]]$N_P<-N_P[,p,,,]
      StockPars[[p]]$Biomass_P<-Biomass_P[,p,,,]
      StockPars[[p]]$SSN_P<-SSN_P[,p,,,]
      StockPars[[p]]$SSB_P<-SSB_P[,p,,,]
      StockPars[[p]]$VBiomass_P<-VBiomass_P[,p,,,]

      StockPars[[p]]$Len_age[,,nyears+y] <- Len_age[,p,]
      StockPars[[p]]$Wt_age[,,nyears+y] <- Wt_age[,p,]
    }

    # ---- Update true abundance ----
    # - used for FMSY ref methods so that FMSY is applied to current abundance
    for (p in 1:np) {
      for (f in 1:nf) {
        M_array <- array(0.5*StockPars[[p]]$M_ageArray[,,nyears+y],
                         dim=c(nsim, n_age, nareas))
        Atemp <- apply(StockPars[[p]]$VBiomass_P[, , y, ] *
                         exp(-M_array), 1, sum) # Abundance (mid-year before fishing)

        MSElist[[p]][[f]][[mm]]@OM$A <- Atemp
      } # end fleets
    } # end stocks

    # --- Apply MP in initial projection year ----
    # - Combined MP -
    if(MPcond=="MMP"){
      # returns a hierarchical list object stock then fleet of Data objects
      # DataList<-getDataList(MSElist,mm)
      DataList<-getDataList(MSElist,mm)
      # returns a hierarchical list object stock then fleet then slot type of Rec
      MPRecs_A <- applyMMP(DataList, MP = MPs[mm], reps = 1, silent=TRUE)
      Data_p_A <- MPrecs_A_blank
      for(p in 1:np)for(f in 1:nf){
        Data_p_A[[p]][[f]]<-MSElist[[p]][[f]][[mm]]
        Data_p_A[[p]][[f]]@TAC<-MPRecs_A[[p]][[f]]$TAC # record TAC rec in Data
      }

    }else if(MPcond=="complex"){
      # A temporary blank hierarchical list object stock by fleet
      MPRecs_A <- Data_p_A <- MPrecs_A_blank
      # need this for aggregating data and distributing TACs over stocks
      realVB<-apply(VBiomass[,,,1:nyears,, drop=FALSE],c(1,2,4),sum,na.rm=T)

      curdat<-multiDataS(MSElist,StockPars,np,mm,nf,realVB)
      runMP <- applyMP(curdat, MPs = MPs[mm], reps = 1, silent=TRUE)  # Apply MP
      
      Stock_Alloc<-realVB[,,nyears, drop=FALSE]/apply(realVB[,,nyears, drop=FALSE],1,sum)

      for(p in 1:np)  for(f in 1:nf){
        MPRecs_A[[p]][[f]]<-runMP[[1]][[1]]
        MPRecs_A[[p]][[f]]$TAC<-runMP[[1]][[1]]$TAC*MOM@Allocation[[p]][,f]*
          Stock_Alloc[,p,1]
        MPRecs_A[[p]][[f]]$Effort<-runMP[[1]][[1]]$Effort*MOM@Efactor[[p]][,f]

        if(length(MPRecs_A[[p]][[f]]$Effort)>0)
          if(is.na(MPRecs_A[[p]][[f]]$Effort[1,1]))
            MPRecs_A[[p]][[f]]$Effort <- matrix(NA,
                                                nrow=0,
                                                ncol=ncol(MPRecs_A[[p]][[f]]$Effort))
        if(length(MPRecs_A[[p]][[f]]$TAC)>0)
          if(is.na(MPRecs_A[[p]][[f]]$TAC[1,1]))
            MPRecs_A[[p]][[f]]$TAC <- matrix(NA,
                                             nrow=0,
                                             ncol=ncol(MPRecs_A[[p]][[f]]$TAC))
        if(is.na(MPRecs_A[[p]][[f]]$Spatial[1,1]))
          MPRecs_A[[p]][[f]]$Spatial <- matrix(NA,
                                               nrow=0,
                                               ncol=ncol(MPRecs_A[[p]][[f]]$TAC))

        Data_p_A[[p]][[f]]<-runMP[[2]]
        Data_p_A[[p]][[f]]@TAC<-MPRecs_A[[p]][[f]]$TAC
      }
    }else{
      # A temporary blank hierarchical list object stock by fleet
      MPRecs_A <- Data_p_A <- MPrecs_A_blank
      for(p in 1:np){
        if(MPcond=="bystock"){
          if(nf>1){
            curdat<-multiData(MSElist,StockPars,p,mm,nf)
          }else{
            curdat<-MSElist[[p]][[f]][[mm]]
          }
          runMP <- applyMP(curdat, MPs = MPs[[p]][mm], reps = 1,
                                   silent=TRUE)  # Apply MP

          # Do allocation calcs
          TAC_A[,p,] <- array(as.vector(unlist(runMP[[1]][[1]]$TAC))*
                                MOM@Allocation[[p]],c(nsim,nf))
          TAE_A[,p,] <- array(as.vector(unlist(runMP[[1]][[1]]$Effort))*
                                MOM@Efactor[[p]],c(nsim,nf))

          for(f in 1:nf){
            MPRecs_A[[p]][[f]]<-runMP[[1]][[1]]
            MPRecs_A[[p]][[f]]$TAC<-matrix(TAC_A[,p,f],nrow=1) # copy allocated TAC
            MPRecs_A[[p]][[f]]$Effort<-matrix(TAE_A[,p,f],nrow=1)
            # This next line is to make the NULL effort recommendations of an
            # output control MP compatible with CalcMPdynamics (expects a null matrix)
            if(is.na(MPRecs_A[[p]][[f]]$Effort[1,1]))
              MPRecs_A[[p]][[f]]$Effort <- matrix(NA,
                                                  nrow=0,
                                                  ncol=ncol(MPRecs_A[[p]][[f]]$Effort))
            if(is.na(MPRecs_A[[p]][[f]]$TAC[1,1]))
              MPRecs_A[[p]][[f]]$TAC<-matrix(NA,
                                             nrow=0,
                                             ncol=ncol(MPRecs_A[[p]][[f]]$TAC))
            if(is.na(MPRecs_A[[p]][[f]]$Spatial[1,1]))
              MPRecs_A[[p]][[f]]$Spatial<-matrix(NA,
                                                 nrow=0,
                                                 ncol=ncol(MPRecs_A[[p]][[f]]$TAC))

            Data_p_A[[p]][[f]]<-runMP[[2]]
            Data_p_A[[p]][[f]]@TAC<-MPRecs_A[[p]][[f]]$TAC   # copy allocated tAC
          }
        }else if(MPcond=="byfleet"){
          for(f in 1:nf){
            curdat<-MSElist[[p]][[f]][[mm]]
            runMP <- MSEtool::applyMP(curdat, MPs = MPrefs[mm,f,p], reps = 1, silent=TRUE)  # Apply MP
            MPRecs_A[[p]][[f]]<-runMP[[1]][[1]]
            Data_p_A[[p]][[f]]<-runMP[[2]]
            Data_p_A[[p]][[f]]@TAC <- MPRecs_A[[p]][[f]]$TAC
          }
        }
      } # end of stocks
    }

    MPCalcs_list <- vector('list', np)

    for(p in 1:np) {
      MPCalcs_list[[p]] <- vector('list', nf)
      for(f in 1:nf) {
        TACused[,p,f] <- apply(Data_p_A[[p]][[f]]@TAC, 2, quantile,
                               p = MOM@pstar, na.rm = T)
        checkNA[p,f,y] <- sum(is.na(TACused[,p,f]))
        LastTAE[,p,f] <-  rep(NA, nsim) # no current TAE exists
        histTAE[,p,f] <- rep(NA, nsim) # no existing TAE
        LastSpatial[,p,f,] <- array(MPA[p,f,nyears,], dim=c(nareas, nsim)) #
        # default assumption of reallocation of effort to open areas
        LastAllocat[,p,f] <- rep(1, nsim)
        LastCatch[,p,f] <- apply(CB[,p,f,,nyears,], 1, sum)
        Effort_pot[,p,f] <- rep(NA, nsim) # No bio-economic model

        MPCalcs <- CalcMPDynamics(MPRecs=MPRecs_A[[p]][[f]], y,
                                  nyears, proyears, nsim,
                                  Biomass_P=StockPars[[p]]$Biomass_P,
                                  VBiomass_P=StockPars[[p]]$VBiomass_P,
                                  LastTAE=LastTAE[,p,f],
                                  histTAE=histTAE[,p,f],
                                  LastSpatial=LastSpatial[,p,f,],
                                  LastAllocat=LastAllocat[,p,f],
                                  LastTAC=LastCatch[,p,f],
                                  TACused=TACused[,p,f],
                                  maxF=maxF,
                                  LR5_P=FleetPars[[p]][[f]]$LR5_P,
                                  LFR_P=FleetPars[[p]][[f]]$LFR_P,
                                  Rmaxlen_P=FleetPars[[p]][[f]]$Rmaxlen_P,
                                  retL_P=FleetPars[[p]][[f]]$retL_P,
                                  retA_P=FleetPars[[p]][[f]]$retA_P,
                                  L5_P=FleetPars[[p]][[f]]$L5_P,
                                  LFS_P=FleetPars[[p]][[f]]$LFS_P,
                                  Vmaxlen_P=FleetPars[[p]][[f]]$Vmaxlen_P,
                                  SLarray_P=FleetPars[[p]][[f]]$SLarray_P,
                                  V_P=FleetPars[[p]][[f]]$V_P,
                                  Fdisc_P=StockPars[[p]]$Fdisc_P,
                                  DR_P=FleetPars[[p]][[f]]$DR_P,
                                  FM_P=FleetPars[[p]][[f]]$FM_P,
                                  FM_Pret=FleetPars[[p]][[f]]$FM_Pret,
                                  Z_P=FleetPars[[p]][[f]]$Z_P,
                                  CB_P=FleetPars[[p]][[f]]$CB_P,
                                  CB_Pret=FleetPars[[p]][[f]]$CB_Pret,
                                  Effort_pot=Effort_pot[,p,f],
                                  StockPars=StockPars[[p]],
                                  FleetPars=FleetPars[[p]][[f]],
                                  ImpPars=ImpPars[[p]][[f]], control=control)


        if(length(SexPars)>0) MPCalcs<- MPCalcsNAs(MPCalcs) # Zeros caused by SexPars

        TACa[,p,f, mm, y] <- TACused[,p,f]#MPCalcs$TACrec # recommended TAC
        LastSpatial[,p,f,] <- MPCalcs$Si
        LastAllocat[,p,f] <- MPCalcs$Ai

        LastTAE[,p,f] <- MPCalcs$TAE # TAE set by MP
        TAE_out[,p,f, mm, y] <- MPCalcs$TAE # TAE
        LastCatch[,p,f] <- MPCalcs$TACrec # TAC et by MP

        Effort[,p,f, mm, y] <- rep(MPCalcs$Effort,nsim)[1:nsim]
        FleetPars[[p]][[f]]$CB_P <- MPCalcs$CB_P # removals
        FleetPars[[p]][[f]]$CB_Pret <- MPCalcs$CB_Pret # retained catch
        FleetPars[[p]][[f]]$FM_P <- MPCalcs$FM_P # fishing mortality
        FM_P[,p,f,,,]<- MPCalcs$FM_P

        FleetPars[[p]][[f]]$FM_Pret <- MPCalcs$FM_Pret # retained fishing mortality
        FMret_P[,p,f,,,]<- MPCalcs$FM_Pret
        #FretA[,p,f,,]<- MPCalcs$FM_Pret
        FleetPars[[p]][[f]]$Z_P <- MPCalcs$Z_P # total mortality
        FleetPars[[p]][[f]]$retA_P <- MPCalcs$retA_P # retained-at-age

        FleetPars[[p]][[f]]$retL_P <- MPCalcs$retL_P # retained-at-length
        FleetPars[[p]][[f]]$V_P <- MPCalcs$V_P  # vulnerable-at-age
        VF[,p,f,,]<- MPCalcs$V_P
        FleetPars[[p]][[f]]$SLarray_P <- MPCalcs$SLarray_P # vulnerable-at-length
        FMa[,p,f,mm,y] <- MPCalcs$Ftot # Total fishing mortality (by stock & fleet)

        MPCalcs_list[[p]][[f]] <- MPCalcs

      }
    }


    # the years in which there are updates
    upyrs <- 1 + (0:(floor(proyears/interval[mm]) - 1)) * interval[mm]
    if(!silent) {
      cat(".")
      flush.console()
    }

    # --- Begin projection years ----
    for (y in 2:proyears) {
      if(!silent) {
        cat(".")
        flush.console()
      }

      # -- Calculate MSY stats for this year ----
      # if selectivity has changed

      for (p in 1:np) {
        for (f in 1:nf) {
          SelectChanged <- FALSE
          if (any(
            FleetPars[[p]][[f]]$retA_P[,,nyears+y] - FleetPars[[p]][[f]]$retA_P[,,nyears+y] !=0))  SelectChanged <- TRUE
          if (any(
            FleetPars[[p]][[f]]$V_P[,,nyears+y] - FleetPars[[p]][[f]]$V_real[,,nyears+y] !=0))  SelectChanged <- TRUE


          # recalculate MSY ref points because selectivity has changed
          V_Pt[,f,,]<-FleetPars[[p]][[f]]$V_P*
            apply(CB[,p,f,,nyears,], 1, sum) # Weighted by catch frac
        }
        if (SelectChanged) {
          #summed over fleets and normalized to 1
          V_P<-nlz(apply(V_Pt,c(1,3,4),sum),c(1,3),"max")
          y1 <- nyears + y
          MSYrefsYr <- sapply(1:nsim, optMSY_eq,
                              M_ageArray=StockPars[[p]]$M_ageArray,
                              Wt_age=StockPars[[p]]$Wt_age,
                              Mat_age=StockPars[[p]]$Mat_age,
                              V=V_P,
                              maxage=StockPars[[p]]$maxage,
                              R0=StockPars[[p]]$R0,
                              SRrel=StockPars[[p]]$SRrel,
                              hs=StockPars[[p]]$hs,
                              yr.ind=y1,
                              plusgroup=StockPars[[p]]$plusgroup)
          MSY_y[,p,mm,y1] <- MSYrefsYr[1,]
          FMSY_y[,p,mm,y1] <- MSYrefsYr[2,]
          SSBMSY_y[,p,mm,y1] <- MSYrefsYr[3,]
          BMSY_y[,p,mm,y1] <- MSYrefsYr[6,]
          VBMSY_y[,p,mm,y1] <- MSYrefsYr[7,]
        }
      } # end of annual MSY

      TACa[,,, mm, y] <- TACa[,,, mm, y-1] # TAC same as last year unless changed

      SAYRt <- as.matrix(expand.grid(1:nsim, 1:n_age, y + nyears, 1:nareas))  # Trajectory year
      SAYt <- SAYRt[, 1:3]
      SAYtMP <- cbind(SAYt, mm)
      SYt <- SAYRt[, c(1, 3)]
      SAY1R <- as.matrix(expand.grid(1:nsim, 1:n_age, y - 1, 1:nareas))
      SAYR <- as.matrix(expand.grid(1:nsim, 1:n_age, y, 1:nareas))
      SY <- SAYR[, c(1, 3)]
      SA <- SAYR[, 1:2]
      S1 <- SAYR[, 1]

      SAY <- SAYR[, 1:3]
      S <- SAYR[, 1]
      SR <- SAYR[, c(1, 4)]
      SA2YR <- as.matrix(expand.grid(1:nsim, 2:n_age, y, 1:nareas))
      SA1YR <- as.matrix(expand.grid(1:nsim, 1:(n_age - 1), y -1, 1:nareas))

      for(p in 1:np){
        Perr[,p]<-StockPars[[p]]$Perr_y[,y+nyears+n_age-1]
        M_agecur_y[,p,]<-StockPars[[p]]$M_ageArray[,,nyears+y]
        Mat_agecur_y[,p,]<-StockPars[[p]]$Mat_age[,,nyears+y]
        K[,p]<-StockPars[[p]]$Karray[,nyears+y]
        Linf[,p]<-StockPars[[p]]$Linfarray[,nyears+y]
        t0[,p]<-StockPars[[p]]$t0array[,nyears+y]

        Len_age[,p,]<-StockPars[[p]]$Len_age[,,nyears+y]
        WatAge[,p,]<-StockPars[[p]]$Wt_age[,,nyears+y]
        SSB0array[,np] <-StockPars[[p]]$SSB0
      }

      # note that Fcur is apical F but, in popdynOneMICE it is DIVIDED in future
      # years between the two areas depending on vulnerabile biomass. So to get
      # Fcur you need to sum over areas (a bit weird)
      NextYrN <- sapply(1:nsim, function(x)
        popdynOneMICE(np,nf,nareas, maxage,
                      Ncur=array(N_P[x,,,y-1,],c(np,n_age,nareas)),
                      Vcur=array(VF[x,,,,nyears+y-1],c(np,nf,n_age)),

                      FMretx=array(FMret_P[x,,,,y-1,],c(np,nf,n_age,nareas)),
                      FMx=array(FM_P[x,,,,y-1,],c(np,nf,n_age,nareas)),
                      PerrYrp=Perr[x,], hsx=hs[x,], aRx=matrix(aR[x,,],nrow=np),
                      bRx=matrix(bR[x,,],nrow=np),
                      movy=array(mov[x,,,,,nyears+y],c(np,n_age,nareas,nareas)),
                      Spat_targ=array(Spat_targ_y[x,,],c(np,nf)),
                      SRrelx=SRrel[x,],
                      M_agecur=matrix(M_agecur_y[x,,],nrow=np),
                      Mat_agecur=matrix(Mat_agecur_y[x,,],nrow=np),
                      Asizex=matrix(Asize[x,,],ncol=nareas), Kx =K[x,],
                      Linfx=Linf[x,],t0x=t0[x,],Mx=M[x,],
                      R0x=R0[x,],R0ax=matrix(R0a[x,,],nrow=np),
                      SSBpRx=matrix(SSBpR[x,,],nrow=np),ax=a_y,
                      bx=b_y,Rel=Rel,SexPars=SexPars,x=x,
                      plusgroup=plusgroup, SSB0x=SSB0array[x,],
                      Len_age=Len_age[x,,], Wt_age=WatAge[x,,]))

      N_P[,,,y,]<-aperm(array(as.numeric(unlist(NextYrN[1,], use.names=FALSE)),
                              dim=c(np,n_age, nareas, nsim)), c(4,1,2,3))
      Biomass_P[,,,y,]<-aperm(array(as.numeric(unlist(NextYrN[23,],
                                                      use.names=FALSE)),
                                    dim=c(np,n_age, nareas, nsim)), c(4,1,2,3))
      SSN_P[,,,y,]<-aperm(array(as.numeric(unlist(NextYrN[24,],
                                                  use.names=FALSE)),
                                dim=c(np,n_age, nareas, nsim)), c(4,1,2,3))
      SSB_P[,,,y,]<-aperm(array(as.numeric(unlist(NextYrN[25,],
                                                  use.names=FALSE)),
                                dim=c(np,n_age, nareas, nsim)), c(4,1,2,3))

      VBiomass_P[,,,y,]<-aperm(array(as.numeric(unlist(NextYrN[19,],
                                                       use.names=FALSE)),
                                     dim=c(np,n_age, nareas, nsim)), c(4,1,2,3))
      FML <- apply(array(FM_P[, ,,, y-1, ],c(nsim,np,nf,n_age,nareas)),
                   c(1, 3), max)

      Len_age <- aperm(array(as.numeric(unlist(NextYrN[14,], use.names=FALSE)),
                             dim=c(np, n_age, nsim)), c(3,1,2))

      Wt_age <- aperm(array(as.numeric(unlist(NextYrN[15,], use.names=FALSE)),
                            dim=c(np, n_age, nsim)), c(3,1,2))

      for(p in 1:np){
        StockPars[[p]]$N_P<-N_P[,p,,,]
        StockPars[[p]]$Biomass_P<-Biomass_P[,p,,,]
        StockPars[[p]]$SSN_P<-SSN_P[,p,,,]
        StockPars[[p]]$SSB_P<-SSB_P[,p,,,]
        StockPars[[p]]$VBiomass_P<-VBiomass_P[,p,,,]
        #for(f in 1:nf)FleetPars[[p]][[f]]$FML<-FML[]

        StockPars[[p]]$Len_age[,,nyears+y] <- Len_age[,p,]
        StockPars[[p]]$Wt_age[,,nyears+y] <- Wt_age[,p,]
      }

      # --- An update year ----
      if (y %in% upyrs) {
        # --- Update Data object ----
        for (p in 1:np) {
          for (f in 1:nf) {

            # TODO - remove this
            OM <- suppressMessages(new('OM')) # temporary while MSEtool::makeData requires this
            OM@nyears <- nyears
            OM@hbiascv <- MOM@Obs[[p]][[f]]@hbiascv
            OM@maxF <- MOM@maxF
            OM@CurrentYr <- MSElist[[1]][[1]][[1]]@LHYear
            OM@reps <- MOM@reps
            OM@nsim <- nsim
            OM@BMSY_B0biascv <- MOM@Obs[[p]][[f]]@BMSY_B0biascv
            OM@proyears <- proyears

            MSElist[[p]][[f]][[mm]] <- updateData(Data=MSElist[[p]][[f]][[mm]],
                                                  OM,
                                                  MPCalcs=MPCalcs_list[[p]][[f]],
                                                  Effort=Effort[,p,f,, ,drop=FALSE],
                                                  Biomass=Biomass[,p,,,],
                                                  N=N[,p,,,],
                                                  Biomass_P=Biomass_P[,p,,,],
                                                  CB_Pret=FleetPars[[p]][[f]]$CB_Pret,
                                                  N_P=N_P[,p,,,],
                                                  SSB=SSB[,p,,,],
                                                  SSB_P=SSB_P[,p,,,],
                                                  VBiomass=VBiomass[,p,,,],
                                                  VBiomass_P=VBiomass_P[,p,,,],
                                                  RefPoints=StockPars[[p]]$ReferencePoints$ReferencePoints,

                                                  retA_P=FleetPars[[p]][[f]]$retA_real,
                                                  retL_P=FleetPars[[p]][[f]]$retL_P,
                                                  StockPars=StockPars[[p]],
                                                  FleetPars=FleetPars[[p]][[f]],
                                                  ObsPars=ObsPars[[p]][[f]],
                                                  ImpPars=ImpPars[[p]][[f]],
                                                  V_P=FleetPars[[p]][[f]]$V_P,
                                                  upyrs=upyrs,
                                                  interval=interval,
                                                  y=y,
                                                  mm=mm,
                                                  Misc=MSElist[[p]][[f]][[mm]]@Misc,
                                                  RealData=multiHist[[p]][[f]]@Data,
                                                  Sample_Area=ObsPars[[p]][[f]]$Sample_Area)

            # ---- Update true abundance ----
            M_array <- array(0.5*StockPars[[p]]$M_ageArray[,,nyears+y],
                             dim=c(nsim, n_age, nareas))
            Atemp <- apply(StockPars[[p]]$VBiomass_P[, , y, ] *
                             exp(-M_array), 1, sum) # Abundance (mid-year before fishing)
            MSElist[[p]][[f]][[mm]]@OM$A <- Atemp

          } # end of fleet
        } # end of stock


        if(MPcond=="MMP"){
          # returns a hierarchical list object stock then fleet of Data objects
          DataList <- getDataList(MSElist,mm)
          # # returns a hierarchical list object stock then fleet then slot type of Rec
          MPRecs_A <- applyMMP(DataList, MP = MPs[mm], reps = 1, silent=TRUE)
          Data_p_A <- MPrecs_A_blank
          for(p in 1:np)for(f in 1:nf){
            Data_p_A[[p]][[f]]<-MSElist[[p]][[f]][[mm]]
            Data_p_A[[p]][[f]]@TAC<-MPRecs_A[[p]][[f]]$TAC # record TAC rec in Data
          }
        }else if(MPcond=="complex"){
          # A temporary blank hierarchical list object stock by fleet
          MPRecs_A <- Data_p_A <- MPrecs_A_blank
          # need this for aggregating data and distributing TACs over stocks
          realVB<-abind::abind(apply(VBiomass[,,,1:nyears,, drop=FALSE],c(1,2,4),sum,na.rm=T),
                               apply(VBiomass_P[,,,1:(y-1), , drop=FALSE],c(1,2,4),sum,na.rm=T),
                               along=3)


          curdat<-multiDataS(MSElist,StockPars,np,mm,nf,realVB)
          runMP <- MSEtool::applyMP(curdat, MPs = MPs[mm], reps = 1, silent=TRUE)  # Apply MP

          Stock_Alloc <- realVB[,,nyears, drop=FALSE]/
            apply(realVB[,,nyears, drop=FALSE],1,sum)

          for(p in 1:np)for(f in 1:nf){
            MPRecs_A[[p]][[f]] <- runMP[[1]][[1]]
            MPRecs_A[[p]][[f]]$TAC <- runMP[[1]][[1]]$TAC *
              MOM@Allocation[[p]][,f] * Stock_Alloc[,p,1]
            MPRecs_A[[p]][[f]]$Effort <- runMP[[1]][[1]]$Effort * MOM@Efactor[[p]][,f]

            if(length(MPRecs_A[[p]][[f]]$Effort)>0)
              if(is.na(MPRecs_A[[p]][[f]]$Effort[1,1]))
                MPRecs_A[[p]][[f]]$Effort <- matrix(NA,
                                                    nrow=0,
                                                    ncol=ncol(MPRecs_A[[p]][[f]]$Effort))
            if(length(MPRecs_A[[p]][[f]]$TAC)>0)
              if(is.na(MPRecs_A[[p]][[f]]$TAC[1,1]))
                MPRecs_A[[p]][[f]]$TAC <- matrix(NA,
                                                 nrow=0,
                                                 ncol=ncol(MPRecs_A[[p]][[f]]$TAC))
            if(is.na(MPRecs_A[[p]][[f]]$Spatial[1,1]))
              MPRecs_A[[p]][[f]]$Spatial <- matrix(NA,
                                                   nrow=0,
                                                   ncol=ncol(MPRecs_A[[p]][[f]]$TAC))

            Data_p_A[[p]][[f]]<-runMP[[2]]
            Data_p_A[[p]][[f]]@TAC<-MPRecs_A[[p]][[f]]$TAC

          }
        } else {
          # A temporary blank hierarchical list object stock by fleet
          MPRecs_A <- Data_p_A <- MPrecs_A_blank

          for(p in 1:np){

            if(MPcond=="bystock"){
              if(nf>1){
                curdat<-multiData(MSElist,StockPars,p,mm,nf)
              }else{
                curdat<-MSElist[[p]][[f]][[mm]]
              }
              runMP <- MSEtool::applyMP(curdat, MPs = MPs[[p]][mm],
                                       reps = MOM@reps, silent=TRUE)  # Apply MP
              # Do allocation calcs
              TAC_A[,p,]<-array(as.vector(unlist(runMP[[1]][[1]]$TAC))*MOM@Allocation[[p]],c(nsim,nf))
              TAE_A[,p,]<-array(as.vector(unlist(runMP[[1]][[1]]$Effort))*MOM@Efactor[[p]],c(nsim,nf))

              for(f in 1:nf){
                MPRecs_A[[p]][[f]]<-runMP[[1]][[1]]
                MPRecs_A[[p]][[f]]$TAC<-matrix(TAC_A[,p,f],nrow=1) # Just pass the allocated TAC
                MPRecs_A[[p]][[f]]$Effort<-matrix(TAE_A[,p,f],nrow=1)
                # This next line is to make the NULL effort recommendations of
                # an output control MP compatible with CalcMPdynamics (expects a null matrix)
                if(is.na(MPRecs_A[[p]][[f]]$Effort[1,1]))
                  MPRecs_A[[p]][[f]]$Effort<-matrix(NA,
                                                    nrow=0,
                                                    ncol=ncol(MPRecs_A[[p]][[f]]$Effort))
                if(is.na(MPRecs_A[[p]][[f]]$TAC[1,1]))
                  MPRecs_A[[p]][[f]]$TAC<-matrix(NA,
                                                 nrow=0,
                                                 ncol=ncol(MPRecs_A[[p]][[f]]$TAC))
                if(is.na(MPRecs_A[[p]][[f]]$Spatial[1,1]))
                  MPRecs_A[[p]][[f]]$Spatial<-matrix(NA,
                                                     nrow=0,
                                                     ncol=ncol(MPRecs_A[[p]][[f]]$TAC))
                Data_p_A[[p]][[f]]<-runMP[[2]]
                Data_p_A[[p]][[f]]@TAC<-MPRecs_A[[p]][[f]]$TAC # Copy the allocated TAC
              }
            } else if(MPcond=="byfleet"){
              for(f in 1:nf){
                curdat<-MSElist[[p]][[f]][[mm]]
                runMP <- MSEtool::applyMP(curdat, MPs = MPrefs[mm,f,p],
                                         reps = MOM@reps, silent=TRUE)  # Apply MP
                MPRecs_A[[p]][[f]]<-runMP[[1]][[1]]
                Data_p_A[[p]][[f]]<-runMP[[2]]
                Data_p_A[[p]][[f]]@TAC <- MPRecs_A[[p]][[f]]$TAC
              }
            } # end of MPcond conditional
          } # end of stock loop
        } # end of MMP

        for(p in 1:np){
          for(f in 1:nf){
            # calculate pstar quantile of TAC recommendation dist
            TACused[,p,f] <- apply(Data_p_A[[p]][[f]]@TAC, 2, quantile,
                                   p = MOM@pstar, na.rm = T)
            checkNA[p,f,y] <-checkNA[p,f,y] + sum(is.na(TACused[,p,f]))

            MPCalcs <- CalcMPDynamics(MPRecs=MPRecs_A[[p]][[f]], y,
                                      nyears, proyears, nsim,
                                      Biomass_P=StockPars[[p]]$Biomass_P,
                                      VBiomass_P=StockPars[[p]]$VBiomass_P,
                                      LastTAE=LastTAE[,p,f],
                                      histTAE=histTAE[,p,f],
                                      LastSpatial=LastSpatial[,p,f,],
                                      LastAllocat=LastAllocat[,p,f],
                                      LastTAC=LastCatch[,p,f],
                                      TACused=TACused[,p,f],
                                      maxF=maxF,
                                      LR5_P=FleetPars[[p]][[f]]$LR5_P,
                                      LFR_P=FleetPars[[p]][[f]]$LFR_P,
                                      Rmaxlen_P=FleetPars[[p]][[f]]$Rmaxlen_P,
                                      retL_P=FleetPars[[p]][[f]]$retL_P,
                                      retA_P=FleetPars[[p]][[f]]$retA_P,
                                      L5_P=FleetPars[[p]][[f]]$L5_P,
                                      LFS_P=FleetPars[[p]][[f]]$LFS_P,
                                      Vmaxlen_P=FleetPars[[p]][[f]]$Vmaxlen_P,
                                      SLarray_P=FleetPars[[p]][[f]]$SLarray_P,
                                      V_P=FleetPars[[p]][[f]]$V_P,
                                      Fdisc_P=StockPars[[p]]$Fdisc_P,
                                      DR_P=FleetPars[[p]][[f]]$DR_P,
                                      FM_P=FleetPars[[p]][[f]]$FM_P,
                                      FM_Pret=FleetPars[[p]][[f]]$FM_Pret,
                                      Z_P=FleetPars[[p]][[f]]$Z_P,
                                      CB_P=FleetPars[[p]][[f]]$CB_P,
                                      CB_Pret=FleetPars[[p]][[f]]$CB_Pret,
                                      Effort_pot=Effort_pot[,p,f],
                                      StockPars=StockPars[[p]],
                                      FleetPars=FleetPars[[p]][[f]],
                                      ImpPars=ImpPars[[p]][[f]], control=control)
            # Zeros caused by SexPars
            if(length(SexPars)>0) MPCalcs<-MPCalcsNAs(MPCalcs)

            TACa[,p,f, mm, y] <- MPCalcs$TACrec # recommended TAC
            LastSpatial[,p,f,] <- MPCalcs$Si
            LastAllocat[,p,f] <- MPCalcs$Ai
            LastTAE[,p,f] <- MPCalcs$TAE # adjustment to TAE
            TAE_out[,p,f, mm, y] <- MPCalcs$TAE # TAE
            LastCatch[,p,f] <- MPCalcs$TACrec

            Effort[,p,f, mm, y] <- rep(MPCalcs$Effort,nsim)[1:nsim]
            FleetPars[[p]][[f]]$CB_P <- MPCalcs$CB_P # removals
            FleetPars[[p]][[f]]$CB_Pret <- MPCalcs$CB_Pret # retained catch
            FleetPars[[p]][[f]]$FM_P <- MPCalcs$FM_P # fishing mortality
            FM_P[,p,f,,,]<- MPCalcs$FM_P
            FleetPars[[p]][[f]]$FM_Pret <- MPCalcs$FM_Pret # retained fishing mortality
            FleetPars[[p]][[f]]$Z_P <- MPCalcs$Z_P # total mortality
            FleetPars[[p]][[f]]$retA_P <- MPCalcs$retA_P # retained-at-age

            FleetPars[[p]][[f]]$retL_P <- MPCalcs$retL_P # retained-at-length
            FleetPars[[p]][[f]]$V_P <- MPCalcs$V_P  # vulnerable-at-age
            VF[,p,f,,]<- MPCalcs$V_P
            FleetPars[[p]][[f]]$SLarray_P <- MPCalcs$SLarray_P # vulnerable-at-length
            FMa[,p,f,mm,y] <- MPCalcs$Ftot # Total fishing mortality (by stock & fleet)
          } # end of fleets
        } # end of stocks

        # end of update year
      } else {
        # ---- Not an update year ----
        for(p in 1:np){
          for(f in 1:nf){
            NoMPRecs <- MPRecs_A[[p]][[f]]
            NoMPRecs$Spatial <- NA

            MPCalcs <- CalcMPDynamics(MPRecs=NoMPRecs, y,
                                      nyears, proyears, nsim,
                                      Biomass_P=StockPars[[p]]$Biomass_P,
                                      VBiomass_P=StockPars[[p]]$VBiomass_P,
                                      LastTAE=LastTAE[,p,f],
                                      histTAE=histTAE[,p,f],
                                      LastSpatial=LastSpatial[,p,f,],
                                      LastAllocat=LastAllocat[,p,f],
                                      LastTAC=LastCatch[,p,f],
                                      TACused=TACused[,p,f],
                                      maxF=maxF,
                                      LR5_P=FleetPars[[p]][[f]]$LR5_P,
                                      LFR_P=FleetPars[[p]][[f]]$LFR_P,
                                      Rmaxlen_P=FleetPars[[p]][[f]]$Rmaxlen_P,
                                      retL_P=FleetPars[[p]][[f]]$retL_P,
                                      retA_P=FleetPars[[p]][[f]]$retA_P,
                                      L5_P=FleetPars[[p]][[f]]$L5_P,
                                      LFS_P=FleetPars[[p]][[f]]$LFS_P,
                                      Vmaxlen_P=FleetPars[[p]][[f]]$Vmaxlen_P,
                                      SLarray_P=FleetPars[[p]][[f]]$SLarray_P,
                                      V_P=FleetPars[[p]][[f]]$V_P,
                                      Fdisc_P=StockPars[[p]]$Fdisc_P,
                                      DR_P=FleetPars[[p]][[f]]$DR_P,
                                      FM_P=FleetPars[[p]][[f]]$FM_P,
                                      FM_Pret=FleetPars[[p]][[f]]$FM_Pret,
                                      Z_P=FleetPars[[p]][[f]]$Z_P,
                                      CB_P=FleetPars[[p]][[f]]$CB_P,
                                      CB_Pret=FleetPars[[p]][[f]]$CB_Pret,
                                      Effort_pot=Effort_pot[,p,f],
                                      StockPars=StockPars[[p]],
                                      FleetPars=FleetPars[[p]][[f]],
                                      ImpPars=ImpPars[[p]][[f]], control=control)

            if(length(SexPars)>0)
              MPCalcs <- MPCalcsNAs(MPCalcs) # Zeros caused by SexPars
            TACa[,p,f, mm, y] <- TACused[,p,f] # recommended TAC

            #TACa[,p,f, mm, y] <- MPCalcs$TACrec # recommended TAC
            LastSpatial[,p,f,] <- MPCalcs$Si
            LastAllocat[,p,f] <- MPCalcs$Ai

            LastTAE[,p,f] <- MPCalcs$TAE
            TAE_out[,p,f, mm, y] <- MPCalcs$TAE # recommended TAE
            # LastEi[,p,f] <- MPCalcs$Ei # adjustment to effort
            LastCatch[,p,f] <- MPCalcs$TACrec

            Effort[,p,f, mm, y] <- rep(MPCalcs$Effort,nsim)[1:nsim]
            FleetPars[[p]][[f]]$CB_P <- MPCalcs$CB_P # removals
            FleetPars[[p]][[f]]$CB_Pret <- MPCalcs$CB_Pret # retained catch
            FleetPars[[p]][[f]]$FM_P <- MPCalcs$FM_P # fishing mortality
            FM_P[,p,f,,,]<- MPCalcs$FM_P
            FleetPars[[p]][[f]]$FM_Pret <- MPCalcs$FM_Pret # retained fishing mortality
            FMret_P[,p,f,,,]<- MPCalcs$FM_Pret
            #FretA[,p,f,,]<- MPCalcs$FM_Pret
            FleetPars[[p]][[f]]$Z_P <- MPCalcs$Z_P # total mortality
            FleetPars[[p]][[f]]$retA_P <- MPCalcs$retA_P # retained-at-age

            FleetPars[[p]][[f]]$retL_P <- MPCalcs$retL_P # retained-at-length
            FleetPars[[p]][[f]]$V_P <- MPCalcs$V_P  # vulnerable-at-age
            VF[,p,f,,]<- MPCalcs$V_P
            FleetPars[[p]][[f]]$SLarray_P <- MPCalcs$SLarray_P # vulnerable-at-length

            FMa[,p,f,mm,y] <- MPCalcs$Ftot # Total fishing mortality (by stock & fleet)

          } # end of fleets
        } # end of stocks
      } # end of not update year
    } # end of projection years

    # SSB relative to SSBMSY
    SB_SBMSYa[, ,mm, ] <- apply(SSB_P, c(1,2, 4), sum, na.rm=TRUE)/array(SSBMSY_y[,,mm,],
                                                                       c(nsim,np,proyears))

    # for(p in 1:np) for(f in 1:nf)
    #   FMa[,p,f, mm, ] <- -log(1 - apply(FleetPars[[p]][[f]]$CB_P, c(1, 3), sum,
    #                                     na.rm=TRUE)/apply(VBiomass_P[,p,,,]+
    #                                                         FleetPars[[p]][[f]]$CB_P,
    #                                                       c(1, 3), sum, na.rm=TRUE))
    for(f in 1:nf)
      F_FMSYa[, ,f,mm, ] <- FMa[,,f, mm, ]/FMSY_y[,,mm,(nyears+1):(nyears+proyears)]

    Ba[, ,mm, ] <- apply(Biomass_P, c(1, 2,4), sum, na.rm=TRUE) # biomass
    SSBa[, ,mm, ] <- apply(SSB_P, c(1, 2,4), sum, na.rm=TRUE) # spawning stock biomass
    VBa[, ,mm, ] <- apply(VBiomass_P, c(1, 2, 4), sum, na.rm=TRUE) # vulnerable biomass

    N_P_mp[,,, mm,,] <- N_P

    for(p in 1:np) for(f in 1:nf)
      Ca[, p,f,mm, ] <- apply(FleetPars[[p]][[f]]$CB_P, c(1, 3),
                              sum, na.rm=TRUE) # removed
    for(p in 1:np) for(f in 1:nf)
      CaRet[, p,f,mm, ] <- apply(FleetPars[[p]][[f]]$CB_Pret, c(1, 3),
                                 sum, na.rm=TRUE) # retained catch

    if (!silent) {
      cat("\n")
      if (all(checkNA != nsim) & !all(checkNA == 0)) {
        # print number of NAs
        # message(checkNA)
        # message(checkNA[upyrs])
        ntot <- sum(checkNA[,,upyrs])
        totyrs <- sum(checkNA[,,upyrs] >0)
        nfrac <- round(ntot/(length(upyrs)*nsim),2)*100

        message(totyrs, ' years had TAC = NA for some simulations (',
                nfrac, "% of total simulations)")
        message('Used TAC_y = TAC_y-1')
      }

      if("progress"%in%names(control))
        if(control$progress)
          shiny::incProgress(1/nMP, detail = round(mm*100/nMP))
    }

  } # end of MP loop

  OM<-Obsout<-list()
  for(p in 1:np) {
    OM[[p]]<-Obsout[[p]]<-list()

    for(f in 1:nf) {
      OM[[p]][[f]]<-MSElist[[p]][[f]][[1]]@OM
      Obsout[[p]][[f]]<-MSElist[[p]][[f]][[1]]@Obs
    }
  }
  Misc <- list()
  # Misc$Data <-MSElist
  Misc[['MOM']]<-MOM

  # need to reformat MMP and complex mode to work with MSEout slot
  if(class(MPs)=="character") MPs<-list(MPs)

  # ---- Create MMSE Object ----
  MSEout <- new("MMSE",
                Name = MOM@Name,
                nyears,
                proyears,
                nMPs=nMP,
                MPs=MPs,
                MPcond=MPcond,
                MPrefs=MPrefs,
                nsim,
                nstocks=np,
                nfleets=nf,
                Snames=Snames,
                Fnames=Fnames,
                Stocks=Stocks,
                Fleets=Fleets,
                Obss=Obs,
                Imps=Imps,
                OM=OM,
                Obs=Obsout,
                SB_SBMSY=SB_SBMSYa,
                F_FMSY=F_FMSYa,
                N=apply(N_P_mp, c(1,2,4,5), sum),
                B=Ba,
                SSB=SSBa,
                VB=VBa,
                FM=FMa,
                SPR=list(),
                Catch=CaRet,
                Removals=Ca,
                Effort = Effort,
                TAC=TACa,
                TAE=TAE_out,
                BioEco=list(),
                RefPoint=list(MSY=MSY_y,
                              FMSY=FMSY_y,
                              SSBMSY=SSBMSY_y),
                multiHist=multiHist,
                PPD=MSElist,
                Misc=Misc)
  MSEout
}



#' Run a multi-fleet multi-stock Management Strategy Evaluation
#'
#' Functions for running a multi-stock and/or multi-fleet Management
#' Strategy Evaluation (closed-loop simulation) for a specified operating model
#'
#' @param MOM A multi-fleet multi-stock operating model (class 'MOM')
#' @param MPs A matrix of methods (nstock x nfleet) (character string) of class MP
#' @param Hist Should model stop after historical simulations? Returns a list
#' containing all historical data
#' @param silent Should messages be printed out to the console?
#' @param parallel Logical. Should the MSE be run using parallel processing?
#' @param checkMPs Logical. Check if the specified MPs exist and can be run on `SimulatedData`?
#' @describeIn multiMSE Run a multi-stock, multi-fleet MSE
#' @return  Functions return objects of class `MMSE` and `multiHist`
#' #' \itemize{
#'   \item SimulateMOM - An object of class `multiHist`
#'   \item ProjectMOM - An object of class `MMSE`
#'   \item multiMSE - An object of class `MMSE`
#' }
#' @author T. Carruthers and A. Hordyk
#' @export
multiMSE <- function(MOM=MSEtool::Albacore_TwoFleet,
                     MPs=list(list(c("AvC","DCAC"),c("FMSYref","curE"))),
                     Hist=FALSE,
                     silent=FALSE,
                     parallel=TRUE,
                     checkMPs=TRUE) {

  # ---- Initial Checks and Setup ----
  if (class(MOM) == 'MOM') {
    if (MOM@nsim <=1) stop("MOM@nsim must be > 1", call.=FALSE)

  } else if ('multiHist' %in% class(MOM)) {
    stop("You must specify an operating model of class `MOM`")

    # if (!silent) message("Using `multiHist` object to reproduce historical dynamics")
    #
    # # --- Extract cpars from Hist object ----
    # nstocks <- length(MOM)
    # nfleets <- length(MOM[[1]])
    #
    # cpars <- list()
    # cpars[[1]] <- list()
    #
    # for (s in 1:nstocks) {
    #   for (f in 1:nfleets) {
    #     cpars[[s]][[f]] <- c(MOM[[s]][[f]]@SampPars$Stock,
    #                          MOM[[s]][[f]]@SampPars$Fleet,
    #                          MOM[[s]][[f]]@SampPars$Obs,
    #                          MOM[[s]][[f]]@SampPars$Imp,
    #                          MOM[[s]][[f]]@OMPars,
    #                          MOM[[s]][[f]]@OM@cpars)
    #   }
    # }
    #
    # # --- Populate a new OM object ----
    # newMOM <- MOM[[1]][[1]]@Misc$MOM
    # newMOM@cpars <- cpars
    # MOM <- newMOM

  } else {
    stop("You must specify an operating model of class `MOM`")
  }

  if (checkMPs) {
    allMPs <- unique(unlist(MPs))
    CheckMPs(MPs=allMPs, silent=silent)
  }

  multiHist <- SimulateMOM(MOM, parallel, silent)

  if (Hist) {
    if(!silent) message("Returning historical simulations")
    return(multiHist)
  }

  if(!silent) message("Running forward projections")

  MSEout <- try(ProjectMOM(multiHist=multiHist, MPs, parallel, silent, checkMPs=FALSE), silent=TRUE)

  if (class(MSEout) == 'try-error') {
    message('The following error occured when running the forward projections: ',
            crayon::red(attributes(MSEout)$condition))
    message('Returning the historical simulations (class `multiHist`). To avoid re-running spool up, ',
            'the forward projections can be run with ',
            '`ProjectMOM(multiHist, MPs, ...)`')
    return(multiHist)
  }
  MSEout
}



