\name{est_multi_poly_within}
\alias{est_multi_poly_within}
%- Also NEED an '\alias' for EACH other topic documented here.

\title{Estimate latent class item response theory (LC-IRT) models for dichotomous and polytomous responses under within-item multidimensionality}

\description{The function performs maximum likelihood estimation of the parameters of the IRT models
assuming a discrete distribution for the ability and within-item multidimensionality. Every ability level corresponds to a latent
class of subjects in the reference population. Maximum likelihood estimation is based on Expectation-
Maximization algorithm.}

\usage{est_multi_poly_within(S, yv = rep(1,ns), k1, k2, X = NULL,
                      start = c("deterministic","random","external"), 
                      link = c("global", "local"),disc = FALSE,
                      difl = FALSE, multi1, multi2, piv1 = NULL,
                      piv2 = NULL, Phi = NULL, ga1c = NULL,
                      ga2c = NULL, De1 = NULL, De2 = NULL,
                      fort = FALSE, tol = 10^-10, disp = FALSE,
                      output = FALSE, out_se = FALSE, glob = FALSE)}
%- maybe also 'usage' for other objects documented here.


\arguments{
  \item{S}{matrix of all response sequences observed at least once in the sample and listed row-by-row
           (use NA for missing response)}
  \item{yv}{vector of the frequencies of every response configuration in \code{S}}
  \item{k1}{number of ability levels (or latent classes) for the 1st latent variable}
  \item{k2}{number of ability levels (or latent classes) for the 2nd latent variable}
  \item{X}{matrix of covariates that affects the weights}
  \item{start}{method of initialization of the algorithm}
  \item{link}{type of link function ("global" for global logits, "local" for local logits);
               with global logits a graded response model results; with local logits a  partial credit model results (with dichotomous responses, global  				logits is the same as using local logits resulting in the Rasch or the 2PL model depending on the value
              assigned to disc)}
  \item{disc}{indicator of constraints on the discriminating indices (FALSE = all equal to one, TRUE = free)}
  \item{difl}{indicator of constraints on the difficulty levels (FALSE = free, TRUE = rating scale parametrization)}
  \item{multi1}{matrix with a number of rows equal to the number of dimensions and elements in each row
               equal to the indices of the items measuring the dimension corresponding to that row for the 1st latent variable}
  \item{multi2}{matrix with a number of rows equal to the number of dimensions and elements in each row
               equal to the indices of the items measuring the dimension corresponding to that row for the 2nd latent variable}
  \item{piv1}{initial value of the vector of weights of the latent classes (if start="external") for the 1st latent variable}
  \item{piv2}{initial value of the vector of weights of the latent classes (if start="external") for the 2nd latent variable}
  \item{Phi}{initial value of the matrix of the conditional response probabilities (if start="external")}
  \item{ga1c}{initial value of the complete vector of discriminating indices (if start="external") for the 1st latent variable}
  \item{ga2c}{initial value of the complete vector of discriminating indices (if start="external") for the 2nd latent variable}
  \item{De1}{initial value of regression coefficients for the covariates (if start="external") affecting the 1st latent variable}
  \item{De2}{initial value of regression coefficients for the covariates (if start="external") affecting the 2st latent variable}
  \item{fort}{to use Fortran routines when possible}
  \item{tol}{tolerance level for checking convergence of the algorithm as relative difference between 
  	         consecutive log-likelihoods}
  \item{disp}{to display the likelihood evolution step by step}
  \item{output}{to return additional outputs (Piv, Pp, lkv, Xlabel, XX1dis, XX2dis)}
  \item{out_se}{to return standard errors}
  \item{glob}{to use global logits in the covariates}  
}

\details{
In order to ensure the model identifiability,  the following conditions must hold. First, the maximum number of items shared by the two latent variables is equal to the total number of items minus one, that is, the union of rows of \code{multi1} must differ from the union of rows of \code{multi2}. Second, suitable  constraints on the item parameters are required: one discriminanting index must be equal to 1 and one difficulty parameter must be equal to 0 for each dimension. The constrained items  may be  chosen  in an arbitrary way: by default the algorithm select the first element of each row of \code{multi1} and \code{multi2}. As a consequence, the user must pay attention to specify matrices \code{multi1} and \code{multi2} so that different items are constrained for each dimension.
}

\value{
\item{piv1}{estimated vector of weights of the latent classes (average of the weights in case of model with covariates) for the 1st latent variable}
\item{piv2}{estimated vector of weights of the latent classes (average of the weights in case of model with covariates) for the 2nd latent variable}
\item{Th1}{estimated matrix of ability levels for each dimension and latent class for the 1st latent variable}
\item{Th2}{estimated matrix of ability levels for each dimension and latent class for the 2nd latent variable}
\item{Bec}{estimated vector of difficulty levels for every item (split in two vectors if difl=TRUE)}
\item{ga1c}{estimated vector of discriminating indices for every item (with all elements equal to 1
	       with Rasch parametrization) for the 1st latent variable}
\item{ga2c}{estimated vector of discriminating indices for every item (with all elements equal to 1
	       with Rasch parametrization) for the 2nd latent variable}
\item{fv1}{vector indicating the reference item chosen for each latent dimension for the 1st latent variable}
\item{fv2}{vector indicating the reference item chosen for each latent dimension for the 2nd latent variable}
\item{Phi}{array of the conditional response probabilities for every item and each of the k1*k2 latent classes}
\item{De1}{matrix of regression coefficients for the multinomial (or global if glob=TRUE) logit model on the class weights for the 1st latent variable}
\item{De2}{matrix of regression coefficients for the multinomial (or global if glob=TRUE) logit model on the class weights for the 2nd latent variable}
\item{Piv}{matrix of the weights for every covariate pattern configuration (if output=TRUE)}
\item{Pp}{matrix of the posterior probabilities for each response configuration and latent class
	      (if output=TRUE)}
\item{lk}{log-likelihood at convergence of the EM algorithm}
\item{np}{number of free parameters}
\item{aic}{Akaike Information Criterion index}
\item{bic}{Bayesian Information Criterion index}
\item{ent}{Entropy index to measure the separation of classes}
\item{lkv}{Vector to trace the log-likelihood evolution across iterations (if output=TRUE)}
\item{seDe1}{Standard errors for vector De1 (if out_se=TRUE) for the 1st latent variable}
\item{seDe2}{Standard errors for vector De2 (if out_se=TRUE) for the 2nd latent variable}
\item{seTh1}{Standard errors for vector Th1 (if out_se=TRUE) for the 1st latent variable}
\item{seTh2}{Standard errors for vector Th2 (if out_se=TRUE) for the 2nd latent variable}
\item{seBec}{Standard errors for vector Bec of difficulty parameters (if out_se=TRUE)}
\item{sega1}{Standard errors for vector ga1c of discrimination indices (if out_se=TRUE) for the 1st latent variable}
\item{sega2}{Standard errors for vector ga2c of discrimination indices (if out_se=TRUE) for the 2nd latent variable}
\item{Vn}{Estimated variance-covariance matrix for all parameter estimates (if out_se=TRUE)}
\item{Xlabel}{Structure of the design matrix, for internal use (if output=TRUE)}
\item{XX1dis}{Design matrix for the covariates affecting the 1st latent variable (if output=TRUE)}
\item{XX2dis}{Design matrix for the covariates affecting the 2st latent variable  (if output=TRUE)}
}

\references{
Bacci, S.  and Bartolucci, F. (2015), A multidimensional finite mixture SEM for non-ignorable missing responses to test items, \emph{Structural Equation Modeling}, \bold{22}, 352-365.

Bartolucci, F. (2007), A class of multidimensional IRT models for testing unidimensionality and clustering
items, \emph{Psychometrika}, \bold{72}, 141-157.

Bacci, S., Bartolucci, F. and Gnaldi, M. (2014), A class of Multidimensional Latent Class IRT models for 
ordinal polytomous item responses, \emph{Communications in Statistics - Theory and Methods}, \bold{43}, 787-800.
}

\author{Francesco Bartolucci, Silvia Bacci - University of Perugia (IT)}

%\note{
%%%  ~~further notes~~
%}

%% ~Make other sections like Warning with \section{Warning }{....} ~
%\section{Warning }{
%}

%\seealso{
%%% ~~objects to See Also as \code{\link{help}}, ~~~
%}

\examples{

\dontrun{
# Fit the model under different within-item multidimensional structures
# for SF12_nomiss data
data(SF12_nomiss)
S = SF12_nomiss[,1:12]
X = SF12_nomiss[,13]

# Graded response model with two latent variables sharing six items (free
# discrimination and difficulty parameters; two latent classes for each
# latent variable; one covariate):
multi1 = c(1:5, 8:12)
multi2 = c(6:12, 1)
tol = 10^-6  # decrease tolerance to obtain more reliable results
out1 = est_multi_poly_within(S=S,k1=2,k2=2,X=X,link="global",disc=TRUE,
                             multi1=multi1,multi2=multi2,disp=TRUE,
                             out_se=TRUE,tol=tol) 
                             
# Partial credit model with two latent variables sharing eleven items
# (free discrimination and difficulty parameters; two latent classes for
# the 1st latent variable and three latent classes for the 2nd latent
# variable; one covariate):
multi1 = 1:12
multi2 = 2:12
out2 = est_multi_poly_within(S=S,k1=2,k2=3,X=X,link="local",disc=TRUE,
                             multi1=multi1,multi2=multi2,disp=TRUE,tol=tol)
                               
# Display output:
out2$lk
out2$Th1 
out2$piv1
out2$Th2
out2$piv2
out2$De1
out2$De2
}

\dontrun{
## Fit the model under different situations for RLMS data
# Example of use of the function to account for non-ignorable missing
# item responses 
data(RLMS)
X = RLMS[,1:4]
Y = RLMS[,6:9]
YR = cbind(Y,1*(!is.na(Y)))
multi1 = 1:4
multi2 = 5:8
tol = 10^-6  # decrease tolerance to obtain more reliable results

# MAR model
out0 = est_multi_poly_within(YR,k1=3,k2=2,X=X,link="global",
                 disc=TRUE,multi1=multi1,multi2=multi2,disp=TRUE,
                 out_se=TRUE,glob=TRUE,tol=tol) 
                 
# NMAR model
multi1 = 1:8
out1 = est_multi_poly_within(YR,k1=3,k2=2,X=X,link="global",
                 disc=TRUE,multi1=multi1,multi2=multi2,disp=TRUE,
                 out_se=TRUE,glob=TRUE,tol=tol)
                   
# testing effect of the latent trait on missingness
c(out0$bic,out1$bic)
(test1 = out1$ga1c[-1]/out1$sega1c[-1])
}
}

%% Add one or more standard keywords, see file 'KEYWORDS' in the
%% R documentation directory.
\keyword{maximum likelihood estimation}
\keyword{Expectation-Maximization algorithm}% __ONLY ONE__ keyword per line
