% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pmt.R
\name{pmt}
\alias{pmt}
\alias{pmts}
\alias{define_pmt}
\title{Syntactic Sugar for Object Construction}
\usage{
pmt(key, ...)

pmts(
  which = c("all", "onesample", "twosample", "ksample", "multcomp", "paired", "rcbd",
    "association", "table")
)

define_pmt(
  inherit = c("twosample", "ksample", "paired", "rcbd", "association", "table"),
  statistic,
  rejection = c("lr", "l", "r"),
  scoring = "none",
  n_permu = 10000,
  name = "User-Defined Permutation Test",
  alternative = NULL,
  depends = character(),
  plugins = character(),
  includes = character()
)
}
\arguments{
\item{key}{a character string specifying the test. Check \code{pmts()} for valid keys.}

\item{...}{extra parameters passed to the constructor.}

\item{which}{a character string specifying the desired tests.}

\item{inherit}{a character string specifying the type of permutation test.}

\item{statistic}{definition of the test statistic. See details.}

\item{rejection}{a character string specifying where the rejection region is.}

\item{scoring}{one of:
- a character string in \code{c("none", "rank", "vw", "expon")} specifying the scoring system
- a function that takes a numeric vector and returns an equal-length score vector}

\item{n_permu}{an integer indicating number of permutations for the permutation distribution. If set to \code{0}, all permutations will be used.}

\item{name}{a character string specifying the name of the test.}

\item{alternative}{a character string describing the alternative hypothesis.}

\item{depends, plugins, includes}{passed to \code{\link[Rcpp:cppFunction]{Rcpp::cppFunction()}}.}
}
\value{
a test object corresponding to the specified key.

a data frame containing keys and corresponding tests implemented in this package.

a test object based on the specified statistic.
}
\description{
Construct test objects in a unified way.
}
\details{
The test statistic can be defined using either R or Rcpp, with the \code{statistic} parameter specified as:
\itemize{
\item R: a function returning a closure that returns a double.
\item Rcpp: a character string defining a captureless lambda (since C++11) returning another lambda that captures by value, accepts parameters of the same type, and returns a double.
}

The purpose of this design is to pre-calculate certain constants that remain invariant during permutation.

When using Rcpp, the parameters for different \code{inherit} are listed as follows. Note that the names can be customized, and the types can be replaced with \code{auto} (thanks to the support for generic lambdas in C++14). See examples.\tabular{ccc}{
   \code{inherit} \tab Parameter 1 \tab Parameter 2 \cr
   \code{"twosample"} \tab \verb{const NumericVector& sample_1} \tab \verb{const NumericVector& sample_2} \cr
   \code{"ksample"} \tab \verb{const NumericVector& combined_sample} \tab \verb{const IntegerVector& one_based_group_index} \cr
   \code{"paired"} \tab \verb{const NumericVector& sample_1} \tab \verb{const NumericVector& sample_2} \cr
   \code{"rcbd"} \tab \verb{const NumericMatrix& block_as_column_data} \tab  \cr
   \code{"association"} \tab \verb{const NumericVector& sample_1} \tab \verb{const NumericVector& sample_2} \cr
   \code{"table"} \tab \verb{const IntegerMatrix& contingency_table} \tab  \cr
}


When using R, the parameters should be the R equivalents of these.
}
\note{
\itemize{
\item \code{statistic} should not cause errors or return missing values.
\item The data is permuted in-place. Therefore, modifications to the data within \code{statistic} may lead to incorrect results. Since R has copy-on-modify semantics but C++ does not, it is recommended to pass const references when using Rcpp in \code{define_pmt}, as shown in the table above.
}
}
\examples{
pmt("twosample.wilcoxon")

pmts("ksample")

x <- rnorm(5)
y <- rnorm(5, 1)

t <- define_pmt(
    inherit = "twosample",
    scoring = base::rank, # equivalent to "rank"
    statistic = function(...) function(x, y) sum(x)
)$test(x, y)$print()

t$scoring <- function(x) qnorm(rank(x) / (length(x) + 1)) # equivalent to "vw"
t$print()

t$n_permu <- 0
t$print()

\donttest{
r <- define_pmt(
    inherit = "twosample", n_permu = 1e5,
    statistic = function(x, y) {
        m <- length(x)
        n <- length(y)
        function(x, y) sum(x) / m - sum(y) / n
    }
)


rcpp <- define_pmt(
    inherit = "twosample", n_permu = 1e5,
    statistic = "[](const auto& x, const auto& y) {
        auto m = x.length();
        auto n = y.length();
        return [=](const auto& x, const auto& y) {
            return sum(x) / m - sum(y) / n;
        };
    }"
)

# equivalent
# rcpp <- define_pmt(
#     inherit = "twosample", n_permu = 1e5,
#     statistic = "[](const NumericVector& x, const NumericVector& y) {
#         R_xlen_t m = x.length();
#         R_xlen_t n = y.length();
#         return [m, n](const NumericVector& x, const NumericVector& y) -> double {
#             return sum(x) / m - sum(y) / n;
#         };
#     }"
# )

options(LearnNonparam.pmt_progress = FALSE)
system.time(r$test(x, y))
system.time(rcpp$test(x, y))
}

}
