#' splitIncucyteDRCPlateData
#'
#' Function to construct a list of IncucyteDRCSet objects from a IncucyteDRCPlateData object.
#'
#' @param platemap Platemap dataframe of the form generated by importPlatemapXML
#' @param platedata IncucyteDRCPlateData object to be split
#' @param group_columns Vector of columns that are present in the data frame to generate the groups.
#' @param cut_time The time at which to extract the data for the dose response curve.  Default is NULL.
#'
#' @return list of IncucyteDRCSet objects
#' @export
#'
#' @examples
#' pm_file <- system.file(file='extdata/example.PlateMap', package='IncucyteDRC')
#' test_pm <- importPlatemapXML(pm_file)
#' data_file <- system.file(file='extdata/example_data.txt', package='IncucyteDRC')
#' test_data <- importIncucyteData(data_file, metric='pc')
#'
#' test_list <- splitIncucyteDRCPlateData(test_pm, test_data, group_columns='growthcondition')
#'
#' print(test_list)


splitIncucyteDRCPlateData <- function(platemap, platedata, group_columns, cut_time=NULL) {

    stopifnot(inherits(platemap, 'data.frame'))
    stopifnot(inherits(platedata, 'IncucyteDRCPlateData'))
    stopifnot(group_columns %in% colnames(platemap))

    if(is.null(attr(platemap, 'IncucyteDRCPlatemap'))) {
        warning('Recommended that platemap data frames are parsed through importPlatemap function to check formatting')
    }

    #get rid of blanks
    platemap <- platemap %>% dplyr::filter(samptype %in% c('C', 'S'))

    #generate group ids using group_columns
    platemap <-  platemap %>%  dplyr::mutate(group_idx = dplyr::group_indices_(platemap, .dots=group_columns) )

    #split platemap based on group_idx
    platemap_list <- split(platemap, platemap$group_idx)

    #lapply makeIncucyteData across list of platemaps
    outdata <- lapply(platemap_list, function(x) makeIncucyteDRCSet(platemap = x, platedata = platedata, cut_time = cut_time, pm_warn=FALSE))

    #lapply populateIncucyteDRCSetMetadata across the list object
    outdata <- lapply(outdata, populateIncucyteDRCSetMetadata, group_columns=group_columns)

    #return a IncucyteDRCSetList or IncucyteDRCSet number of IncucyteDRCSets
    if(length(outdata) > 1) {
        class(outdata) <- 'IncucyteDRCSetList'
    } else {
        outdata <- outdata[[1]]
    }

    return(outdata)

}
