% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/starfm_job.R
\name{starfm_job}
\alias{starfm_job}
\title{Execute a single self-contained self-contained time-series imagefusion job using STARFM}
\usage{
starfm_job(
  input_filenames,
  input_resolutions,
  input_dates,
  pred_dates,
  pred_filenames,
  pred_area,
  winsize,
  date1,
  date3,
  n_cores,
  logscale_factor,
  spectral_uncertainty,
  temporal_uncertainty,
  number_classes,
  hightag,
  lowtag,
  MASKIMG_options,
  MASKRANGE_options,
  output_masks,
  use_nodata_value,
  use_strict_filtering,
  double_pair_mode,
  use_temp_diff_for_weights,
  do_copy_on_zero_diff,
  verbose = TRUE
)
}
\arguments{
\item{input_filenames}{A string vector containing the filenames of the input images}

\item{input_resolutions}{A string vector containing the resolution-tags (corresponding to the arguments \code{hightag} and \code{lowtag}, which are by default "high" and "low") of the input images.}

\item{input_dates}{An integer vector containing the dates of the input images.}

\item{pred_dates}{An integer vector containing the dates for which images should be predicted.}

\item{pred_filenames}{A string vector containing the filenames for the predicted images. Must match \code{pred_dates} in length and order. Must include an extension relating to one of the \href{https://gdal.org/drivers/raster/index.html}{drivers supported by GDAL}, such as ".tif".}

\item{pred_area}{(Optional) An integer vector containing parameters in image coordinates for a bounding box which specifies the prediction area. The prediction will only be done in this area. (x_min, y_min, width, height). By default will use the entire area of the first input image.}

\item{winsize}{(Optional) Window size of the rectangle around the current pixel. Default is 51.}

\item{date1}{(Optional) Set the date of the first input image pair. By default, will use the pair with the lowest date value.}

\item{date3}{(Optional) Set the date of the second input image pair. By default, will use the pair with the highest date value. Disregarded if using double pair mode.}

\item{n_cores}{(Optional) Set the number of cores to use when using parallelization. Default is 1.}

\item{logscale_factor}{(Optional) When using a positive scale, the logistic weighting formula is used, which reduces the influence of spectral and temporal differences. Default is 0, i. e. logistic formula not used}

\item{spectral_uncertainty}{(Optional) This spectral uncertainty value will influence the spectral difference value. Default is 1 for 8 bit images (INT1U and INT1S), 50 otherwise.}

\item{temporal_uncertainty}{(Optional) This spectral uncertainty value will influence the spectral difference value. Default is 1 for 8 bit images (INT1U and INT1S), 50 otherwise.}

\item{number_classes}{(Optional) The number of classes used for similarity. Note all channels of a pixel are considered for similarity. So this value holds for each channel, e. g. with 3 channels there are n^3 classes. Default: 40.}

\item{hightag}{(Optional) A string which is used in \code{input_resolutions} to describe the high-resolution images. Default is "high".}

\item{lowtag}{(Optional) A string which is used in \code{input_resolutions} to describe the low-resolution images.  Default is "low".}

\item{MASKIMG_options}{(Optional) A string containing information for a mask image (8-bit, boolean, i. e. consists of 0 and 255). "For all input images the pixel values at the locations where the mask is 0 is replaced by the mean value." Example: \code{--mask-img=some_image.png}}

\item{MASKRANGE_options}{(Optional) Specify one or more intervals for valid values. Locations with invalid values will be masked out. Ranges should be given in the format \code{'[<float>,<float>]'}, \code{'(<float>,<float>)'}, \code{'[<float>,<float>'} or \code{'<float>,<float>]'}. There are a couple of options:' \itemize{
\item{"--mask-valid-ranges"}{ Intervals which are marked as valid. Valid ranges can excluded from invalid ranges or vice versa, depending on the order of options.}
\item{"--mask-invalid-ranges"}{ Intervals which are marked as invalid. Invalid intervals can be excluded from valid ranges or vice versa, depending on the order of options.}
\item{"--mask-high-res-valid-ranges"}{ This is the same as --mask-valid-ranges, but is applied only for the high resolution images.}
\item{"--mask-high-res-invalid-ranges"}{ This is the same as --mask-invalid-ranges, but is applied only for the high resolution images.}
\item{"--mask-low-res-valid-ranges"}{ This is the same as --mask-valid-ranges, but is applied only for the low resolution images.}
\item{"--mask-low-res-invalid-ranges"}{ This is the same as --mask-invalid-ranges, but is applied only for the low resolution images.}
}}

\item{output_masks}{(Optional) Write mask images to disk? Default is "false".}

\item{use_nodata_value}{(Optional) Use the nodata value as invalid range for masking? Default is "true".}

\item{use_strict_filtering}{(Optional) Use strict filtering, which means that candidate pixels will be accepted only if they have less temporal \emph{and} spectral difference than the central pixel (like in the paper). Default is "false".}

\item{double_pair_mode}{(Optional) Use two dates \code{date1} and \code{date3} for prediction, instead of just \code{date1} for all predictions? Default is "true" if \emph{all} the pred dates are in between input pairs, and "false" otherwise. Note: It may be desirable to predict in double-pair mode where possible, as in the following example: \code{[(7) 10 12 (13) 14] } , where we may wish to predict 10 and 12 in double pair mode, but can only predict 14 in single-pair mode. Do achieve this it is necessary to split the task into different jobs. Default is "true" if all pred_dates are between pair dates and "false" otherwise.}

\item{use_temp_diff_for_weights}{(Optional) Use temporal difference in the candidates weight (like in the paper)? Default is to use temporal weighting in double pair mode, and to not use it in single pair mode.}

\item{do_copy_on_zero_diff}{(Optional) Predict for all pixels, even for pixels with zero temporal or spectral difference (behavior of the reference implementation). Default is "false".}

\item{verbose}{(Optional) Print progress updates to console? Default is "true".}
}
\value{
Nothing. Output files are written to disk. The Geoinformation for the output images is adopted from the first input pair images.
}
\description{
A wrapper function for \code{execute_starfm_job_cpp}. Intended to execute a single job, that is a number of predictions based on the same input pair(s). It ensures that all of the arguments passed are of the correct type and creates sensible defaults.
}
\details{
Executes the STARFM algorithm to create a number of synthetic high-resolution images from either two pairs (double pair mode) or one pair (single pair mode) of matching high- and low-resolution images. Assumes that the input images already have matching size. See the original paper for details. \itemize{
\item{  For the weighting (10) states: \eqn{C = S T D}  but we use  \eqn{C = (S+1)(T+1)D}, according to the reference implementation. With \code{logscale_factor}, the weighting formula can be changed to \eqn{C = ln{(Sb+2)}ln{(Tb+1)D}}}
\item{ In addition to the temporal uncertainty \eqn{\sigma_t} (see \code{temporal_uncertainty}) and the spectral uncertainty\eqn{ \sigma_s} (see \code{spectral_uncertainty}) there will be used a \emph{combined uncertainty} \eqn{\sigma_c := \sqrt{\sigma_t^2 + \sigma_s^2} }. This will be used in the candidate weighting: If \eqn{(S + 1) \, (T + 1) < \sigma_c }, then \eqn{C = 1 } instead of the formula above.}
\item{Considering candidate weighting again, there is an option \code{use_tempdiff_for_weights} to not use the temporal difference for the weighting (also not for the combined uncertainty check above), i. e. T = 0 then. This is also the default behavior.}{ }
\item{The basic assumption of the original paper that with zero spectral or temporal difference the central pixel will be chosen is wrong since there might be multiple pixels with zero difference within one window. Also due to the addition of 1 to the spectral and temporal differences, the weight will not increase so dramatically from a zero difference. However, these assumptions can be enforced with \code{do_copy_on_zero_diff}, which is the default behavior.}{ }
\item{The paper states that a good candidate should satisfy (15) and (16). This can be set with use_strict_filtering, which is by default used. However the other behavior, that a candidate should fulfill (15) or (16), as in the reference implementation, can be also be selected with that option.}
\item{The paper uses max in (15) and (16), which would choose the largest spectral and temporal difference from all input pairs (only one or two are possible). Since this should filter out bad candidates, we believe this is a mistake and should be min instead of max, like it is done in the reference implementation. So this implementation uses min here.}
}
}
\examples{
# Load required libraries
library(ImageFusion)
library(raster)
# Get filesnames of high resolution images
landsat <- list.files(
  system.file("landsat/filled",
              package = "ImageFusion"),
  ".tif",
  recursive = TRUE,
  full.names = TRUE
)

# Get filesnames of low resolution images
modis <- list.files(
  system.file("modis",
              package = "ImageFusion"),
  ".tif",
  recursive = TRUE,
  full.names = TRUE
)

#Select the first two landsat images 
landsat_sel <- landsat[1:2]
#Select the corresponding modis images
modis_sel <- modis[1:10]
# Create output directory in temporary folder
out_dir <- file.path(tempdir(),"Outputs")
if(!dir.exists(out_dir)) dir.create(out_dir, recursive = TRUE)
#Run the job, fusing two images
starfm_job(input_filenames = c(landsat_sel,modis_sel),
           input_resolutions = c("high","high",
                                 "low","low","low",
                                 "low","low","low",
                                 "low","low","low","low"),
           input_dates = c(68,77,68,69,70,71,72,73,74,75,76,77),
           pred_dates = c(73,77),
           pred_filenames = c(file.path(out_dir,"starfm_73.tif"),
                              file.path(out_dir,"starfm_77.tif"))
)
# remove the output directory
unlink(out_dir,recursive = TRUE)
}
\references{
Gao, Feng, et al. "On the blending of the Landsat and MODIS surface reflectance: Predicting daily Landsat surface reflectance." IEEE Transactions on Geoscience and Remote sensing 44.8 (2006): 2207-2218.
}
\author{
Christof Kaufmann (C++)

Johannes Mast (R)
}
