% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/objectExtract.R
\name{objectExtract}
\alias{objectExtract}
\title{Object Extraction}
\source{
For image decompression, Lee Kamentsky's code porting from \url{https://github.com/ome/bioformats/blob/4146b9a1797501f0fec7d6cfe69124959bff96ee/components/formats-bsd/src/loci/formats/in/FlowSightReader.java}\cr
cited in \url{https://linkinghub.elsevier.com/retrieve/pii/S1046-2023(16)30291-2}\cr\cr
BSD implementations of Bio-Formats readers and writers\cr
%%
Copyright (C) 2005 - 2017 Open Microscopy Environment:\cr
  - Board of Regents of the University of Wisconsin-Madison\cr
  - Glencoe Software, Inc.\cr
  - University of Dundee\cr
%%
Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:\cr
1. Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.\cr
2. Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation
    and/or other materials provided with the distribution.\cr
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.
}
\usage{
objectExtract(ifd, param, verbose = FALSE, bypass = FALSE, ...)
}
\arguments{
\item{ifd}{list of sub elements of IFD data information extracted by \code{\link{getIFD}}. This parameter can't be missing.}

\item{param}{object of class `IFC_param`, containing extraction parameters defined by \code{\link{objectParam}}.\cr
This argument is not mandatory but it may allow to save time for repeated image export on same file.
If this parameter is missing, \code{\link{objectExtract}} will use extra ... to pass arguments to \code{\link{objectParam}} to control object extraction.\cr
However, if 'param' is provided, '...' will be ignored.}

\item{verbose}{whether to display information (use for debugging purpose). Default is FALSE.}

\item{bypass}{whether to bypass checks on 'ifd' and 'param'. Default is FALSE.}

\item{...}{other arguments to be passed to \code{\link{objectParam}}.\cr
If 'param' is not provided then '...' will be used to compute 'param'.\cr
/!\ If not any of 'fileName', 'info' can be found in '...' then attr(ifd, "fileName_image") will be used as 'fileName' input parameter to pass to \code{\link{objectParam}}.}
}
\value{
A list (for every extracted objects) of list (for every exported channels) depending on "export" parameter:\cr
-"matrix", a matrix when 'mode' is set to "raw" or "gray" OR an array when 'mode' == "rgb",\cr
-"base64", a data-uri string,\cr
-"file", an invisible file path corresponding to the location of exported file(s).
}
\description{
Extracts / Decompress objects stored in RIF or CIF Files.
}
\examples{
if(requireNamespace("IFCdata", quietly = TRUE)) {
  ## use a cif file
  file_cif <- system.file("extdata", "example.cif", package = "IFCdata")
  cif_offs <- getOffsets(fileName = file_cif, fast = TRUE)
  ## extract infomation
  info <- getInfo(fileName = file_cif, from = "analysis")
  ## retrieve number of objects stored
  nobj <- as.integer(info$objcount)
  ## randomly subset the offsets of at most 5 "img" objects
  sel = sample(0:(nobj-1), min(5, nobj))
  sub_offs <- subsetOffsets(cif_offs, objects = sel, image_type = "img")
  ## read IFDs from these "img" objects
  IFDs <- getIFD(fileName = file_cif, offsets = sub_offs)
  ## extract raw data of these"img" objects to matrix
  raw = objectExtract(ifd = IFDs, info = info, mode = "raw", 
                      export = "matrix")
  ## extract base64 "rgb" colorized version of these "img" objects to base64
  b64 = objectExtract(ifd = IFDs, info = info, mode = "rgb", 
                      export = "base64", base64_id = TRUE,
                      write_to = "example_\%o_\%c.png")
  ## use DisplayGallery to show the first "img" objects and play with ... extra parameters
  ## force_range, add_noise, selection, composite, see objectParam
  DisplayGallery(info = info, offsets = cif_offs, objects = sel,
                 base64_id = TRUE, write_to = "example_\%o_\%c.png",
                 force_range = c(FALSE,TRUE,FALSE,TRUE), add_noise = FALSE,
                 selection = c(1,2,4,6), composite = "1.7/4.3")
} else {
  message(sprintf('Please run `install.packages("IFCdata", repos = "\%s", type = "source")` \%s',
                  'https://gitdemont.github.io/IFCdata/',
                  'to install extra files required to run this example.'))
}
}
