% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_GeDSMethod.R
\name{plot,GeDS-method}
\alias{plot,GeDS-method}
\alias{plot.GeDS}
\title{Plot method for GeDS objects.}
\usage{
\S4method{plot}{GeDS}(
  x,
  f = NULL,
  which,
  DEV = FALSE,
  ask = FALSE,
  main,
  legend.pos = "topright",
  new.window = FALSE,
  wait = 0.5,
  n = 3L,
  type = c("none", "Polygon", "NCI", "ACI"),
  ...
)
}
\arguments{
\item{x}{a \code{\link{GeDS-Class}} object from which the GeDS fit(s) should
be extracted.}

\item{f}{(optional) specifies the underlying function or generating process
to which the model was fit. This parameter is useful if the user wishes to
plot the specified function/process alongside the model fit and the data}

\item{which}{a numeric vector specifying the iterations of stage A for which
the corresponding GeDS fits should be plotted.
It has to be a subset of  \code{1:nrow(x$stored)}. See details.}

\item{DEV}{logical variable specifying whether a plot representing the
deviance at each iteration of stage A should be produced or not.}

\item{ask}{logical variable specifying whether the user should be prompted
before changing the plot page.}

\item{main}{optional character string to be used as a title of the plot.}

\item{legend.pos}{the position of the legend within the panel. See
\link[graphics]{legend} for details.}

\item{new.window}{logical variable specifying whether the plot should be
shown in a new window or in the active one.}

\item{wait}{time, in seconds, the system should wait before plotting a new
page. Ignored if \code{ask = TRUE}.}

\item{n}{integer value (2, 3 or 4) specifying the order (\eqn{=} degree
\eqn{+ 1}) of the GeDS fit that should be plotted. By default equal to
\code{3L}. Non-integer values will be passed to the function
\code{\link{as.integer}}.}

\item{type}{character string specifying the type of plot required. Should be
set either to "\code{Polygon}" if the user wants to get also the control
polygon of the GeDS fit,  \code{"NCI"} or  \code{"ACI"} if 95\% confidence
bands for the predictions should be plotted (see details) or \code{"none"} if
only the fitted GeDS curve should be plotted. Applies only when plotting a
univariate spline regression.}

\item{...}{further arguments to be passed to the
\code{\link[graphics]{plot.default}} function.}
}
\description{
Plot method for GeDS objects. Plots GeDS fits.
}
\details{
This method is provided in order to allow the user to plot the GeDS  fits
contained in the \code{\link{GeDS-Class}} objects.

Since in Stage A of the GeDS algorithm the knots of a linear spline fit are
sequentially located, one at a time, the user may wish to visually inspect
this process using the argument \code{which}. The latter specifies a
particular iteration number (or a vector of such numbers) for which the
corresponding linear fit(s) should be plotted. The \code{ask} and \code{wait}
arguments can help the user to manage these pages.

By means of \code{ask} the user can determine for how long each page should
appear on the screen. Pages are sequentially replaced by pressing the enter
button.

Note that, in order to ensure stability, if the object was produced by the
function \code{\link{GGeDS}}, plotting intermediate fits of stage A is
allowed  only if \code{n = 2}, in contrast to objects produced by 
\code{\link{NGeDS}} for which plotting intermediate results is allowed also
for \code{n = }2 or 3 results.

The confidence intervals obtained by setting \code{type = "NCI"} are
approximate local bands obtained considering the knots as fixed constants.
Hence the columns of the design matrix are seen as covariates and standard
methodology relying on the \code{se.fit} option of \code{predict.lm} or
\code{predict.glm} is applied.

Setting \code{type = "ACI"}, asymptotic confidence intervals are plotted.
This option is applicable only if the canonical link function has been used
in the fitting procedure.
}
\examples{
###################################################
# Generate a data sample for the response variable
# Y and the single covariate X, assuming Normal noise
set.seed(123)
N <- 500
f_1 <- function(x) (10*x/(1+100*x^2))*4+4
X <- sort(runif(N, min = -2, max = 2))
# Specify a model for the mean of Y to include only a component
# non-linear in X, defined by the function f_1
means <- f_1(X)
# Add (Normal) noise to the mean of Y
Y <- rnorm(N, means, sd = 0.1)

# Fit a Normal GeDS regression using NGeDS
(Gmod <- NGeDS(Y ~ f(X), beta = 0.6, phi = 0.995, Xextr = c(-2,2)))

# Plot the final quadratic GeDS fit (red solid line)
# with its control polygon (blue dashed line)
plot(Gmod)

# Plot the quadratic fit obtained from the linear fit at the 10th
# iteration of stage A i.e. after 9 internal knots have been inserted
# by the GeDS procedure
plot(Gmod, which=10)

# Generate plots of all the intermediate fits obtained
# by running the GeDS procedure
\dontrun{
plot(Gmod, which=1:16)
}

###################################################
# Generate a data sample for the response variable Y and the covariate
# X assuming Poisson distributed error and a log link function

set.seed(123)
N <- 500
f_1 <- function(x) (10*x/(1+100*x^2))*4+4
X <- sort(runif(N ,min = -2, max = 2))
# Specify a model for the mean of Y to include only a component
# non-linear in X, defined by the function f_1
means <- exp(f_1(X))
# Generate Poisson distributed Y according to the mean model
Y <- rpois(N,means)

# Fit a Poisson GeDS regression model using GGeDS
(Gmod2 <- GGeDS(Y ~ f(X), beta = 0.2, phi = 0.995, family = poisson(),
                Xextr = c(-2,2)))

# similar plots as before, but for the linear fit
plot(Gmod2, n = 2)
plot(Gmod2, which = 10, n = 2)
\dontrun{
plot(Gmod2, which = 1:16, n = 2)
plot(Gmod2, which = 1:16, n = 2, ask = T)
}

}
\seealso{
\code{\link{NGeDS}} and \code{\link{GGeDS}};
\code{\link[graphics]{plot}}.
}
