% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mvspectrum2wcov.R
\name{mvspectrum2wcov}
\alias{mvspectrum2wcov}
\alias{weightvector2entropy_wcov}
\title{Compute (weighted) covariance matrix from frequency spectrum}
\usage{
mvspectrum2wcov(mvspectrum.output, kernel.weights = 1)

weightvector2entropy_wcov(
  weightvector = NULL,
  f.U,
  f.current = NULL,
  entropy.control = list()
)
}
\arguments{
\item{mvspectrum.output}{an object of class \code{"mvspectrum"} representing
the multivariate spectrum of \eqn{\mathbf{X}_t} (not necessarily \code{normalize}d).}

\item{kernel.weights}{numeric; weights for each frequency. By default uses 
weights that average out to \code{1}.}

\item{weightvector}{numeric; weights \eqn{\mathbf{w}} for 
\eqn{y_t = \mathbf{U}_t \mathbf{w}}. Must have unit norm in \eqn{\ell^2}.}

\item{f.U}{multivariate spectrum of class \code{'mvspectrum'} with 
\code{normalize = TRUE}.}

\item{f.current}{numeric; spectral density estimate of 
\eqn{y_t=\mathbf{U}_t \mathbf{w}} for the current estimate 
\eqn{\widehat{\mathbf{w}}_i} (required for 
\code{foreca.EM.M_step}; optional for \code{foreca.EM.h}).}

\item{entropy.control}{list; control settings for entropy estimation.
See \code{\link{complete_entropy_control}} for details.}
}
\value{
A symmetric \eqn{n \times n} matrix.

If \code{kernel.weights} \eqn{\geq 0}, then it is positive semi-definite;
otherwise, it is symmetric but not necessarily positive semi-definite.
}
\description{
\code{mvspectrum2wcov} computes a (weighted) covariance matrix estimate 
from the frequency spectrum (see Details).

\code{weightvector2entropy_wcov} computes the weighted covariance
matrix using the negative entropy of the univariate spectrum (given the
weightvector) as kernel weights.  This matrix is the objective matrix
for many \code{foreca.*} algorithms.
}
\details{
The covariance matrix of a multivariate time series satisfies the identity
\deqn{
\Sigma_{X} \equiv \int_{-\pi}^{\pi} S_{X}(\lambda) d \lambda.
}

A generalized covariance matrix estimate can thus be obtained using a weighted average
\deqn{
\tilde{\Sigma}_X = \int_{-\pi}^{\pi} K(\lambda) S_{X}(\lambda) d \lambda,
}
where \eqn{K(\lambda)} is a kernel symmetric around \eqn{0} which averages out to
\eqn{1} over the interval \eqn{[-\pi, \pi]}, i.e.,
 \eqn{\frac{1}{2 \pi} \int_{-\pi}^{\pi} K(\lambda) d \lambda = 1}. 
 This allows one to remove or amplify specific frequencies in the covariance matrix 
 estimation.

For ForeCA \code{mvspectrum2wcov} is especially important as we use
\deqn{
K(\lambda) = -\log f_y(\lambda),
}
as the \emph{weights} (their average is not \eqn{1}!).  This particular kernel
weight is implemented as a wrapper in \code{weightvector2entropy_wcov}.
}
\examples{

nn <- 50
YY <- cbind(rnorm(nn), arima.sim(n = nn, list(ar = 0.9)), rnorm(nn))
XX <- YY \%*\% matrix(rnorm(9), ncol = 3)  # random mix
XX <- scale(XX, scale = FALSE, center = TRUE)

# sample estimate of covariance matrix
Sigma.hat <- cov(XX)
dimnames(Sigma.hat) <- NULL

# using the frequency spectrum
SS <- mvspectrum(XX, "wosa")
Sigma.hat.freq <- mvspectrum2wcov(SS)

layout(matrix(1:4, ncol = 2))
par(mar = c(2, 2, 1, 1))
plot(c(Sigma.hat/Sigma.hat.freq))
abline(h = 1)

image(Sigma.hat)
image(Sigma.hat.freq)
image(Sigma.hat / Sigma.hat.freq)

# examples for entropy wcov
XX <- diff(log(EuStockMarkets)) * 100
UU <- whiten(XX)$U
ff <- mvspectrum(UU, 'wosa', normalize = TRUE)

ww0 <- initialize_weightvector(num.series = ncol(XX), method = 'rnorm')

weightvector2entropy_wcov(ww0, ff,
                          entropy.control = 
                            list(prior.weight = 0.1))
}
\seealso{
\code{\link{mvspectrum}}
}
\keyword{ts}
