% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/community.consensus.R
\name{community.consensus}
\alias{community.consensus}
\title{Applies the Consensus Clustering Method (Louvain only)}
\usage{
community.consensus(
  network,
  order = c("lower", "higher"),
  resolution = 1,
  consensus.method = c("highest_modularity", "iterative", "most_common", "lowest_tefi"),
  consensus.iter = 1000,
  correlation.matrix = NULL,
  allow.singleton = FALSE,
  membership.only = TRUE,
  ...
)
}
\arguments{
\item{network}{Matrix or \code{\link{igraph}} network object}

\item{order}{Character (length = 1).
Defaults to \code{"higher"}.
Whether \code{"lower"} or \code{"higher"} order memberships from
the Louvain algorithm should be obtained for the consensus.
The \code{"lower"} order Louvain memberships are from the first
initial pass of the Louvain algorithm whereas the \code{"higher"}
order Louvain memberships are from the last pass of the Louvain
algorithm}

\item{resolution}{Numeric (length = 1).
A parameter that adjusts modularity to allow the algorithm to
prefer smaller (\code{resolution} > 1) or larger
(0 < \code{resolution} < 1) communities.
Defaults to \code{1} (standard modularity computation)}

\item{consensus.method}{Character (length = 1).
Defaults to \code{"most_common"}.
Available options for arriving at a consensus (\emph{Note}: 
All methods except \code{"iterative"} are considered experimental
until validated):

\itemize{

\item{\code{"highest_modularity"} --- }
{\strong{EXPERIMENTAL.} Selects the community solution with the highest modularity across
the applications. Modularity is a reasonable metric for identifying the number
of communities in a network but it comes with limitations (e.g., resolution limit)}

\item{\code{"iterative"} --- }
{The original approach proposed by Lancichinetti & Fortunato (2012). See
"Details" for more information}

\item{\code{"most_common"} --- }
{Selects the community solution that appears the most
frequently across the applications. The idea behind this method is that the solution
that appears most often will be the most likely solution for the algorithm as well
as most reproducible. Can be less stable as the number of nodes increase requiring
a larger value for \code{consensus.iter}.  This method is the \strong{default}}

\item{\code{"lowest_tefi"} --- }
{\strong{EXPERIMENTAL.} Selects the community solution with the lowest Total Entropy
Fit Index (\code{\link[EGAnet]{tefi}}) across the applications. TEFI is a reasonable metric
to identify the number of communities in a network based on Golino, Moulder et al. (2020)}

}}

\item{consensus.iter}{Numeric (length = 1).
Number of algorithm applications to the network.
Defaults to \code{1000}}

\item{correlation.matrix}{Symmetric matrix.
Used for computation of \code{\link[EGAnet]{tefi}}.
Only needed when \code{consensus.method = "tefi"}}

\item{allow.singleton}{Boolean (length = 1).
Whether singleton or single node communities should be allowed.
Defaults to \code{FALSE}.
When \code{FALSE}, singleton communities will be set to
missing (\code{NA}); otherwise, when \code{TRUE}, singleton
communities will be allowed}

\item{membership.only}{Boolean.
Whether the memberships only should be output.
Defaults to \code{TRUE}.
Set to \code{FALSE} to obtain all output for the
community detection algorithm}

\item{...}{Not actually used but makes it easier for general functionality
in the package}
}
\value{
Returns either a vector with the selected solution
or a list when \code{membership.only = FALSE}:

\item{selected_solution}{Resulting solution from the consensus method}

\item{memberships}{Matrix of memberships across the consensus iterations}

\item{proportion_table}{For methods that use frequency, a table that
reports those frequencies alongside their corresponding memberships}
}
\description{
Applies the consensus clustering method introduced by (Lancichinetti & Fortunato, 2012).
The original implementation of this method applies a community detection algorithm repeatedly
to the same network. With stochastic networks, the algorithm is likely to identify different
community solutions with many repeated applications.
}
\details{
The goal of the consensus clustering method is to identify a stable solution across
algorithm applications to derive a "consensus" clustering. The standard or "iterative"
approach is to apply the community detection algorithm \emph{N} times. Then, a co-occurrence
matrix is created representing how often each pair of nodes co-occurred across the
applications. Based on some cut-off value (e.g., 0.30), co-occurrences below this value
are set to zero, forming a "new" sparse network. The procedure proceeds until all nodes
co-occur with all other nodes in their community (or a proportion of 1.00).

Variations of this procedure are also available in this package but are
\strong{experimental}. Use these experimental procedures with caution.
More work is necessary before these experimental procedures are validated

\emph{At this time, seed setting for consensus clustering is not supported}
}
\examples{
# Load data
wmt <- wmt2[,7:24]

# Estimate correlation matrix
correlation.matrix <- auto.correlate(wmt)

# Estimate network
network <- EBICglasso.qgraph(data = wmt)

# Compute standard Louvain with highest modularity approach
community.consensus(
  network,
  consensus.method = "highest_modularity"
)

# Compute standard Louvain with iterative (original) approach
community.consensus(
  network,
  consensus.method = "iterative"
)

# Compute standard Louvain with most common approach
community.consensus(
  network,
  consensus.method = "most_common"
)

# Compute standard Louvain with lowest TEFI approach
community.consensus(
  network,
  consensus.method = "lowest_tefi",
  correlation.matrix = correlation.matrix
)

}
\references{
\strong{Louvain algorithm} \cr
Blondel, V. D., Guillaume, J.-L., Lambiotte, R., & Lefebvre, E. (2008).
Fast unfolding of communities in large networks.
\emph{Journal of Statistical Mechanics: Theory and Experiment}, \emph{2008}(10), P10008.

\strong{Consensus clustering} \cr
Lancichinetti, A., & Fortunato, S. (2012).
Consensus clustering in complex networks.
\emph{Scientific Reports}, \emph{2}(1), 1–7.

\strong{Entropy fit indices} \cr
Golino, H., Moulder, R. G., Shi, D., Christensen, A. P., Garrido, L. E., Nieto, M. D., Nesselroade, J., Sadana, R., Thiyagarajan, J. A., & Boker, S. M. (2020).
Entropy fit indices: New fit measures for assessing the structure and dimensionality of multiple latent variables.
\emph{Multivariate Behavioral Research}.
}
\author{
Hudson Golino <hfg9s at virginia.edu> and Alexander P. Christensen <alexpaulchristensen@gmail.com>
}
