% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fun_as_dyadic.R
\name{as.dyadic}
\alias{as.dyadic}
\title{Extract a \code{Dyadic} object from a numeric matrix}
\usage{
as.dyadic(mat, type, height, breadth)
}
\arguments{
\item{mat}{A dyadic matrix with the classic R matrix representation.}

\item{type}{string, one of the following character strings: \code{horiz},
\code{vert},\code{symm}, and \code{asymm}, which indicates the type of dyadic
object to be extracted;}

\item{height}{The height of the dyadic matrix.}

\item{breadth}{The breadth of the dyadic matrix.}
}
\value{
A \code{Dyadic} object of the input type, height, and breadth
representing the input matrix.
}
\description{
This function extract a \code{Dyadic} object of
given height and breadth from a classic matrix. If the corresponding
sub-matrix extracted is not dyadic, the returned result will be wrong.
}
\details{
This function converts a dyadic matrix of the classic matrix
form into the corresponding \code{Dyadic} object. If the input matrix is
not dyadic it extracts the entries for the dyadic structure of the given
height and breadth that fits to the upper-left hand side corner. Entries
outside the fitted dyadic structure are neglected even if they are not
equal to zero.
}
\examples{
#-------------------------------------------------------------#
#------------Creating vertically dyadic matrices--------------#
#-------------------------------------------------------------#

N <- 4
k <- 3
d <- k * (2^N - 1)

mat1 <- matrix(0, nrow = d, ncol = d)
mat2 <- matrix(0, nrow = d, ncol = d)

for (i in 1:N) {
    st_col_id <- (2^(i - 1) - 1) * k + 1
    en_col_id <- (2^(i - 1) - 1) * k + k
    for (j in 1:2^(N - i)) {
        st_row_id <- st_col_id - (2^(i - 1) - 1) * k
        en_row_id <- en_col_id + (2^(i - 1) - 1) * k
        mat1[st_row_id:en_row_id, st_col_id:en_col_id] <-
            as.matrix(rnorm((2^i - 1) * k^2), ncol = k, nrow = (2^i - 1) * k)
        mat2[st_row_id:en_row_id, st_col_id:en_col_id] <-
            as.matrix(rnorm((2^i - 1) * k^2), ncol = k, nrow = (2^i - 1) * k)
        st_col_id <- st_col_id + 2^i * k
        en_col_id <- en_col_id + 2^i * k
    }
}

mat1
mat2

#-------------------------------------------------------------#
#----------Creating corresponding dyadic objects--------------#
#-------------------------------------------------------------#

V1 <- as.dyadic(mat1, "vert", N, k) # A "vert" dyadic object
V2 <- as.dyadic(mat2, "vert", N, k) # A "vert" dyadic object

mat1S <- t(mat1) \%*\% mat1 # A symmetrically dyadic matrix
mat1AS <- t(mat2) \%*\% mat1 # An asymmetrically dyadic matrix
S <- as.dyadic(mat1S, "symm", N, k) # A "symm" dyadic object
AS <- as.dyadic(mat1AS, "asymm", N, k) # A "asymm" dyadic object

all(as.matrix(S) == mat1S) # Should be TRUE.
all(as.matrix(AS) == mat1AS) # Should be TRUE.


#-------------------------------------------------------------#
#---------------Creating a non-dyadic matrices----------------#
#-------------------------------------------------------------#

mat3 <- diag(d + 5)
mat3[1:d, 1:d] <- mat1

V3 <- as.dyadic(mat3, "vert", N, k) # Extract the upper-left dxd dyadic sub-matrix
all(as.matrix(V3) == mat1) # Should be TRUE.
}
\seealso{
\code{\link{Dyadic-class}} for a description of the class;
}
