% Generated by roxygen2 (4.0.2): do not edit by hand
\name{Deriv}
\alias{Deriv}
\alias{drule}
\title{Symbollic differentiation of an expression or function}
\usage{
Deriv(f, x = if (is.function(f)) names(formals(f)) else NA, env = if
  (is.function(f)) environment(f) else parent.frame())
}
\arguments{
\item{f}{An expression or function to be differentiated.
f can be \itemize{
 \item a user defined function: \code{function(x) x**n}
 \item a string: \code{"x**n"}
 \item an expression: \code{expression(x**n)}
 \item a call: \code{call("^", quote(x), quote(n))}
 \item a language: \code{quote(x**n)}
 \item a right hand side of a formula: \code{~ x**n} or \code{y ~ x**n}
}}

\item{x}{A character string with variable name(s) with resptect to which
\code{f} must be differentiated. If \code{f} is a function \code{x} is
optional andis set to \code{names(formals(f))}. If f is a primitive
function, x is set to c("x" [, "y"[, "z"]]) depending on the maximal
number of allowed arguments in drule table. For functions with
unlimited number of arguments, like sum(), three names are used:
"x", "y" and "z".}

\item{env}{An environment where the symbols and functions are searched for.
Defaults to \code{parent.frame()} for \code{f} expression and to
\code{environment(f)} if \code{f} is a function.}
}
\value{
\itemize{
 \item a function if \code{f} is a function
 \item an expression if \code{f} is an expression
 \item a character string if \code{f} is a character string
 \item a language (usually a so called 'call' but may be also a symbol or just a numeric) for other types of \code{f}
}
}
\description{
Symbollic differentiation of an expression or function
}
\details{
R already contains two differentiation functions: D and deriv.  D does
simple univariate differentiation.  "deriv" uses D do to multivariate
differentiation.  The output of "D" is an expression, whereas the output of
"deriv" is an executable function.
R's existing functions have several limitations.  They can probably be fixed,
but since they are written in C, this would probably require a lot of work.
Limitations include:
\itemize{
 \item The derivatives table can't be modified at runtime, and is only available
in C.
 \item Function cannot substitute function calls.  eg:
	f <- function(x, y) x + y; deriv(~f(x, x^2), "x")
}

So, here are the advantages of this implementation:

\itemize{
 \item It is entirely written in R, so would be easier to maintain.
 \item Can do multi-variate differentiation.
 \item Can differentiate function calls:
 \item It's easy to add custom entries to the derivatives table, e.g.
 \item The output is an executable function, which makes it suitable
for use in optimization problems.
}

Two working environments \code{drule} and \code{simplifications} are
exported in the package namescape.
As their names indicates, they contain tables of derivative and
simplification rules.
To see the list of defined rules do \code{ls(drule)}.
To add your own derivative rule for a function called say \code{sinpi(x)} calculating sin(pi*x), do \code{drule[["sinpi"]] <- list(quote(pi*._d1*cospi(._1)))}.
Here, "._1" stands for the "first arguments", "._d1" for the first derivative of the first arguments. For a function that might have more than one argument,
e.g. log(x, base=exp(1)), the drule entry must be a list with one rule
per number of possible arguments. See \code{drule$log} for an example to follow.
After adding \code{sinpi} you can derivate expressions like \code{Deriv(~ sinpi(x^2), "x")}
}
\examples{
\dontrun{f <- function(x) x^2}
\dontrun{Deriv(f)}
# function (x)
# 2 * x

\dontrun{f <- function(x, y) sin(x) * cos(y)}
\dontrun{Deriv(f)}
# function (x, y)
# c(x = cos(x) * cos(y), y = -(sin(x) * sin(y)))

\dontrun{f_ <- Deriv(f)}
\dontrun{f_(3, 4)}
#              x         y
# [1,] 0.6471023 0.1068000

\dontrun{Deriv(~ f(x, y^2), "y")}
# -(2 * (sin(x) * y * sin(y^2)))

\dontrun{Deriv(quote(f(x, y^2)), c("x", "y"))}
# c(x = cos(x) * cos(y^2), y = -(2 * (sin(x) * y * sin(y^2))))

\dontrun{Deriv(expression(sin(x^2) * y), "x")}
# expression(cos(x^2) * (2 * x) * y)

Deriv("sin(x^2) * y", "x")
"2 * (x * cos(x^2) * y)"
}
\concept{
symbollic differentiation
}

