% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/connectivity_estimation.multinomial.R
\name{d.rel.conn.multinomial.unnorm}
\alias{d.rel.conn.multinomial.unnorm}
\title{Calculates unnormalized probability density for relative connectivity values 
from multiple distinct sites}
\usage{
d.rel.conn.multinomial.unnorm(phis, ps, ks, n.sample, log = FALSE,
  dirichlet.prior.alphas = 1/(length(phis) + 1))
}
\arguments{
\item{phis}{Vector of fractions of individuals (i.e., eggs) from the source 
populations settling at the destination population}

\item{ps}{Vector of fractions of individuals (i.e., eggs) marked in each of 
the source populations}

\item{ks}{Vector of numbers of marked settlers from each source population 
found in the sample}

\item{n.sample}{Vector of total numbers of settlers collected}

\item{log}{Boolean indicating whether or not to return the log probability 
density.  Defaults to \code{FALSE}.}

\item{dirichlet.prior.alphas}{Parameter value for a Dirichlet prior 
distribution for the \code{phis}. Can be a single value for a Dirichlet 
prior with uniform parameters, or a vector of length = 
\code{length(phis)+1}. Defaults to \code{1/(length(phis)+1)}, the value for
the "reference distance" non-informative prior of Berger et al. 2015.}
}
\value{
The unnormalized probability density value.  If \code{log=TRUE}, then
  the logarithm of the probability density value will be returned.
}
\description{
This functions calculates the unnormalized probability density function for 
the relative (to all settlers at the destination site) connectivity value for
larval transport between multiple source sites to a destination site. An 
arbitrary number of source sites can be evaluated.
}
\details{
As this function returns the unnormalized probability density, it must be
normalized somehow to be produce a true probability density.  This can be
acheived using a variety of approaches, including brute force integration of
the unnormalized probability density and MCMC algorithms.
}
\examples{
library(ConnMatTools)

ps <- c(0.7,0.5) # Fraction of eggs "marked" at each source site
ks <- c(4,5) # Number of marked settlers among sample from each source site
n.sample <- 20 # Total sample size.  Must be >= sum(ks)


phis0 = runif(3,min=0.05)
phis0 = phis0 / sum(phis0)
phis0 = phis0[1:2] # Don't include relative connectivity of unknown sites

nbatch=1e4

library(mcmc)
ans = metrop(d.rel.conn.multinomial.unnorm,
             initial=phis0,nbatch=nbatch,scale=0.1,
             log=TRUE,ps=ps,ks=ks,n.sample=n.sample)
# A more serious test would adjust blen and scale to improve results, and would repeat
# multiple times to get results from multiple MCMC chains.

# Plot marginal distribution of relative connectivity from first site
h=hist(ans$batch[,1],xlab="Rel. Conn., Site 1",
       main="Relative Connectivity for Source Site 1")

# For comparison, add on curve that would correspond to single site calculation
phi = seq(0,1,length.out=40)
d1 = d.rel.conn.beta.prior(phi,ps[1],ks[1],n.sample)

lines(phi,d1*nbatch*diff(h$breaks)[1],col="red",lwd=5)

# Image plot of bivariate probability density
t=table(cut(ans$batch[,1],phi),cut(ans$batch[,2],phi))
image(t,col=heat.colors(12)[12:1],xlab="Rel. Conn., Site 1",ylab="Rel. Conn., Site 2")

# Add line indicate region above which one can never find results as that would 
# lead to a total connectivity great than 1
abline(1,-1,col="black",lty="dashed",lwd=3)
}
\author{
David M. Kaplan \email{dmkaplan2000@gmail.com}
}
\references{
Kaplan DM, Cuif M, Fauvelot C, Vigliola L, Nguyen-Huu T, Tiavouane J and Lett C 
  (in press) Uncertainty in empirical estimates of marine larval connectivity. 
  ICES Journal of Marine Science. doi:10.1093/icesjms/fsw182.

Berger JO, Bernardo JM, Sun D (2015) Overall Objective Priors. 
  Bayesian Analysis 10:189-221. doi:10.1214/14-BA915
}
\seealso{
Other connectivity.estimation: \code{\link{d.rel.conn.beta.prior}},
  \code{\link{d.rel.conn.dists.func}},
  \code{\link{d.rel.conn.finite.settlement}},
  \code{\link{d.rel.conn.multiple}},
  \code{\link{d.rel.conn.unif.prior}},
  \code{\link{dual.mark.transmission}},
  \code{\link{optim.rel.conn.dists}},
  \code{\link{r.marked.egg.fraction}}
}

