#' Required seed amount in a tricot project
#' 
#' Calculate the required amount of seeds (or other technology) required 
#' for a triadic comparison of technologies (tricot) project.
#' 
#' @author Kauê de Sousa
#' @param unit optional, a character specifying the metric unit used
#' @param nitems number of items tested in the project
#' @param nseeds an integer for the metric of seeds each bag receives
#' @inheritParams randomise
#' @return a dataframe with required number of seeds
#' @examples
#'   
#' # allocate 0.2 kg of seeds per variety in a project with 500 
#' # participants and 14 varieties
#' seedNeed(npackages = 500,
#'           ncomp = 3,
#'           nitems = 14, 
#'           nseeds = 0.2)
#' 
#' # allocate 100 seedlings per variety in a project with 400 
#' # participants, 8 varieties and 3 comparisons between varieties
#' seedNeed(npackages = 400,
#'           ncomp = 3,
#'           nitems = 9, 
#'           nseeds = 100,
#'           unit = "unit")
#' @export
seedNeed <- function(npackages = 100, 
                     ncomp = 3, nitems = 10, nseeds = 0.15, unit = "kg") {
  
  # number of bags in total 
  nbags <- npackages * ncomp
  
  # number of bags per item
  bagsvar <- ceiling(nbags / nitems)
  
  # refresh nbags
  nbags <- bagsvar * nitems
  
  # number of seeds per variety
  seedsvar <- ceiling(bagsvar * nseeds)
  
  # number of seeds in total
  seedstotal <- ceiling(seedsvar * nitems)
  
  namevar <- c("N bags", "Bags per variety", 
               "Seeds per variety", "Seeds total")
  
  result <- data.frame(var = namevar,
                       quant= c(nbags, bagsvar, seedsvar, seedstotal),
                       unit = c(rep("unit",2), rep(unit, 2)),
                       stringsAsFactors = FALSE)
  
  class(result) <- union("CM_df", class(result))
  
  return(result)
  
  
}


