% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/set_priors.R
\name{make_priors}
\alias{make_priors}
\title{Make Priors}
\usage{
make_priors(
  model,
  alphas = NA,
  distribution = NA,
  node = NA,
  label = NA,
  statement = NA,
  confound = NA,
  nodal_type = NA,
  param_names = NA,
  param_set = NA
)
}
\arguments{
\item{model}{A model created with \code{make_model}}

\item{alphas}{Real positive numbers giving hyperparameters of the Dirichlet distribution}

\item{distribution}{String (or list of strings) indicating a common prior distribution (uniform, jeffreys or certainty)}

\item{node}{A string (or list of strings) indicating nodes for which priors are to be altered}

\item{label}{A string. Label for nodal type indicating nodal types for which priors are to be altered}

\item{statement}{A causal query (or list of queries) that determines nodal types for which priors are to be altered}

\item{confound}{A confound named list that restricts nodal types for which priors are to be altered. Adjustments are limited to nodes in the named list.}

\item{nodal_type}{A string. Label for nodal type indicating nodal types for which priors are to be altered}

\item{param_names}{A string. The name of specific parameter in the form of, for example, 'X.1', 'Y.01'}

\item{param_set}{A string. Indicates the name of the set of parameters to be modified (useful when setting confounds)}
}
\value{
A vector indicating the hyperparameters of the prior distribution of the nodal types.

For instance \code{confound = list(X  = Y[X=1]> Y[X=0])} adjust parameters on X that are conditional on nodal types for Y.
}
\description{
A flexible function to generate priors for a model.
}
\details{
Seven arguments govern *which* parameters should be altered. The default is 'all' but this can be reduced by specifying

* \code{label} or \code{nodal_type} The label of a particular nodal type, written either in the form Y0000 or Y.Y0000

* \code{node}, which restricts for example to parameters associated with node 'X'

* \code{statement}, which restricts for example to nodal types that satisfy the statement 'Y[X=1] > Y[X=0]'

* \code{confound}, which restricts for example to nodal types that satisfy the statement 'Y[X=1] > Y[X=0]'

* \code{param_set}, which us useful when setting confound statements that produces several sets of parameters

* \code{param_names}, which restricts in specific parameters by naming them

Two arguments govern what values to apply:

* \code{alphas} is one or more non negative numbers and

* \code{distribution} indicates one of a common class: uniform, jeffreys, or 'certain'

Any arguments entered as lists or vectors of size > 1 should be of the same length as each other.
}
\examples{

# Pass all nodal types
model <- make_model("Y <- X")
make_priors(model, alphas = .4)
make_priors(model, distribution = "jeffreys")

# Passing by names of node, parameter set or label
model <- make_model('X -> M -> Y')
make_priors(model, param_name = "X.1", alphas = 2)
make_priors(model, node = 'X', alphas = 3)
make_priors(model, param_set = 'Y', alphas = 5)
make_priors(model, node = c('X', 'Y'), alphas = 3)
make_priors(model, param_set = c('X', 'Y'), alphas = 5)
make_priors(model, node = list('X', 'Y'), alphas = list(3, 6))
make_priors(model, param_set = list('X', 'Y'), alphas = list(4, 6))
make_priors(model, node = c('X', 'Y'), distribution = c('certainty', 'jeffreys'))
make_priors(model, param_set = c('X', 'Y'), distribution = c('jeffreys', 'certainty'))
make_priors(model, label = '01', alphas = 5)
make_priors(model, node = 'Y', label = '00', alphas = 2)
make_priors(model, node =c('M', 'Y'), label = '11', alphas = 4)

# Passing a causal statement
make_priors(model, statement = 'Y[M=1] > Y[M=0]', alphas = 3)
make_priors(model, statement = c('Y[M=1] > Y[M=0]', 'M[X=1]== M[X=0]'), alphas = c(3, 2))

# Passing a confound statement
model <- make_model('X->Y') \%>\%
 set_confound(list(X = 'Y[X=1] > Y[X=0]', X = 'Y[X=1] < Y[X=0]'))

make_priors(model,
            confound = list(X='Y[X=1] > Y[X=0]',
                            X='Y[X=1] < Y[X=0]'),
            alphas = c(3, 6))

make_priors(model, confound= list(X='Y[X=1] > Y[X=0]'), alphas = 4)
make_priors(model, param_set='X_1', alphas = 5)
make_priors(model, param_names='X_2.1', alphas = .75)

make_model('X -> Y') \%>\%
  set_confound(list(X = 'Y[X=1]>Y[X=0]'))\%>\%
  make_priors(statement = 'X[]==1',
              confound = list(X = 'Y[X=1]>Y[X=0]', X = 'Y[X=1]<Y[X=0]'),
              alphas = c(2, .5))
}
\seealso{
Other priors: 
\code{\link{get_priors}()},
\code{\link{make_par_values_multiple}()},
\code{\link{make_par_values}()},
\code{\link{make_values_task_list}()},
\code{\link{set_priors}()}
}
\concept{priors}
