% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ProxGrad.R
\name{ProxGrad}
\alias{ProxGrad}
\title{Generalized Linear Models via Proximal Gradients}
\usage{
ProxGrad(
  x,
  y,
  glm_type = c("Linear", "Logistic", "Gamma", "Poisson")[1],
  include_intercept = TRUE,
  alpha_s = 3/4,
  lambda_sparsity,
  acceleration = FALSE,
  tolerance = 1e-05,
  max_iter = 1e+05
)
}
\arguments{
\item{x}{Design matrix.}

\item{y}{Response vector.}

\item{glm_type}{Description of the error distribution and link function to be used for the model. Must be one of "Linear", 
"Logistic", "Gamma" or "Poisson". Default is "Linear".}

\item{include_intercept}{Argument to determine whether there is an intercept. Default is TRUE.}

\item{alpha_s}{Elastic net mixing parmeter. Default is 3/4.}

\item{lambda_sparsity}{Sparsity tuning parameter value.}

\item{acceleration}{Argument to determine whether a gradient acceleration method is used. Default is FALSE.}

\item{tolerance}{Convergence criteria for the coefficients. Default is 1e-3.}

\item{max_iter}{Maximum number of iterations in the algorithm. Default is 1e5.}
}
\value{
An object of class ProxGrad.
}
\description{
\code{ProxGrad} computes the coefficients for generalized linear models using accelerated proximal gradients.
}
\examples{
\donttest{
# Data simulation
set.seed(1)
n <- 50
N <- 2000
p <- 1000
beta.active <- c(abs(runif(p, 0, 1/2))*(-1)^rbinom(p, 1, 0.3))
# Parameters
p.active <- 100
beta <- c(beta.active[1:p.active], rep(0, p-p.active))
Sigma <- matrix(0, p, p)
Sigma[1:p.active, 1:p.active] <- 0.5
diag(Sigma) <- 1

# Train data
x.train <- mvnfast::rmvn(n, mu = rep(0, p), sigma = Sigma) 
prob.train <- exp(x.train \%*\% beta)/
              (1+exp(x.train \%*\% beta))
y.train <- rbinom(n, 1, prob.train)
# Test data
x.test <- mvnfast::rmvn(N, mu = rep(0, p), sigma = Sigma)
prob.test <- exp(x.test \%*\% beta)/
             (1+exp(x.test \%*\% beta))
y.test <- rbinom(N, 1, prob.test)

# ProxGrad - Single Group
proxgrad.out <- ProxGrad(x.train, y.train,
                         glm_type = "Logistic",
                         include_intercept = TRUE,
                         alpha_s = 3/4,
                         lambda_sparsity = 0.01, 
                         acceleration = TRUE,
                         tolerance = 1e-5, max_iter = 1e5)

# Predictions
proxgrad.prob <- predict(proxgrad.out, newx = x.test, type = "prob")
proxgrad.class <- predict(proxgrad.out, newx = x.test, type = "class")
plot(prob.test, proxgrad.prob, pch = 20)
abline(h = 0.5,v = 0.5)
mean((prob.test-proxgrad.prob)^2)
mean(abs(y.test-proxgrad.class))

}

}
\seealso{
\code{\link{coef.ProxGrad}}, \code{\link{predict.ProxGrad}}
}
\author{
Anthony-Alexander Christidis, \email{anthony.christidis@stat.ubc.ca}
}
