% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/optimCEGO.R
\name{optimCEGO}
\alias{optimCEGO}
\title{Combinatorial Efficient Global Optimization}
\usage{
optimCEGO(x = NULL, fun, control = list())
}
\arguments{
\item{x}{Optional initial design as a list. If NULL (default), \code{creationFunction} (in \code{control} list) is used to create initial design. 
If \code{x} has less individuals than specified by \code{control$evalInit}, \code{creationFunction} will fill up the design.}

\item{fun}{target function to be minimized}

\item{control}{(list), with the options of optimization and model building approaches employed:
\describe{
\item{\code{evalInit}}{ Number of initial evaluations (i.e., size of the initial design), integer, default is \code{2}}
\item{\code{vectorized}}{ Boolean. Defines whether target function is vectorized (takes a list of solutions as argument) or not (takes single solution as argument). Default: FALSE}
\item{\code{verbosity}}{ Level of text output during run. Defaults to 0, no output.}
\item{\code{plotting}}{ Plot optimization progress during run (TRUE) or not (FALSE). Default is FALSE.}
\item{\code{targetY}}{ optimal value to be found, stopping criterion, default is \code{-Inf}}
\item{\code{evalBudget}}{ maximum number of target function evaluations, default is \code{100}}
\item{\code{creationRetries}}{ When a model does not predict an actually improving solution, a random exploration step is performed. \code{creationRetries} solutions are created randomly. 
        For each, distance to all known solutions is calculated. The minimum distance is recorded for each random solution. 
        The random solution with maximal minimum distance is chosen doe be evaluated in the next iteration.}
\item{\code{model}}{ Model to be used as a surrogate of the target function. Default is "K" (Kriging). Also
    available are: "LM" (linear, distance-based model), "RBFN" Radial Basis Function Network.}
\item{\code{modelSettings}}{ List of settings for \code{model} building, passed on as the \code{control} argument to the model training functions \code{\link{modelKriging}}, \code{\link{modelLinear}}, \code{\link{modelRBFN}}.}
\item{\code{infill}}{ This parameter specifies a function to be used for the infill criterion (e.g., the default is expected improvement \code{infillExpectedImprovement}).
To use no specific infill criterion this has to be set to \code{NA}, in which case the prediction of the surrogate model is used. Infill criteria are only used with models that may provide some error estimate with predictions.}
\item{\code{optimizer}}{ Optimizer that finds the minimum of the surrogate model. Default is \code{\link{optimEA}}, an Evolutionary Algorithm.}
\item{\code{optimizerSettings}}{ List of settings (\code{control}) for the \code{optimizer} function.}
\item{\code{initialDesign}}{ Design function that generates the initial design. Default is \code{designMaxMinDist}, which creates a design that maximizes the minimum distance between points.}
\item{\code{initialDesignSettings}}{ List of settings (\code{control}) for the \code{initialDesign} function.}
\item{\code{creationFunction}}{ Function to create individuals/solutions in search space. Default is a function that creates random permutations of length 6}
\item{\code{distanceFunction}}{ distanceFunction a suitable distance function of type f(x1,x2), returning a scalar distance value, preferably between 0 and 1.
     Maximum distances larger 1 are not a problem, but may yield scaling bias when different measures are compared.
        Should be non-negative and symmetric. With the setting \code{control$model="K"} this can also be a list of different fitness functions.
   Default is Hamming distance for permutations: distancePermutationHamming.}
}}
}
\value{
a list:
\describe{
\item{\code{xbest}}{ best solution found}
\item{\code{ybest}}{ fitness of the best solution}
\item{\code{x}}{ history of all evaluated solutions}
\item{\code{y}}{ corresponding target function values f(x)}
\item{\code{fit}}{ model-fit created in the last iteration}
\item{\code{fpred}}{ prediction function created in the last iteration}
\item{\code{count}}{ number of performed target function evaluations}
\item{\code{message}}{ message string, giving information on termination reason}
\item{\code{convergence}}{ error/status code: \code{-1} for termination due 
to failed model building, \code{0} for termination due to depleted budget, 
\code{1} if attained objective value is equal to or below target (\code{control$targetY})}
}
}
\description{
Model-based optimization for combinatorial or mixed problems. Based on measures of distance or dissimilarity.
}
\examples{
seed <- 0
#distance
dF <- distancePermutationHamming
#mutation
mF <- mutationPermutationSwap
#recombination
rF <-  recombinationPermutationCycleCrossover 
#creation
cF <- function()sample(5)
#objective function
lF <- landscapeGeneratorUNI(1:5,dF)
#start optimization
set.seed(seed)
res1 <- optimCEGO(,lF,list(
			creationFunction=cF,
			distanceFunction=dF,
			optimizerSettings=list(budget=100,popsize=10,
			mutationFunction=mF,recombinationFunction=rF),
	evalInit=5,budget=15,targetY=0,verbosity=1,model=modelKriging,
	vectorized=TRUE)) ##target function is "vectorized", expects list as input
set.seed(seed)
res2 <- optimCEGO(,lF,list(
			creationFunction=cF,
			distanceFunction=dF,
			optimizerSettings=list(budget=100,popsize=10,
			mutationFunction=mF,recombinationFunction=rF),
			evalInit=5,budget=15,targetY=0,verbosity=1,model=modelRBFN,
	vectorized=TRUE)) ##target function is "vectorized", expects list as input
res1$xbest 
res2$xbest 

}
\references{
Zaefferer, Martin; Stork, Joerg; Friese, Martina; Fischbach, Andreas; Naujoks, Boris; Bartz-Beielstein, Thomas. (2014). Efficient global optimization for combinatorial problems. In Proceedings of the 2014 conference on Genetic and evolutionary computation (GECCO '14). ACM, New York, NY, USA, 871-878. DOI=10.1145/2576768.2598282 http://doi.acm.org/10.1145/2576768.2598282

Zaefferer, Martin; Stork, Joerg; Bartz-Beielstein, Thomas. (2014). Distance Measures for Permutations in Combinatorial Efficient Global Optimization. In Parallel Problem Solving from Nature - PPSN XIII (p. 373-383). Springer International Publishing.
}
\seealso{
\code{\link{modelKriging}}, \code{\link{modelLinear}}, \code{\link{modelRBFN}}, \code{\link{buildModel}}, \code{\link{optimEA}}
}
