% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/glm_fixed.R
\name{glm.fixed.a0}
\alias{glm.fixed.a0}
\title{Model fitting for generalized linear models with fixed a0}
\usage{
glm.fixed.a0(
  data.type,
  data.link,
  y = 0,
  n = 1,
  x = matrix(),
  borrow.treat = FALSE,
  historical = list(),
  lower.limits = rep(-100, 50),
  upper.limits = rep(100, 50),
  slice.widths = rep(1, 50),
  nMC = 10000,
  nBI = 250,
  current.data = TRUE,
  prior.beta.var = rep(10, 50)
)
}
\arguments{
\item{data.type}{Character string specifying the type of response. The options are "Normal", "Bernoulli", "Binomial", "Poisson" and "Exponential".}

\item{data.link}{Character string specifying the link function. The options are "Logistic", "Probit", "Log", "Identity-Positive", "Identity-Probability" and "Complementary Log-Log". Does not apply if \code{data.type} is "Normal".}

\item{y}{Vector of responses.}

\item{n}{(For binomial data only) vector of integers specifying the number of subjects who have a particular value of the covariate vector. If the data is binary and all covariates are discrete, collapsing Bernoulli data into a binomial structure can make the slice sampler much faster.}

\item{x}{Matrix of covariates. The first column should be the treatment indicator with 1 indicating treatment group. The number of rows should equal the length of the response vector \code{y}.}

\item{borrow.treat}{Logical value indicating whether the historical information is used to inform the treatment effect parameter. The default value is FALSE. If TRUE, the first column of the historical covariate matrix must be the treatment indicator.
If FALSE, the historical covariate matrix must NOT have the treatment indicator, since the historical data is assumed to be from the control group only.}

\item{historical}{(Optional) list of historical dataset(s). East historical dataset is stored in a list which contains three \emph{named} elements: \code{y0}, \code{x0} and \code{a0}.
\itemize{
\item \code{y0} is a vector of responses.
\item \code{x0} is a matrix of covariates. If \code{borrow.treat} is FALSE (the default), \code{x0} should NOT have the treatment indicator. Apart from missing the treatment indicator, \code{x0} should have the same set of covariates in the same order as \code{x}.
If \code{borrow.treat} is TRUE, \code{x0} should have the same set of covariates in the same order as \code{x}, where the first column of \code{x0} must be the treatment indicator.
\item \code{a0} is a number between 0 and 1 indicating the discounting parameter value for that historical dataset.
}}

\item{lower.limits}{Vector of lower limits for parameters to be used by the slice sampler. The length of the vector should be equal to the total number of parameters, i.e. P+1 where P is the number of covariates. The default is -100 for all parameters (may not be appropriate for all situations). Does not apply if \code{data.type} is "Normal".}

\item{upper.limits}{Vector of upper limits for parameters to be used by the slice sampler. The length of the vector should be equal to the total number of parameters, i.e. P+1 where P is the number of covariates. The default is 100 for all parameters (may not be appropriate for all situations). Does not apply if \code{data.type} is "Normal".}

\item{slice.widths}{Vector of initial slice widths for parameters to be used by the slice sampler. The length of the vector should be equal to the total number of parameters, i.e. P+1 where P is the number of covariates. The default is 1 for all parameter (may not be appropriate for all situations). Does not apply if \code{data.type} is "Normal".}

\item{nMC}{Number of iterations (excluding burn-in samples) for the slice sampler or Gibbs sampler. The default is 10,000.}

\item{nBI}{Number of burn-in samples for the slice sampler or Gibbs sampler. The default is 250.}

\item{current.data}{Logical value indicating whether current data is included. The default is TRUE. If FALSE, only historical data is included in the analysis,
and the posterior samples can be used as a discrete approximation to the sampling prior in \code{\link{power.glm.fixed.a0}}.}

\item{prior.beta.var}{Only applies if current.data = FALSE. If no current data is provided, the initial priors used for \eqn{\beta} are i.i.d. normal distributions with mean zero and variance equal to \code{prior.beta.var}.
The length of the vector should be equal to the length of \eqn{\beta}. The default variance is 10.}
}
\value{
The function returns a S3 object with a \code{summary} method. If \code{data.type} is "Normal", posterior samples of \eqn{\beta}, \eqn{\tau} and \eqn{\tau_k}'s (if historical data is given) are returned.
For all other data types, a matrix of posterior samples of \eqn{\beta} is returned. The first column contains posterior samples of the intercept.
The second column contains posterior samples of \eqn{\beta_1}, the parameter for the treatment indicator.
}
\description{
Model fitting using power priors for generalized linear models with fixed \eqn{a_0}
}
\details{
If \code{data.type} is "Normal", the response \eqn{y_i} is assumed to follow \eqn{N(x_i'\beta, \tau^{-1})} where \eqn{x_i} is the vector of covariates for subject \eqn{i}.
Each historical dataset \eqn{D_{0k}} is assumed to have a different precision parameter \eqn{\tau_k}.
The initial prior for \eqn{\tau} is the Jeffery's prior, \eqn{\tau^{-1}}, and the initial prior for \eqn{\tau_k} is \eqn{\tau_k^{-1}}.
The initial prior for \eqn{\beta} is the uniform improper prior. Posterior samples are obtained through Gibbs sampling.

For all other data types, posterior samples are obtained through slice sampling.
The default lower limits for the parameters are -100. The default upper limits
for the parameters are 100. The default slice widths for the parameters are 1.
The defaults may not be appropriate for all situations, and the user can specify the appropriate limits
and slice width for each parameter.

When \code{current.data} is set to FALSE, only historical data is included in the analysis,
and the posterior samples can be used as a discrete approximation to the sampling prior in \code{\link{power.glm.fixed.a0}}.
}
\examples{
data.type <- "Bernoulli"
data.link <- "Logistic"

# Simulate current data
set.seed(1)
p <- 3
n_total <- 100
y <- rbinom(n_total,size=1,prob=0.6)
# The first column of x is the treatment indicator.
x <- cbind(rbinom(n_total,size=1,prob=0.5),
           matrix(rnorm(p*n_total),ncol=p,nrow=n_total))

# Simulate two historical datasets
# Note that x0 does not have the treatment indicator
historical <- list(list(y0=rbinom(n_total,size=1,prob=0.2),
                        x0=matrix(rnorm(p*n_total),ncol=p,nrow=n_total), a0=0.2),
                   list(y0=rbinom(n_total, size=1, prob=0.5),
                        x0=matrix(rnorm(p*n_total),ncol=p,nrow=n_total), a0=0.3))

# Set parameters of the slice sampler
lower.limits <- rep(-100, 5) # The dimension is the number of columns of x plus 1 (intercept)
upper.limits <- rep(100, 5)
slice.widths <- rep(1, 5)

nMC <- 1000 # nMC should be larger in practice
nBI <- 250
result <- glm.fixed.a0(data.type=data.type, data.link=data.link, y=y, x=x, historical=historical,
                       lower.limits=lower.limits, upper.limits=upper.limits,
                       slice.widths=slice.widths, nMC=nMC, nBI=nBI)

summary(result)

}
\references{
Neal, Radford M. Slice sampling. Ann. Statist. 31 (2003), no. 3, 705--767.
}
\seealso{
\code{\link{power.glm.fixed.a0}}
}
