#' Scatterplots of samples
#'
#' This fonction compute a matrix of scatterplots of at most 8 samples.
#'
#' @param Sampling MCMC.list: that is generated by \code{\link{jags.model}} in \code{\link{AgeS_Computation}},
#' \code{\link{AgeC14_Computation}} or \code{\link{Age_OSLC14}}.
#' @param SampleNames character vector: names of samples.
#' @param SelectedSamples numeric vector: indicate if the sample is selected to be plot with the other samples.
#' If the sample with names \code{SampleNames[i]} is selecteted, then \code{SelectedSamples[i]=1}; if not \code{SelectedSamples=0}.
#' \code{length(SelectedSamples)}=\code{length(SampleNames)}.
#' At least two samples must be selected.
#' @param sc numeric (with default): to define the number of values on the axis.
#' @param Nb_chaines integer (with default): number of independent chains used in \code{\link{AgeS_Computation}}
#' to generate the sampling (for more information see \code{\link{jags.model}}).
#'
#' @return
#' A matrix that contains at row \code{i} and column \code{j} the scatterplot of the posterior distribution of
#' the sample ID \code{i} plotted against the one of the sample ID \code{j}
#'
#' @author Claire Christophe, Anne Philippe, Guillaume Guerin
#'
#' @seealso \code{\link{Age_Computation}}, \code{\link{AgeS_Computation}}, \code{\link{AgeC14_Computation}},
#' and \code{\link{rjags}} packages.
#'
#' @examples
#' data(AgeS,envir = environment())
#' ScatterSamples(Sampling=AgeS$Sampling,SampleNames=c("GDB5","GDB3"),SelectedSamples=c(1,1))
#'
#' @export
ScatterSamples<-function(Sampling,SampleNames,SelectedSamples,sc=3,Nb_chaines=3){
  ind_sample=which(SelectedSamples==1)
  N <- length(ind_sample)
  if(N>9){
    stop("[ScatterSamples()] Too many samples have been selected. Please select less than 8 samples!",call. = FALSE)
  }
  sample=Sampling[[1]]
  for(i in 2:Nb_chaines){
    sample=rbind(sample,Sampling[[i]])
  }
  Names <- paste("A",ind_sample,sep="")

  ##plot
  hexbin::hexplom(
    sample[, ind_sample],
    upper.panel = NULL,
    xlab = "Age (ka)",
    pscales = sc,
    colramp = function(n)
      terrain.colors(n),
    varnames = Names,
    cex.axis = 0.2
  )

}

