#' Sets default values for constants  used by the optimization functions
#' in FORTRAN
#'
#' @title Default control list
#'
#' @param control a list with configurations to be passed to the
#' optimization subroutines. Missing arguments will receive default values.
#' See \sQuote{Details}.
#'
#' @details  The \code{control} argument is a list that can supply any of the
#' following components:
#'
#'  \describe{
#'
#'   \item{\code{method}}{The optimization method. Current available options
#'   are \code{"L-BFGS-B"} and \code{"Nelder-Mead"}. Default is \code{"L-BFGS-B"}.}
#'
#'   \item{\code{maxit}}{The maximum number of iterations. Defaults to \code{1000}.}
#'
#'   \item{\code{iprint}}{The frequency of reports if \code{control$trace}
#'   is positive. Defaults is -1 (no report).
#'   \itemize{
#' 		  \item For \code{"L-BFGS-B"} method:
#'
#' 		      iprint<0    no output is generated;
#'
#' 		      iprint=0    print only one line at the last iteration;
#'
#' 		      0<iprint<99 print also f and |proj g| every iprint iterations;
#'
#' 		      iprint=99   print details of every iteration except n-vectors;
#'
#' 		      iprint=100  print also the changes of active set and final x;
#'
#' 		      iprint>100  print details of every iteration including x and g;
#'
#' 		  \item For \code{"Nelder-Mead"} method:
#'
#' 		      iprint<0 No printing
#'
#' 		     iprint=0 Printing of parameter values and the function
#'  		         		Value after initial evidence of convergence.
#'
#' 		      iprint>0 As for iprint = 0 plus progress reports after every
#'  							   Iprint evaluations, plus printing for the initial simplex.
#'  }}
#'
#'
#'   \item{\code{factr}}{controls the convergence of the \code{"L-BFGS-B"}
#'   method. Convergence occurs when the reduction in the objective is
#'   within this factor of the machine tolerance. The iteration will stop when
#'
#'             \deqn{(f^k - f^{k+1})/max\{|f^k|,|f^{k+1}|,1\} \le factr*epsmch}
#'
#'   where epsmch is the machine precision, which is automatically
#'   generated by the code. Typical values
#'   for \code{factr}: 1.e+12 for low accuracy; 1.e+7 for moderate accuracy;
#'   1.e+1 for extremely high accuracy. Default is \code{1e7}, that is a
#'   tolerance of about \code{1e-8}.}
#'
#'   \item{\code{pgtol}}{helps control the convergence of the \code{"L-BFGS-B"}
#'   method. It is a tolerance on the projected gradient in the current
#'   search direction. the iteration will stop when
#'
#'   \deqn{max\{|proj g_i |, i = 1, ..., n\} \le pgtol}
#'
#'   where \eqn{pg_i} is the ith component of the projected gradient.
#'   Default is \code{1e-12}.}
#'
#'   \item{\code{stopcr}}{The criterion applied to the standard deviation of
#'   the values of objective function at the points of the simplex, for
#'   "Nelder-Mead" method.}
#' }
#'
#' @return a list with all arguments in \sQuote{Details}.
#'
#' @examples
#' BTSR::fit.control()
#'
#' @export
fit.control <- function(control = list()){

  con <- list(method = "L-BFGS-B",
              maxit = 1000,
              iprint = -1,
              factr = 1e+7,
              pgtol = 1e-12,
              stopcr = 1e-4)

  con[names(control)] <- control

  return(con)
}




#' This function calculates initial values for the parameter vector,
#' to pass to the optimization function.
#'
#' @title Initial values for optimization
#'
#' @param model character; The model to be fitted to the data. One of
#' \code{"BARFIMA"}, \code{"KARFIMA"}, \code{"GARFIMA"}, \code{"BARC"}.
#' Default is "Generic" so that no specific structure is assumed.
#'
#' @param yt a univariate time series. Missing values (NA's)
#' are not  allowed.
#'
#' @param linkg character; one of  \code{"linear"}, \code{"logit"},
##' \code{"log"}, \code{"loglog"}, \code{"cloglog"}.
##' If only one name is provided, the same link will be used for the conditional mean, that is
##' to define \eqn{g(\mu)} and for the observed time series in the AR part
##' of the model, that is, \eqn{g(y[t])}.
##'
#' @param xreg optional; a vector or matrix of external regressors,
#' which must have the same number of rows as x.
#'
#' @param p an integer; the AR order. Default is zero.
#'
#' @param q an integer; for \code{BARC} models represents the dimension of
#' the parameter associated to the map \eqn{T}. For other models is the
#' MA order. Default is zero.
#'
#' @param d logical; if FALSE, \eqn{d} is fixed as zero. Default is TRUE.
#'
#' @param y.start optional; an initialization value for \eqn{y[t]},
#' for \eqn{t \le 0}, to be used in the AR recursion. If not provided,
#' the default assume \eqn{y[t] = 0}, when using a \code{"linear"} link for
#'  \eqn{yt}, and \eqn{g(y[t]) = 0}, otherwise.
#'
#' @param y.lower lower limit for the distribution support.
#' Default is \code{-Inf}.
#'
#' @param y.upper upper limit for the distribution support.
#' Default is \code{Inf}.
#'
#' @param lags optional; a list with the components \code{beta},
#' \code{phi} and \code{theta} specifying which lags must be included
#' in the model. An empty list or missing component indicates that, based on the values \code{nreg},
#' \code{p}  e \code{q}), all lags must be includes in the model.
#'
#' @param fixed.values optional; a list with the fixed values for
#' each component, if any. If fixed values are provided, either \code{lags}
#' or \code{fixed.lags} must also be provided.
#'
#' @param fixed.lags optional; a list with the components \code{beta},
#' \code{phi} and \code{theta} specifying which lags must be fixed.
#' An empty list implies that fixed values will be set based on
#' \code{lags}.
#'
#' @return a list with starting values for the parameters of the selected
#' model. Possible outputs are:
#'
#'  \item{alpha}{the intercept}
#'  \item{beta}{the coefficients for the regressors}
#'  \item{phi}{the AR coefficients}
#'  \item{theta}{for BARC models, the map parameter.  For any other model,
#'  the MA coefficients}
#'  \item{d}{the long memory parameter}
#'  \item{nu}{the precison parameter}
#'
#' @importFrom  stats lm.fit fitted residuals
#'
#' @export
#'
#' @examples
#' mu = 0.5
#' nu = 20
#'
#' yt = rbeta(100, shape1 = mu*nu, shape2 = (1-mu)*nu)
#' coefs.start(model = "BARFIMA", yt = yt,
#'             linkg = "linear", d = FALSE,
#'             y.lower = 0, y.upper = 1)
#'
#' yt = rgamma(100, shape = nu, rate = mu*nu)
#' coefs.start(model = "GARFIMA", yt = yt,
#'             linkg = "linear", d = FALSE,
#'             y.lower = 0, y.upper = Inf)
#'
coefs.start <- function(model = "Generic",
                        yt, linkg = c("linear","linear"), xreg = NULL,
                        p = 0,  q = 0, d = TRUE, y.start = NULL,
                        y.lower = -Inf, y.upper = Inf,
                        lags = list(), fixed.values = list(),
                        fixed.lags = list()){

  if(is.null(y.lower)) y.lower = -Inf
  if(is.null(y.upper)) y.upper = Inf
  if(y.lower == -Inf) y.lower = .Machine$double.xmin
  if(y.upper == Inf) y.upper = .Machine$double.xmax

  if(length(linkg) == 1) linkg = c(linkg, linkg)
  ##-------------------
  ## link function
  ##-------------------
  linktemp1 <- link.btsr(link = linkg[1])
  linkfun1 <- linktemp1$linkfun
  g1y <- linkfun1(yt, ctt.ll = 1, y.lower = y.lower, y.upper = y.upper)
  if(linkg[2] == linkg[1]){
    g2y <- g1y
    linkfun2 <- linkfun1
  }
  else{
    linktemp2 <- link.btsr(link = linkg[2])
    linkfun2 <- linktemp2$linkfun
    g2y <- linkfun2(yt, ctt.ll = 1, y.lower = y.lower, y.upper = y.upper)
  }

  if(p > 0){
    if(is.null(y.start)) gystart <- NA
    else  gystart <-  linkfun2(y.start, ctt.ll = 1,
                               y.lower = y.lower, y.upper = y.upper)
  }

  n <- length(g1y)
  if(is.null(xreg)) nreg <- 0
  else nreg <- ncol(as.matrix(xreg))

  ##----------------------------------------------------
  ## starting values for alpha, phi and beta
  ##----------------------------------------------------
  X <- matrix(1, nrow = n)
  nreg1 <- nreg
  if(nreg > 0){
    lag <- 1:nreg
    if(!is.null(lags$beta)) lag = lags$beta
    else{
      if(!is.null(fixed.lags$beta)){
        fl <- fixed.lags$beta
        lag <- lag[-fl]
      }}
    if(length(lag) > 0) X <- cbind(X, as.matrix(xreg)[,lag])
  }
  p1 <- p
  if(p > 0){
    lag <- 1:p
    if(!is.null(lags$phi)) lag = lags$phi
    else{
      if(!is.null(fixed.lags$phi)){
        fl <- fixed.lags$phi
        lag <- lag[-fl]
      }}
    p1 <- length(lag)
    if(p1 > 0){
      P <- matrix(gystart, nrow = n, ncol = p1)
      for(i in 1:p1) P[-c(1:lag[i]), i] <- g2y[1:(n-lag[i])]
      X <- cbind(X,P)
    }
  }
  w <- sum(is.na(X[,ncol(X)]))
  if(w > 0){
    X <- X[-c(1:w), ]
    g1y <- g1y[-c(1:w)]
  }

  fit <- lm.fit(x = X, y = g1y)
  mqo <- c(fit$coefficients, use.names = FALSE)
  mqo[is.na(mqo)] <- 0
  k <- length(mqo)

  ##--------------------------------------
  ## initializing the parameter values
  ##--------------------------------------
  alpha <- NULL
  a <- as.integer(is.null(fixed.values$alpha))
  if(a == 1) alpha <- mqo[1]
  else{mqo = mqo[-1]; k = k-1}
  beta <- NULL
  if(nreg1 > 0) beta <-  mqo[(a+1):(a+nreg1)]
  phi <- NULL
  if(p1 > 0) phi <- mqo[(a+nreg1+1):k]
  theta <- NULL
  q1 <-  max(q - max(length(fixed.values$theta),length(fixed.lags$theta)),
             length(lags$theta))
  if(q1 > 0) theta <- rep(0, q1)  # for BARC models this will be fixed in the main program
  dd <- NULL
  if(d == TRUE){
    if(is.null(fixed.values$d)) dd <- 0.01
  }
  nu <- NULL
  if(is.null(fixed.values$nu)){
    n1 <- length(g1y)
    mu <- fitted(fit)
    mu <- linktemp1$linkinv(mu, ctt.ll = 1, y.lower = y.lower,
                            y.upper = y.upper)
    dlink <- linktemp1$diflink(mu, ctt.ll = 1, y.lower = y.lower,
                               y.upper = y.upper)
    er <- residuals(fit)
    sigma2 <- sum(er^2)/((n1 - k) * (dlink)^2)
    nu.type <- switch (EXPR = model[1],
                       BARFIMA = 1,
                       BARC = 1,
                       GARFIMA = 2,
                       KARFIMA = 3,
											 UWARFIMA = 3)
    if(nu.type == 1) nu = mean(mu * (1 - mu)/sigma2) - 1
    if(nu.type == 2) nu = mean(mu^2/sigma2)
    if(nu.type == 3) nu = 5
  }
  par <- list(alpha = alpha, beta = beta, phi = phi,
              theta = theta, d = dd, nu = nu)
  return(par)
}




##-------------------------------------------------------------------------
## internal function.
## Convert the coefficients and its bounds to the correct format to pass
## to FORTRAN
##-------------------------------------------------------------------------
.coefs.fit.config <- function(model = "Generic",
                              coefs = list(),lags = list(),
                              fixed.values = list(), fixed.lags = list(),
                              lower = list(), upper = list(),
                              p = 0, q = 0, nreg = 0){

  ##---------------------------------------------------
  ##  checking for fixed and initial values.
  ##---------------------------------------------------
  out <- .coefs.convert.all(model = model, coefs = coefs, lags = lags,
                            fixed.values = fixed.values,fixed.lags = fixed.lags,
                            p = p, q = q, nreg = nreg)

  ##---------------------------------------------------
  ##  setting the bounds
  ##---------------------------------------------------
  lwr <- NULL
  upr <- NULL
  nbd <- NULL

  ## alpha
  if(out$alpha$nfix == 0){
    cb <- .bounds.convert(npar = 1, lower = lower$alpha, upper = upper$alpha)
    lwr <- c(lwr, alpha = cb$lower)
    upr <- c(upr, alpha = cb$upper)
    nbd <- c(nbd, alpha = cb$nbd)
  }
  ## beta
  npar <- nreg - out$beta$nfix
  if(npar > 0){
    cb <- .bounds.convert(npar = npar, lower = lower$beta, upper = upper$beta)
    lwr <- c(lwr, beta = cb$lower)
    upr <- c(upr, beta = cb$upper)
    nbd <- c(nbd, beta = cb$nbd)
  }
  ## phi
  npar <- p - out$phi$nfix
  if(npar > 0){
    cb <- .bounds.convert(npar = npar, lower = lower$phi, upper = upper$phi)
    lwr <- c(lwr, phi = cb$lower)
    upr <- c(upr, phi = cb$upper)
    nbd <- c(nbd, phi = cb$nbd)
  }
  ## theta
  npar <- q - out$theta$nfix
  if(npar > 0){
    cb <- .bounds.convert(npar = npar, lower = lower$theta, upper = upper$theta)
    lwr <- c(lwr, theta = cb$lower)
    upr <- c(upr, theta = cb$upper)
    nbd <- c(nbd, theta = cb$nbd)
  }
  if(!(model == "BARC")){
    ## d - not implemented for BARC models
    if(out$d$nfix == 0){
      cb <- .bounds.convert(npar = 1, lower = lower$d, upper = upper$d)
      lwr <- c(lwr, d = cb$lower)
      upr <- c(upr, d = cb$upper)
      nbd <- c(nbd, d = cb$nbd)
    }
  }
  ## nu
  if(out$nu$nfix == 0){
    cb <- .bounds.convert(npar = 1, lower = lower$nu, upper = upper$nu)
    lwr <- c(lwr, nu = cb$lower)
    upr <- c(upr, nu = cb$upper)
    nbd <- c(nbd, nu = cb$nbd)
  }

  out$lower <- lwr
  out$upper <- upr
  out$nbd <- as.integer(nbd)

  invisible(out)
}


##-------------------------------------------------------------------------
## internal function.
## Performs several checks to make sure that
## the correct type of variables will be passed to FORTRAN
##-------------------------------------------------------------------------
.fit.configs <- function(model, yt, y.start, y.lower, y.upper, openIC,
                         xreg, xnew, nnew, xreg.start, linkg, p, d, q,
                         inf, m, xregar, error.scale, start, ignore.start,
                         lags, fixed.values, fixed.lags, lower, upper,
                         control, sco, info, extra,...){

  ##----------------------------------------------------------
  ## checking if the data has NA's or any value outside (ylower, yupper)
  ##----------------------------------------------------------
  out <- .data.check(yt = yt, lower = y.lower, upper = y.upper, openIC = openIC)
  if(!is.null(out$conv)) return(invisible(out))
  out$n <- as.integer(length(yt))
  out$m <- as.integer(m)
  out$y.lower = y.lower
  out$y.upper = y.upper

  ##-------------------------
  ## link for mu and y
  ##-------------------------
	dummy <- .link.check(model = model, link = linkg)
  if(length(linkg) == 1) linkg <- c(linkg, linkg)
  out$linkg <- .link.convert(link = linkg)

  ##-----------------
  ## regressors
  ##-----------------
  temp <- .xreg.convert(xreg = xreg, xnew = xnew, n = out$n,
                        nnew = nnew, skip.forecast = FALSE)
  out[names(temp)] <- temp

  ##---------------------------------------------------------
  ## initial values: using y.default = y.lower -1
  ## assures that the Fortran subroutine will set g(y) = 0
  ##---------------------------------------------------------
  if(out$nreg == 0) xregar = FALSE
  temp <- .data.start.convert(y.start = y.start, xreg.start = xreg.start,
                              nreg = out$nreg, xregar = xregar,
                              y.default = y.lower - 1)
  out[names(temp)] <- temp

  ##----------------------------------------------------------------------------
  ## parameters initialization and fixed values identification
  ##----------------------------------------------------------------------------
  ## updating configurations: in case the user passed one of the lists as NULL
  ## instead of an empty list, this step will avoid breaking the code
  st <- FALSE
  uc <- .fix.null.configs(coefs = start, lags = lags, fixed.values = fixed.values,
                          fixed.lags = fixed.lags, lower = lower, upper = upper)
  if(is.null(uc$coefs)) st <- TRUE
  if(d == FALSE){
    uc$fixed.values$d = 0
    if(!is.null(uc$coefs)) start$d = NULL
  }else{
    if(!is.null(uc$fixed.values$d) & !is.null(start$d)){
      stop(paste0("An initial value for d was provided:",
                  "\n start$d = ", start$d,
                  "\n but d was also fixed:",
                  "\n fixed.values$d = ", uc$fixed.values$d,".",
                  "\n If you wish to fix d, remove d from starting values or set d = FALSE. ",
                  "\n If you wish to fit d, remove d from the list of fixed values"))
    }
  }
  ##----------------------------------------------------------------------------
  ## checking if parameter initialization is required.
  ## in case no starting values were not provided, uses the default values.
  ## in case ignore.start = TRUE, starting values are ignored and recalculated.
  ## partial starting values are not allowed.
  ##----------------------------------------------------------------------------
  if(st | ignore.start)
    start <- coefs.start(model = model, yt = yt, linkg = linkg, xreg = xreg,
                         p = p, d = d, q = q, y.start = y.start, y.lower = y.lower,
                         y.upper = y.upper, lags = uc$lags, fixed.values = uc$fixed.values,
                         fixed.lags = uc$fixed.lags)

  ##----------------------------------------------------------------------------
  ## in case the user does not provide a value for the dispersion,
  ## and initialization was not required sets nu = 50
  ## (no particular reason for this choice)
  ##----------------------------------------------------------------------------
  if(is.null(c(start$nu, uc$fixed.values$nu))) start$nu <- 50

  ##----------------------------------------------------------------------------
  ## organizing the values to be passed to FORTRAN
  ##----------------------------------------------------------------------------
  if(model == "BARC"){
    temp <- list(...)
    if(!is.null(start$theta)) start$theta <- temp[["theta.barc"]]
  }

  temp <- .coefs.fit.config(model = model, coefs = start, lags = uc$lags,
                            fixed.values = uc$fixed.values,
                            fixed.lags = uc$fixed.lags, lower = uc$lower,
                            upper = uc$upper, p = p, q = q, nreg = out$nreg)
  out[names(temp)] <- temp
  out$p <- as.integer(p)

  ##-------------------------------
  ## Other configurations
  ##-------------------------------
  out$inf <- as.integer(inf)
  if(!is.null(out$d)){
    if((out$d$nfix == 0 | out$d$fvalues != 0) & out$inf < 100)
      warning(paste("non-zero d and inf = ", inf,
                    ". Be carefull, this value may be too small",
                    sep = ""), immediate. = TRUE)}

  out$error.scale <- as.integer(error.scale)
  out$xregar <- as.integer(xregar)

  out$sco <- as.integer(sco)
  out$info <- as.integer(info)
  out$extra <- as.integer(extra)

  out$control <- fit.control(control)
  out$npar <- length(out$coefs)
  # dummy in case npar  = 0
  if(out$npar == 0) out$coefs <- 0

  if(!(model == "BARC")) out$q <- as.integer(q)

  invisible(out)
}



##-------------------------------------------------------------------------
## Internal function.
## Used to print information about the selected model
##-------------------------------------------------------------------------
.fit.print <- function(model, p, q, d, nreg){

  dname <- ifelse(d, "d","0")
  msg <- model

  if(nreg == 0) msg <- paste(msg,"(", p, sep = "")
  else msg <- paste(msg,"X(",p, sep = "")

  if(!(model == "BARC")) msg <- paste(msg,",", dname, ",", q,") model", sep = "")
  else msg <- paste(msg,") model",sep = "")

  msg
}



##-------------------------------------------------------------------------
## Internal function.
## Used to extract information from the object returned by
## the FORTRAN function that fits the model
##-------------------------------------------------------------------------
.fit.get.results <- function(model, obj, configs){

  out <- c()
  out$model <- model

  ##----------------------------------
  ##  Convergence
  ##----------------------------------
  out$convergence <- obj$conv
  if(configs$control$method  == "L-BFGS-B"){
    out$message <- switch(EXPR = paste(obj$conv),
                          "0" =  "SUCCESSFUL TERMINATION",
                          "1" = "")
  }
  else{
    out$message <- switch(EXPR = paste(obj$conv),
                          "0" = "SUCCESSFUL TERMINATION",
                          "1" = "MAXIMUM NO. OF FUNCTION EVALUATIONS EXCEEDED",
                          "2" = "NOP < 1 OR STOPCR <= 0")
  }
  if(obj$conv != 0) warning("FAIL / FUNCTION DID NOT CONVERGE!", immediate. = TRUE)
  out$counts <- obj$neval
  con <- fit.control(control = list())
  nm <- names(con)
  nc <- nm %in% names(obj)
  con[nc] <- obj[nm[nc]]
  out$control <- con[nc]
  out$control$method <- configs$control$method


  ##---------------------------------------------------
  ##  Coefficients: starting values and final values
  ##---------------------------------------------------
  out$start <- configs$coefs
  out$coefficients <- obj$coefs

  ##--------------------------------------
  ##  Series
  ##--------------------------------------
  out$n = as.integer(obj$n)
  out$series  <- obj$yt
  out$gyt <- obj$gy
  out$xreg <- NULL
  if(obj$nreg > 0) out$xreg = obj$xreg
  out$fitted.values <- obj$mut
  out$etat <- obj$etat
  out$error.scale <- obj$escale
  out$error <- obj$error
  if(obj$escale == 1) out$residuals <- obj$yt - obj$mut
  else out$residuals <- obj$error
  out$forecast <- NULL
  if(obj$nnew > 0){
    out$forecast <- obj$ynew
    if(obj$nreg > 0) out$xnew = obj$xnew
  }
  if(model == "BARC"){
    out$Ts <- obj$Ts
    out$TS.forecast <- NULL
    if(obj$nnew > 0) out$Ts.forecast <- obj$Tnew
  }


  ##------------------------------------------------
  ##  likelihood, gradient and information matrix
  ##------------------------------------------------
  out$sll <- NULL
  out$score <- NULL
  out$info.Matrix <- NULL
  if(obj$llk == 1) out$sll <- obj$sll
  if(obj$sco == 1) out$score <- obj$U
  if(obj$info == 1){
    out$info.Matrix <- as.matrix(obj$K)
    colnames(out$info.Matrix) <- names(obj$coefs)
    rownames(out$info.Matrix) <- names(obj$coefs)
  }

  ##------------------------------------------------
  ##  Extra information for prediction
  ##------------------------------------------------
  nms <- names(out)
  nmsc <- names(configs)
  nmse <- !(nmsc %in% nms)
  out$configs[nmsc[nmse]] <- configs[nmse]
  out$configs$llk <- as.integer(obj$llk)

  class(out) <- c("btsr", class(out))
  invisible(out)
}




##' @title Summary Method of class BTSR
##'
##' @description \code{summary} method for class \code{"btsr"}.
##'
##' @name summary
##'
##' @aliases summary.btsr
##' @aliases print.summary.btsr
##'
##' @param object object of class \code{"btsr"}.
##' @param ... further arguments passed to or from other methods.
##'
##' @details
##'
##' @return
##' The function \code{summary.btsr} computes and returns a list
##' of summary statistics of the fitted model given in \code{object}.
##' Returns a list of class \code{summary.btsr}, which contains the
##' following components:
##'
##'  \item{model}{the corresponding model.}
##'
##'  \item{call}{the matched call.}
##'
##'  \item{residuals}{the residuals of the model. Depends on the definition
##'  of \code{error.scale}. If error.scale= 1, \eqn{residuals = g(y) - g(\mu)}.
##'  If error.scale = 0, \eqn{residuals = y - \mu}.}
##'
##'  \item{coefficients}{a \eqn{k \times 4}{k x 4} matrix with columns for
##'  the estimated coefficient, its standard error, z-statistic and corresponding
##'  (two-sided) p-value. Aliased coefficients are omitted.}
##'
##'  \item{aliased}{named logical vector showing if the original coefficients
##'  are aliased.}
##'
##'  \item{sigma.res}{the square root of the estimated variance of the random
##'  error \deqn{\hat\sigma^2 = \frac{1}{n-k}\sum_i{r_i^2},}{\sigma^2 = \frac{1}{n-k} \sum_i r[i]^2,}
##'  where \eqn{r_i}{r[i]} is the \eqn{i}-th residual, \code{residuals[i]}.}
##'
##'  \item{df}{degrees of freedom, a 3-vector \eqn{(k, n-k, k*)}, the first
##'  being the number of non-aliased coefficients, the last being the total
##'  number of coefficients.}
##'
##'  \item{vcov}{a \eqn{k \times k}{k \times k} matrix of (unscaled) covariances.
##'  The inverse ov the information matrix.}
##'
##'  \item{loglik}{the sum of the log-likelihood values}
##'
##'  \item{aic}{the AIC value. \eqn{AIC = -2*loglik+2*k}.}
##'
##'  \item{bic}{the BIC value. \eqn{BIC = -2*loglik + log(n)*k}.}
##'
##'  \item{hqc}{the HQC value. \eqn{HQC = -2*loglik + log(log(n))*k}.}
##'
##' @importFrom stats pnorm
##'
##' @export
##'
summary.btsr <- function(object,...){

  if(!"btsr" %in% class(object))
    stop("the argument 'object' must be a 'btsr' object")
  if(is.null(object$info.Matrix))
    stop(paste0("\nsummary cannot be reported because\n",
                "the information matriz is not present\n",
                "in the object provided in the imput.\n",
                "Please, fit the model again setting info = TRUE"))

  npar <- length(object$coefficients)

  ans <- c()
  ans$model <- object$model
  ans$call <- object$call
  ans$residuals <- object$residuals
  n <- length(object$residuals)
  rdf <- ans$df.residuals <- n-npar
  ans$aliased <- is.na(object$coefficients)  # used in print method
  ans$sigma.res <- sqrt(sum(ans$residuals^2)/rdf)
  class(ans) <- "summary.btsr"

  if(npar == 0){
    ans$df <- c(0L, n, length(ans$aliased))
    ans$coefficients <- matrix(NA_real_, 0L, 4L,
                               dimnames = list(NULL,c("Estimate", "Std. Error",
                                                      "z value", "Pr(>|t|)")))
    return(ans)
  }
  ans$df = c(npar, rdf, length(ans$aliased))
  ans$vcov <- solve(object$info.Matrix)
  stderror <- sqrt(diag(abs(ans$vcov)))
  zstat <- abs(object$coefficients/stderror)
  ans$coefficients <- cbind(Estimate = object$coefficients,
                            `Std. Error` = stderror,
                            `z value` = zstat,
                            `Pr(>|t|)` = 2*(1 - pnorm(zstat)))

  ans$loglik <- object$sll
  ans$aic <- -2*ans$loglik+2*npar
  ans$bic <- -2*ans$loglik + log(n)*npar
  ans$hq <- -2*ans$loglik + log(log(n))*npar
  ans$coefficients
  return(ans)

}

##' Users are not encouraged to call these internal functions directly.
##' Internal functions for package BTSR.
##'
##' @title Print Method of class BTSR
##'
##' @description
##' Print method for objects of class \code{btsr}.
#'
##' @param x object of class \code{btsr}.
##' @param digits  minimal number of significant digits, see
##' \code{\link{print.default}}.
##' @param ... further arguments to be passed to or from other methods.
##' They are ignored in this function
##'
##' @return Invisibly returns its argument, \code{x}.
##'
##' @importFrom stats coef
##'
##' @export
##'
print.btsr <- function(x, digits = max(3L, getOption("digits") - 3L), ...)
{

  if(length(coef(x))) {
    cat("Coefficients:\n")
    print.default(format(coef(x), digits = digits),
                  print.gap = 2L, quote = FALSE)
  } else cat("No coefficients\n")
  cat("\n")
  invisible(x)
}


##-----------------------------------------------
## Internal function for printing the summary
##-----------------------------------------------
##' @rdname summary
##' @importFrom stats quantile printCoefmat
##'
##' @param x an object of class \code{"summary.btsr"},
##' usually, a result of a call to \code{summary.btsr}.
##' @param digits  minimal number of significant digits, see
##' \code{\link{print.default}}.
##' @param signif.stars logical. If \code{TRUE},
##' \sQuote{significance stars} are printed for each coefficient.
##'
##' @details
##' \code{print.summary.btsr} tries to be smart about formatting the
##' coefficients, standard errors, etc. and additionally provides
##' \sQuote{significance stars}.
##'
##' @export
print.summary.btsr <- function (x, digits = max(3L, getOption("digits") - 3L),
                                signif.stars = getOption("show.signif.stars"),	...)
{

  resid <- x$residuals
  df <- x$df
  rdf <- df[2L]
  cat("\n")
  cat("-----------------------------------------------")
  cat("\nCall:\n",
      paste(deparse(x$call), sep="\n", collapse = "\n"), "\n\n", sep = "")
  if (rdf > 5L) {
    nam <- c("Min", "1Q", "Median", "3Q", "Max")
    rq <- if (length(dim(resid)) == 2L)
      structure(apply(t(resid), 1L, quantile),
                dimnames = list(nam, dimnames(resid)[[2L]]))
    else  {
      zz <- zapsmall(quantile(resid), digits + 1L)
      structure(zz, names = nam)
    }
    print(rq, digits = digits, ...)
  }
  else if (rdf > 0L) {
    print(resid, digits = digits, ...)
  } else { # rdf == 0 : perfect fit!
    cat("ALL", df[1L], "residuals are 0: no residual degrees of freedom!")
    cat("\n")
  }
  if (length(x$aliased) == 0L) {
    cat("\nNo Coefficients\n")
  } else {
    if (nsingular <- df[3L] - df[1L])
      cat("\nCoefficients: (", nsingular,
          " not defined because of singularities)\n", sep = "")
    else cat("\nCoefficients:\n")
    coefs <- x$coefficients
    if(any(aliased <- x$aliased)) {
      cn <- names(aliased)
      coefs <- matrix(NA, length(aliased), 4, dimnames=list(cn, colnames(coefs)))
      coefs[!aliased, ] <- x$coefficients
    }

    printCoefmat(coefs, digits = digits, signif.stars = signif.stars,
                 na.print = "NA", ...)
  }
  ##
  cat("\nResidual standard error:",
      format(signif(x$sigma, digits)), "on", rdf, "degrees of freedom")
  cat("\n")
  cat("-----------------------------------------------\n")
  cat("\n")
  invisible(x)
}

##---------------------------------------------------------------------------
## internal function:
## Interface between R and FORTRAN
## Also used to summarize the results of the optimization
## Returns only the relevant variables
##---------------------------------------------------------------------------
.btsr.fit <- function(model, yt, configs, debug){

  mdl <- .check.model(model[1],"fit")

  k1 <- max(1,configs$n*configs$extra)
  k2 <- max(1,(configs$npar-1+configs$nu$nfix)*configs$extra)

  if(configs$control$method == "L-BFGS-B"){
    foo <- paste("optimlbfgsb", mdl, sep = "")
    temp <- .Fortran(foo,
                     npar = max(1L,configs$npar),
                     coefs = configs$coefs,
                     nbd = configs$nbd,
                     lower = configs$lower,
                     upper = configs$upper,
                     n = configs$n,
                     yt = yt,
                     gy = numeric(configs$n),
                     ystart = configs$y.start,
                     nreg = configs$nreg,
                     xreg = configs$xreg,
                     xstart = configs$xreg.start,
                     mut = numeric(configs$n),
                     etat = numeric(configs$n),
                     error = numeric(configs$n),
                     escale = configs$error.scale,
                     nnew = configs$nnew,
                     xnew = configs$xnew,
                     ynew = numeric(max(1,configs$nnew)),
                     linkg = configs$linkg,
                     fixa = configs$alpha$nfix,
                     alpha = configs$alpha$fvalues,
                     fixb = configs$beta$nfix,
                     flagsb = configs$beta$flags,
                     beta = configs$beta$fvalues,
                     p = configs$p,
                     fixphi = configs$phi$nfix,
                     flagsphi = configs$phi$flags,
                     phi = configs$phi$fvalues,
                     xregar = configs$xregar,
                     q = configs$q,
                     fixtheta = configs$theta$nfix,
                     flagstheta = configs$theta$flags,
                     theta = configs$theta$fvalues,
                     fixd = configs$d$nfix,
                     d = configs$d$fvalues,
                     fixnu = configs$nu$nfix,
                     pdist = configs$nu$fvalues,
                     inf = configs$inf,
                     m = configs$m,
                     sll = 1,
                     U = numeric(max(1L,configs$npar)),
                     info = configs$info,
                     K = diag(max(1,configs$npar*configs$info)),
                     extra = configs$extra,
                     Drho = matrix(0, k1, k2),
                     T = numeric(k1),
                     E = matrix(0, k1,1+2*(1-configs$nu$nfix)*configs$extra),
                     h = numeric(k1),
                     iprint = as.integer(configs$control$iprint),
                     factr = configs$control$factr,
                     pgtol = configs$control$pgtol,
                     maxit = as.integer(configs$control$maxit),
                     neval = 0L,
                     conv = 0L)
  }else{
    foo <- paste("optimnelder", mdl, sep = "")
    temp <- .Fortran(foo,
                     npar = max(1L,configs$npar),
                     coefs = configs$coefs,
                     nbd = configs$nbd,
                     lower = configs$lower,
                     upper = configs$upper,
                     n = configs$n,
                     yt = yt,
                     gy = numeric(configs$n),
                     ystart = configs$y.start,
                     nreg = configs$nreg,
                     xreg = configs$xreg,
                     xstart = configs$xreg.start,
                     mut = numeric(configs$n),
                     etat = numeric(configs$n),
                     error = numeric(configs$n),
                     escale = configs$error.scale,
                     nnew = configs$nnew,
                     xnew = configs$xnew,
                     ynew = numeric(max(1,configs$nnew)),
                     linkg = configs$linkg,
                     fixa = configs$alpha$nfix,
                     alpha = configs$alpha$fvalues,
                     fixb = configs$beta$nfix,
                     flagsb = configs$beta$flags,
                     beta = configs$beta$fvalues,
                     p = configs$p,
                     fixphi = configs$phi$nfix,
                     flagsphi = configs$phi$flags,
                     phi = configs$phi$fvalues,
                     xregar = configs$xregar,
                     q = configs$q,
                     fixtheta = configs$theta$nfix,
                     flagstheta = configs$theta$flags,
                     theta = configs$theta$fvalues,
                     fixd = configs$d$nfix,
                     d = configs$d$fvalues,
                     fixnu = configs$nu$nfix,
                     pdist = configs$nu$fvalues,
                     inf = configs$inf,
                     m = configs$m,
                     sll = 0,
                     sco = configs$sco,
                     U = numeric(max(1,configs$npar*configs$sco)),
                     info = configs$info,
                     K = diag(max(1,configs$npar*configs$info)),
                     extra = configs$extra,
                     Drho = matrix(0,k1,k2),
                     T = numeric(k1),
                     E = matrix(0,k1,1+2*(1-configs$nu$nfix)*configs$extra),
                     h = numeric(k1),
                     iprint = as.integer(configs$control$iprint),
                     stopcr = configs$control$stopcr,
                     maxit = as.integer(configs$control$maxit),
                     neval = 0L,
                     conv = 0L)
  }

  temp$llk = 1
  temp$sco = configs$sco
  out <- .fit.get.results(model = model[1], temp, configs = configs)
  if(debug) out$out.Fortran <- temp
  invisible(out)
}

