% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bootstrap.R
\name{boot.BTLLasso}
\alias{boot.BTLLasso}
\title{Bootstrap function for BTLLasso}
\usage{
boot.BTLLasso(model, quantiles = c(0.025, 0.975), B = 500, lambda = NULL,
  cores = 1, trace = TRUE, trace.cv = TRUE)
}
\arguments{
\item{model}{A \code{cv.BTLLasso} object.}

\item{quantiles}{Vector of two quantiles that are computed, 
will be used as borders of confidence intervals 
plotted by \code{\link{ci.BTLLasso}}.}

\item{B}{Number of bootstrap iterations.}

\item{lambda}{Vector of tuning parameters. If not specified (default),
tuning parameters from \code{cv.BTLLasso} object are used. See also details.}

\item{cores}{Number of cores for (parallelized) computation.}

\item{trace}{Should the trace of the BTLLasso algorithm be printed?}

\item{trace.cv}{Should the trace fo the cross-validation be printed? If
parallelized, the trace is not working on Windows machines.}
}
\value{
\item{cv.model}{\code{cv.BTLLasso} object} \item{estimatesB}{Matrix
containing all B estimates for original parameters. For internal use.}
\item{estimatesBrepar}{Matrix containing all B estimates for reparameterized
(symmetric side constraints) parameters.} \item{lambdaB}{vector of used
tuning parameters} \item{conf.ints}{Bootstrap confidence intervals for
original parameters. For internal use.} \item{conf.ints.repar}{Bootstrap
confidence intervals for reparameterized (symmetric side constraint)
parameters.} \item{lambda.max.alert}{Was the largest value of lambda chosen
in at least one bootstrap iteration?} \item{lambda.min.alert}{Was the
smallest value of lambda chosen in at least one bootstrap iteration?}
}
\description{
Performs bootstrap for BTLLasso to get bootstrap confidence intervals. Main
input argument is a \code{cv.BTLLasso} object. The bootstrap is performed on
level of the cross-validation. Therefore, within every bootstrap iteration
the complete cross-validation procedure from the \code{cv.BTLLasso} object
is performed. The resulting \code{boot.BTLLasso} object is put into
\code{\link{ci.BTLLasso}} to plot bootstrap confidence intervals.
}
\details{
The method can be highly time-consuming, for high numbers of tuning
parameters, high numbers of folds in the crossvalidation and high number of
bootstrap iterations B.  The number of tuning parameters can be reduced by
specifying lambda in the \code{boot.BTLLasso} function. You can control if
the range of prespecified tuning parameters was to small by looking at the
output values \code{lambda.max.alert} and \code{lambda.min.alert}. They are
set \code{TRUE} if the smallest or largest of the specifed lambda values was
chosen in at least one bootstrap iteration.
}
\examples{

\dontrun{
##### Example with simulated data set containing X, Z1 and Z2
data(SimData)

## Specify tuning parameters
lambda <- exp(seq(log(151),log(1.05),length=30))-1

## Specify control argument, allow for object-specific order effects and penalize intercepts
ctrl <- ctrl.BTLLasso(penalize.intercepts = TRUE, object.order.effect = TRUE,
                      penalize.order.effect.diffs = TRUE)

## Simple BTLLasso model for tuning parameters lambda
m.sim <- BTLLasso(Y = SimData$Y, X = SimData$X, Z1 = SimData$Z1, 
                  Z2 = SimData$Z2, lambda = lambda, control = ctrl)

singlepaths(m.sim, x.axis = "loglambda")

## Cross-validate BTLLasso model for tuning parameters lambda
set.seed(5)
m.sim.cv <- cv.BTLLasso(Y = SimData$Y, X = SimData$X, Z1 = SimData$Z1, 
                        Z2 = SimData$Z2, lambda = lambda, control = ctrl)


singlepaths(m.sim.cv, x.axis = "loglambda", plot.order.effect = FALSE, plot.intercepts = FALSE, 
            plot.Z2 = FALSE)
paths(m.sim.cv, y.axis="L2")

## Example for bootstrap confidence intervals for illustration only
## Don't calculate bootstrap confidence intervals with B = 10
set.seed(5)
m.sim.boot <- boot.BTLLasso(m.sim.cv, B = 10, cores = 10)
ci.BTLLasso(m.sim.boot)

##### Example with small version from GLES data set
data(GLESsmall)

## vector of subtitles, containing the coding of the X covariates
subs.X <- c("","female (1); male (0)")

## vector of tuning parameters
lambda <- exp(seq(log(61),log(1),length=30))-1


## compute BTLLasso model 
m.gles <- BTLLasso(Y = GLESsmall$Y, X = GLESsmall$X, Z1 = GLESsmall$Z1, lambda = lambda)

singlepaths(m.gles, x.axis = "loglambda", subs.X = subs.X)
paths(m.gles, y.axis="L2")

## Cross-validate BTLLasso model 
m.gles.cv <- cv.BTLLasso(Y = GLESsmall$Y, X = GLESsmall$X, Z1 = GLESsmall$Z1, lambda = lambda)

singlepaths(m.gles.cv, x.axis = "loglambda", subs.X = subs.X)
}

}
\author{
Gunther Schauberger\cr \email{gunther@stat.uni-muenchen.de}\cr
\url{http://www.statistik.lmu.de/~schauberger/}
}
\references{
Schauberger, Gunther and Tutz, Gerhard (2015): Modelling
Heterogeneity in Paired Comparison Data - an L1 Penalty Approach with an
Application to Party Preference Data, \emph{Department of Statistics, LMU
Munich}, Technical Report 183
}
\seealso{
\code{\link{BTLLasso}}, \code{\link{cv.BTLLasso}},
\code{\link{ci.BTLLasso}}
}
\keyword{BTLLasso}
\keyword{bootstrap}
\keyword{confidence}
\keyword{interval}

