% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MIXclustering.R
\name{MIXclustering}
\alias{MIXclustering}
\title{Bayesian Nonparametric Model for Clustering with Mixed Scale Variables}
\usage{
MIXclustering(Y, var_type, n_iter_out = 2000, n_burn = 100, n_thin = 2,
  a_fix = NULL, alpha = 0.5, d_0_a = 1, d_1_a = 1, b_fix = NULL,
  d_0_b = 1, d_1_b = 1, eta = 2, d_0_z = 2.1, d_1_z = 30, kappa = 5,
  delta = 4, d_0_mu = 2.1, d_1_mu = 30, sampling_prob = NULL,
  expansion_f = NULL, log_file = NULL)
}
\arguments{
\item{Y}{Matrix or data frame containing the data to be clustered.}

\item{var_type}{Character vector that indicates the type of variable in each column of x. Three possible types:
\itemize{
  \item "\strong{c}" for continuous variables. It is assumed to be Gaussian-shaped.
  \item "\strong{o}" for ordinal variables (binary and ordered categorical).
  \item "\strong{m}" for nominal variables (non-ordered categorical).
}}

\item{n_iter_out}{Number of effective iterations in the MCMC procedure for clustering.}

\item{n_burn}{Number of iterations discarded as part of the burn-in period at the beginning MCMC procedure.}

\item{n_thin}{Number of iterations discarded for thining the chain (reducing the autocorrelation). We keep 1 of every n_thin iterations.}

\item{a_fix}{A numeric value to set the parameter \eqn{a} in the model. If \code{NULL} (default), the parameter \eqn{a} is assigned a prior distribution. See \code{details}.}

\item{alpha}{Hyperparameter in the prior distribution of \eqn{a}. See \code{details}.}

\item{d_0_a}{Hyperparameter in the prior distribution of \eqn{a}. See \code{details}.}

\item{d_1_a}{Hyperparameter in the prior distribution of \eqn{a}. See \code{details}.}

\item{b_fix}{A numeric value to set the parameter \eqn{b} in the model. If \code{NULL} (default), the parameter \eqn{b} is assigned a prior distribution. See \code{details}.}

\item{d_0_b}{Hyperparameter in the prior distribution of \eqn{b}. See \code{details}.}

\item{d_1_b}{Hyperparameter in the prior distribution of \eqn{b}. See \code{details}.}

\item{eta}{Tuning parameter controlling the proposal in the \emph{Metropolis-Hastings} step for \eqn{b}.}

\item{d_0_z}{Hyperparameter in the prior distribution of the variance for the latent variables. See \code{details}.}

\item{d_1_z}{Hyperparameter in the prior distribution of the variance for the latent variables. See \code{details}.}

\item{kappa}{Tuning parameter controlling the proposal in the \emph{Metropolis-Hastings} step for the variance of latent variables.}

\item{delta}{Tuning parameter controlling the proposal in the \emph{Metropolis-Hastings} step for the correlation of latent variables.}

\item{d_0_mu}{Hyperparameter in the prior distribution of the variance of the location in each cluster. See \code{details}.}

\item{d_1_mu}{Hyperparameter in the prior distribution of the variance of the location in each cluster. See \code{details}.}

\item{sampling_prob}{vector with the sampling probabilities \eqn{\pi_i} for each individual in case that the data come from a complex survey sample. By default \eqn{\pi_i=1}.}

\item{expansion_f}{vector with the expansion factors, the reciprocal of the sampling probabilities, \eqn{w_i = 1/\pi_i}. If both \code{sampling_prob} and \code{expansion_f} are specified, preference is given to \code{sampling_prob}.}

\item{log_file}{Specifies a file to save the details with the execution time and the parameters used.}
}
\value{
\code{MIXclustering} returns a S3 object of class "\code{MIXcluster}".

The generic methods \code{\link{summary}} and \code{\link{plot}} are defined for this class.

An object of class "MIXcluster" is a list containing the following components:
\describe{
    \item{\code{cluster}}{vector with the cluster allocation for each row in the data. It corresponds to the iteration which is Closest-To-Average (CTA) arrangement.}
    \item{\code{cluster_heterogeneity}}{Heterogeneity Measure (HM) for the cluster in the previous point. The HM measure is discussed in section 4 of Carmona et al. (2017).}
    \item{\code{Y.cluster.summary}}{a summary of the data divided by the allocation in \code{$cluster}.}
    \item{\code{Y.var_type}}{vector with the variable types in the data.}
    \item{\code{Y.na}}{vector specifying the rows with missing values.}
    \item{\code{Y.n}}{number of rows in the data.}
    \item{\code{Y.p}}{number of variables in the data.}
    \item{\code{MC.clusters}}{matrix with the cluster allocation for each row in the data. Each column corresponds to an effective iteration in the MCMC simulation of the model (after discarding burn-in and thinning iterations).}
    \item{\code{MC.clusters_heterogeneity}}{Heterogeneity Measure (HM) for all the clusters returned in \code{MC.clusters}.}
    \item{\code{cluster.matrix.avg}}{average similarity matrix of size \eqn{n} by \eqn{n}.}
    \item{\code{MC.values}}{a list with the simulated values of the chains for the parameters \eqn{a},\eqn{b},\eqn{\Lambda},\eqn{\Omega}.}
    \item{\code{MC.accept.rate}}{a named vector with the acceptance rates for each parameter. It includes iterations that are discarded in the burn-in period and thinning.}
    \item{\code{call}}{the matched call.}
}
}
\description{
\code{MIXclustering} is used to perform cluster analisis of individuals using a Bayesian nonparametric mixture model that jointly models mixed scale data and accommodates for different sampling probabilities. The model is described in Carmona, C., Nieto-Barajas, L. E., Canale, A. (2016).
}
\details{
The model consists on a bayesian non-parametric approach for clustering that is capable to combine different types of variables through the usage of associated continuous latent variables. The clustering mechanism is based on a location mixture model with a Poisson-Dirichlet (\eqn{PD}) process prior on the location parameters \eqn{\mu_i ; i=1,...,n} of the associated latent variables.

Computational inference about the cluster allocation and the posterior distribution of the parameters are performed using MCMC simulations.

A full description of the model is in the article Carmona et al. (2016) (preprint: \url{http://arxiv.org/abs/1612.00083}). See \code{Reference}.

The model consider an individual \eqn{y_i} that is characterized by a multivariate response of dimension \eqn{p}, i.e., \eqn{y_i=(y_{i,1},...,y_{i,p})}. The total number of variables \eqn{p} is divided into \eqn{c} continuous variables, \eqn{o} ordinal variables, and \eqn{m} nominal variables such that \eqn{p=c+o+m}.

For the continuous variables, it is convenient that the variables have a real support. The user may have transformed the original values before using the function \code{MIXclustering}.

For each response \eqn{y_i=(y_{i,1},...,y_{i,p})} (of dimension \eqn{p}) a corresponding latent vector \eqn{z_i=(z_{i,1},...,z_{i,q})} (of dimension \eqn{q}) is created, according to the following:
\itemize{
   \item For each continuous variable \eqn{y_{i,j} ; j=1,...,c} the algorithm uses a latent with the same values \eqn{z_{i,j}=y_{i,j}}.
   \item For each ordinal variable \eqn{y_{i,j} , j = c+1,...,c+o}, with \eqn{K_j} different ordered values, the algorithm creates one latent \eqn{z_{i,j}}, that allows to map the categories into continuous values divided by thresholds. For example, for a binary \eqn{y_j}, we have \eqn{y_j=0} iff \eqn{z_j<0} and \eqn{y_j=1} iff \eqn{z_j>0}
   \item For each nominal variable \eqn{y_{i,j} , j = c+o+1,...,c+o+m}, with \eqn{L_j} categories, the algorithm require \eqn{L_j-1} latent variables, whose relative order is consistent with the observed category.
   }

The data may come from a complex survey sample where each individual \eqn{y_i} has known sampling probability \eqn{\pi_i, i = 1,...,n}. The reciprocal of these sampling probabilities, \eqn{w_i=1/\pi_i}, are called expansion factors or sampling design weights.

The joint model for the latent vector is therefore:

\eqn{(z_i | \mu_i,\Sigma)} ~ \eqn{N_q(\mu_i, \pi_i \Sigma )}

(Note: the final model in Carmona et al. (2016) has variance \eqn{\kappa \pi_i \Sigma}. This value of \eqn{\kappa} can be used in the package through a transformed sampling probability vector \eqn{\pi_i^*=\kappa\pi_i})

The clustering model will be based in an appropriate choice of the prior distribution on the \eqn{\mu_i}'s. A clustering of the \eqn{\mu_i}'s will induce a clustering of the \eqn{y_i}'s. Our prior on the \eqn{\mu_i}'s will be:

\eqn{\mu_i | G}~\eqn{G}, iid for \eqn{i=1,...,n}

Where \eqn{G}~\eqn{PD(a,b,G_0)} is a Poisson-Dirichlet process with parameters \eqn{a \in [0,1)}, \eqn{b>-a} and centring measure \eqn{G_0}. The Dirichlet and the normalized stable processes arise when \eqn{a=0} and when \eqn{b=0}, respectively.

In consequence, this choice of prior implies that the \eqn{\mu_i}'s are exchangeable with marginal distribution \eqn{\mu_i}~\eqn{G_0} for all \eqn{i=1,...,n}.

In our case, \eqn{G(\mu)=N(0,\Sigma_\mu)}, where \eqn{\Sigma_\mu = diag( \sigma^2_{\mu 1} ,...,\sigma^2_{\mu q} )}.

The parameters \eqn{a} and \eqn{b} in the model define the \eqn{PD} process and therefore control the number of groups. These parameters can be fixed, resulting in a larger/smaller number of groups if assigned a larger/smaller value, respectively.

There are 9 hyperparameters in the function that also characterize the prior distributions in the model:
\itemize{
    \item f(a) = alpha * I(a=0) + (1-alpha) * dbeta( a | d_0_a , d_0_a )
    \item f(b | a) = dgamma( b + a | d_0_b , d_1_b )
    \item sigma^2 ~ inverse-gamma( d_0_z , d_1_z)
    \item sigma^2_mu ~ inverse-gamma( d_0_mu , d_1_mu )
}

The definition of these values also affect the number of resulting clusters since they affect the variance implied in the model.

For example, increasing the values of \code{d_1_a} and \code{d_1_b} reduce the number of groups.

Finally, the function parameters \code{eta}, \code{kappa}, \code{delta} are tuning parameters that control the acceptance rate in the random-walk MH steps of the new proposed values for the parameters \eqn{b}, \eqn{\Lambda_{j,j}} (variance of latents) and \eqn{\Omega_{i,j}} (correlation of latents). These parameters are not recomended to be changed (used in the internal functions: \code{sampling_b} , \code{sampling_Lambda_jj} , \code{sampling_Omega_ij}).
}
\examples{

##############################
#     Simulation study 1     #
#    Carmona et al. (2017)   #
##############################

# Data and parameters are discussed in section 5.1 of Carmona et al. (2017) #

# Set seed for reproducibility #
set.seed(0) 


# Specification of data Y #
help(Y_ex_5_1)
# Choose 1, 2, or 3 #
ex_i <- 1

# specification of parameters #
help(meta_param_ex)
# Choose "a", "b" or "c" #
param_j <- "c"

var_type_Y_ex_5_1 <- list( c("c","c","c"),
                           c("o","o"),
                           c("o","o","o","c") )
\dontrun{
cluster_ex <- MIXclustering( Y = as.matrix(Y_ex_5_1[[ ex_i ]]),
                             var_type=var_type_Y_ex_5_1[[ ex_i ]],
                             
                             n_iter_out=1500,
                             n_burn=200,
                             n_thin=3,
                             
                             alpha = meta_param_ex[ param_j, "alpha" ],
                             d_0_a = meta_param_ex[ param_j, "d_0_a"],
                             d_1_a = meta_param_ex[ param_j, "d_1_a" ],
                             d_0_b = meta_param_ex[ param_j, "d_0_b" ],
                             d_1_b = meta_param_ex[ param_j, "d_1_b" ],
                             eta = meta_param_ex[ param_j, "eta" ],
                             kappa = meta_param_ex[ param_j, "kappa" ],
                             delta = meta_param_ex[ param_j, "delta" ],
                             
                             d_0_z = meta_param_ex[ param_j, "d_0_z" ],
                             d_1_z = meta_param_ex[ param_j, "d_1_z" ],
                             d_0_mu = meta_param_ex[ param_j, "d_0_mu" ],
                             d_1_mu = meta_param_ex[ param_j, "d_1_mu" ] )
# Summary of clustering results
summary(cluster_ex)

# Representation of clustering results
plot(cluster_ex,type="heatmap")
plot(cluster_ex,type="chain")

# Comparison of cluster configurations #
# 1) Minimum distance with average MCMC iterations
# 2) Minimum Heterogeneity Measure (HM)
plot( x=jitter(cluster_ex$cluster),y=jitter(cluster_ex$clusterHMmin), col="#FF000080", pch=20,
      main=paste("Comparison of two relevant cluster configurations"),
      xlab="minimizes distance to average MCMC grouping", ylab="minimizes Heterogeneity Measure" )

# Comparison with the original clusters in the simulated data
plot(x=jitter(Z_latent_ex_5_1$cluster),
     y=jitter(cluster_ex$cluster),
     main=paste("Comparison real configuration with the model results"),
     xlab="Real cluster",
     ylab="Model cluster",
     pch=19, col="#FF000080")
}

##############################
#      Households data       #
#    Carmona et al. (2017)   #
##############################

# Testing "MIXclustering" function with poverty.data #
# Data and parameters are discussed in section 5.3 of Carmona et al. (2017) #

# Set seed for reproducibility #
set.seed(0) 


\dontrun{
# relevant variables for clustering households #
Y_names <- c("ict_norm",
             "ic_ali","ic_asalud","ic_cv",
             "ic_rezedu","ic_sbv","ic_segsoc",
             "niv_ed","tam_loc")
Y_var_type <- c("c","o","o","o","o","o","o","o","m")

# using only data from state 15 (Edomex) #
aux_subset <- rep(T,nrow(poverty.data))
aux_subset <- aux_subset & is.element(substr(poverty.data$folioviv,1,2),"15")

Y_data <- poverty.data[aux_subset,Y_names]

### Sampling probability dependin on the scenario ###
# Scenario description in section 5.3 of Carmona et al. (2017) #
# Choose 1, 2 or 3 #
poverty_sampling_spec <- 3


if (poverty_sampling_spec == 1) {
  k <- 1
  sampling_prob_pov <- rep(1,nrow(Y_data))
} else if (poverty_sampling_spec == 2) {
  k <- 2 * mean(poverty.data[aux_subset,"factor_hog"])
  sampling_prob_pov <- 1/poverty.data[aux_subset,"factor_hog"]
} else if (poverty_sampling_spec == 3) {
  k <- 4 * mean(poverty.data[aux_subset,"factor_hog"])
  sampling_prob_pov <- 1/poverty.data[aux_subset,"factor_hog"]
}

cluster_poverty <- MIXclustering( Y=Y_data,
                                  var_type=Y_var_type,
                                  n_iter_out=1500,
                                  n_burn=200,
                                  n_thin=3,
                                  
                                  alpha = 0.5,
                                  d_0_a = 1, d_1_a = 1,
                                  d_0_b = 1, d_1_b = 1,
                                  
                                  eta = 2,
                                  kappa = 5,
                                  delta = 4,
                                  
                                  d_0_z = 2.1, d_1_z = 30,
                                  d_0_mu = 2.1, d_1_mu = 30,
                                  
                                  sampling_prob = k * sampling_prob_pov )

summary(cluster_poverty)
plot(cluster_poverty,type="heatmap")
plot(cluster_poverty,type="chain")
}

}
\references{
Carmona, C., Nieto-Barajas, L. E. & Canale, A. (2017). \emph{Model based approach for household clustering with mixed scale variables.} (preprint: \url{http://arxiv.org/abs/1612.00083})
}
\seealso{
\code{\link{summary.MIXcluster}} for a summary of the clustering results, \code{\link{plot.MIXcluster}} for graphical representation of results.
}
