% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_prior.R
\name{get_prior}
\alias{get_prior}
\title{Construct Prior Parameters for BKP/DKP Models}
\usage{
get_prior(
  prior = c("noninformative", "fixed", "adaptive"),
  model = c("BKP", "DKP"),
  r0 = 2,
  p0 = NULL,
  y = NULL,
  m = NULL,
  Y = NULL,
  K = NULL
)
}
\arguments{
\item{prior}{Type of prior: \code{"noninformative"} (default),
\code{"fixed"}, or \code{"adaptive"}.}

\item{model}{A character string specifying the model type: \code{"BKP"}
(binary outcome) or \code{"DKP"} (multi-class outcome).}

\item{r0}{Global prior precision (used when \code{prior = "fixed"} or
\code{"adaptive"}).}

\item{p0}{For BKP, a scalar in \code{(0,1)} specifying the prior mean of
success probability when \code{prior = "fixed"}. For DKP, a numeric vector
of length equal to the number of classes specifying the global prior mean,
which must sum to 1.}

\item{y}{A numeric vector of observed successes (length \code{n}).}

\item{m}{A numeric vector of total binomial trials (length \code{n}),
corresponding to each \code{y}.}

\item{Y}{A numeric matrix of observed class counts (\code{n × q}), required
only when \code{model = "DKP"}, where \code{n} is the number of
observations and \code{q} the number of classes.}

\item{K}{A precomputed kernel matrix, typically obtained from
\code{\link{kernel_matrix}}. Can be rectangular (\code{m × n}), where
\code{n} is the number of observed points and \code{m} the number of
prediction locations.}
}
\value{
\itemize{
\item If \code{model = "BKP"}: a list with
\describe{
\item{\code{alpha0}}{Vector of prior alpha parameters for the Beta
distribution, length \code{n}.}
\item{\code{beta0}}{Vector of prior beta parameters for the Beta
distribution, length \code{n}.}
}
\item If \code{model = "DKP"}: a list containing
\describe{
\item{\code{alpha0}}{Matrix of prior Dirichlet parameters at each input
location (\code{n × q}).}
}
}
}
\description{
Computes prior parameters for the Beta Kernel Process (BKP, for
binary outcomes) or Dirichlet Kernel Process (DKP, for multi-class
outcomes). Supports \code{prior = "noninformative"}, \code{"fixed"}, and
\code{"adaptive"} strategies.
}
\details{
\itemize{
\item \code{prior = "noninformative"}: flat prior; all parameters set to 1.
\item \code{prior = "fixed"}:
\itemize{
\item BKP: uniform Beta prior \code{Beta(r0 * p0, r0 * (1 - p0))} across locations.
\item DKP: all rows of \code{alpha0} set to \code{r0 * p0}.
}
\item \code{prior = "adaptive"}:
\itemize{
\item BKP: prior mean estimated at each location via kernel smoothing of observed
proportions \code{y/m}, with precision \code{r0}.
\item DKP: prior parameters computed by kernel-weighted smoothing of observed
class frequencies in \code{Y}, scaled by \code{r0}.
}
}
}
\examples{
# -------------------------- BKP ---------------------------
set.seed(123)
n <- 10
X <- matrix(runif(n * 2), ncol = 2)
y <- rbinom(n, size = 5, prob = 0.6)
m <- rep(5, n)
K <- kernel_matrix(X)
prior_bkp <- get_prior(
  model = "BKP", prior = "adaptive", r0 = 2, y = y, m = m, K = K
)

# -------------------------- DKP ---------------------------
set.seed(123)
n <- 15; q <- 3
X <- matrix(runif(n * 2), ncol = 2)
true_pi <- t(apply(X, 1, function(x) {
  raw <- c(
    exp(-sum((x - 0.2)^2)),
    exp(-sum((x - 0.5)^2)),
    exp(-sum((x - 0.8)^2))
  )
  raw / sum(raw)
}))
m <- sample(10:20, n, replace = TRUE)
Y <- t(sapply(1:n, function(i) rmultinom(1, size = m[i], prob = true_pi[i, ])))
K <- kernel_matrix(X, theta = rep(0.2, 2), kernel = "gaussian")
prior_dkp <- get_prior(
  model = "DKP", prior = "adaptive", r0 = 2, Y = Y, K = K
)

}
\references{
Zhao J, Qing K, Xu J (2025). \emph{BKP: An R Package for Beta
Kernel Process Modeling}. arXiv.
https://doi.org/10.48550/arXiv.2508.10447
}
\seealso{
\code{\link{fit_BKP}} for fitting Beta Kernel Process models,
\code{\link{fit_DKP}} for fitting Dirichlet Kernel Process models,
\code{\link{predict.BKP}} and \code{\link{predict.DKP}} for making
predictions, \code{\link{kernel_matrix}} for computing kernel matrices used
in prior construction.
}
