\name{phaseCalibrator} 
\alias{phaseCalibrator} 
\title{Generates an SPD for each phase in a dataset} 
\description{Generates a data frame of SPDs, one for each phase. }
\usage{phaseCalibrator(data, CalArray, width = 200, remove.external = FALSE)}
\arguments{
  \item{data}{
	A dataframe of 14C dates. Requires 'age' and 'sd', and at least one of 'site' and 'phase'. Optional 'datingType' comprising '14C' and/or anything else.
	}
  \item{CalArray}{
	A 2D probability array of the calibration curve generated by \code{\link[=makeCalArray]{makeCalArray}} containing row names and column names. 
	}
  \item{width}{
	A timespan in C14 years used to automatically bin dates if they have not been phased, i.e., 'phase' is missing from the data. Default = 200.
	}
  \item{remove.external}{
	Default FALSE retains the SPDs of all phases, even if some have little or no probability mass inside the date range. If TRUE, those phases with the majority of their probability mass outside the date range are removed. 
	}	
}
\value{
	Returns a data frame of probabilities. Each column provides the SPD of a phase. Column names are the phase names, row names are the calendar years.
	}

\details{

Generates an SPD for each phase using \code{\link[=summedCalibrator]{summedCalibrator}} with the default normalisation = 'standard'. Returns a data frame of probabilities, with phase names assigned to column names, and calendar years assigned to row names. 

As a minimum requirement the data must include either 'phase' or 'site'. If 'phase' is unavailable the function will automatically use 'site' to bin the dates into site-phases. This binning is achieved with a crude algorithm that assigns a date to a bin if it is within 200 years of any other date in that bin, based on the uncalibrated C14 mean date (non-14C dates are also mapped to 14C time for the purpose of this binning). The need to bin dates into phases is an important step in modelling population dynamics to adjust for the data ascertainment bias of some archaeological finds having more dates by virtue of a larger research interest/budget. This binning algorithm provides a simple and useful solution to handling large datasets that have not been phased, but is not an alternative to an OxCal phase model if the objective is to directly estimate phase boundary dates at a specific site.

Optionally 'datingType' can be provided in the data. Only '14C' will be calibrated in the usual way, anything else is assumed to be provided in calendar time. If 'datingType' is not provided, all dates are assumed to be 14C.

Each column of the output data frame is a vector of probabilities representing the SPD of a phase. However, if the date range used to generate CalArray does not encompass the entire dataset, some phases will have PD outside the date range, giving a SPD area < 1. This avoids deleterious edge effects.

If using the output data frame to search for a population model, it is crucial to exclude dates outside (or mostly outside) the date range. This is achieved with remove.external = TRUE. 
	}
\examples{

	CalArray <- makeCalArray(intcal20, calrange = c(9000,11000), inc = 5 )

	# minimum data requirement includes 'mean' and 'sd' and either 'site' or 'phase'   
	data <- data.frame(age = c(8350,8500,8900,9200), 
			sd = c(50,50,50,50), 
			site = c('field','field','field','garden'))
	x <- phaseCalibrator(data, CalArray)

	# notice three phases were automatically generated, each with a total SPD area = 1
	colSums(x)*5

	# in contrast, three dates are specified as coming from the same phase, 
	# and the 'garden.1' phase is partly outside the date range
	data <- data.frame(age = c(8350,8500,8900,9480), 
			sd = c(50,50,50,50), 
			phase = c('field.1','field.1','field.1','garden.1'))
	x <- phaseCalibrator(data, CalArray)
	colSums(x)*5
	}
