% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aat_covreliability.R
\name{aat_covreliability}
\alias{aat_covreliability}
\alias{print.aat_covreliability}
\alias{aat_covreliability_jackknife}
\alias{print.aat_covreliability_jackknife}
\alias{plot.aat_covreliability_jackknife}
\title{Compute a dataset's reliability from its covariance matrix}
\usage{
aat_covreliability(
  ds,
  subjvar,
  stimvar,
  pullvar,
  targetvar = NULL,
  rtvar,
  aggfunc = c("mean", "median"),
  algorithm = c("calpha", "lambda2", "lambda4"),
  iters = 5
)

\method{print}{aat_covreliability}(x, ...)

aat_covreliability_jackknife(
  ds,
  subjvar,
  stimvar,
  pullvar,
  targetvar = NULL,
  rtvar,
  algorithm = c("calpha", "lambda2", "lambda4"),
  iters = 5,
  holdout = c("both", "participant", "stimulus", "cross")
)

\method{print}{aat_covreliability_jackknife}(x, ...)

\method{plot}{aat_covreliability_jackknife}(x, ...)
}
\arguments{
\item{ds}{the \code{data.frame} to use}

\item{subjvar}{Name of the subject-identifying variable}

\item{stimvar}{Name of the stimulus-identifying variable}

\item{pullvar}{Name of the movement-direction identifying variable}

\item{targetvar}{Optional. Name of the stimulus-category identifying variable}

\item{rtvar}{Name of the reaction-time identifying variable}

\item{aggfunc}{The function with which to aggregate the RTs before computing difference scores. Defaults to mean but can be changed to median.}

\item{algorithm}{The reliability formula to use. Defaults to Cronbach's alpha, but Guttman's Lambda-2 is recommended instead.}

\item{iters}{If there are missing values (which is almost inevitable) then
multiple imputation will be used to complete the covariance matrix - this option sets
the number of multiple imputations to be used.}

\item{x}{Object to be printed}

\item{...}{Ignored}

\item{holdout}{What should be removed from the data for computation of jackknife statistics?
"both" computes reliability when stimuli and participants are separately removed,
while "cross" computes  reliability when stimuli and participants are simultaneously removed.}
}
\value{
Returns an \code{aat_covreliability} object containing the reliability value
as well as the dataset and covariance matrix with replaced missing values. When
the argument \code{targetvar} is provided, the output also contains the reliability of the
individual stimulus categories and their intercorrelation.

\code{aat_covreliability_jackknife()} returns an \code{aat_covreliability_jackknife} object,
containing jackknife reliability statistics. If argument \code{holdout} was set to "cross",
then these statistics are provided in a matrix where rows represent participants and columns represent stimuli.
Otherwise, they are provided in \code{data.frame}s where the stimulus or participant is represented in a column
alongside the associated reliability value.
}
\description{
This function computes mean single-difference scores (push minus pull) for individual stimuli,
and computes the reliability from that information.
Missing values are dealt with using multiple imputation.

This function computes the reliability when stimuli and participants are removed,
allowing for the diagnosis of potential sources of unreliability within the data.
}
\details{
When only one stimulus category is indicated, one of the commonly known reliability algorithms
provided with the \code{algorithm} argument is used.
When two stimulus categories are indicated, this function uses Lord's (1963) algorithm to
compute the reliability of a double mean difference score, using the algorithms in \code{algorithm}
to estimate the reliability of indiviau lstimulus categories.

When one wants to compute the reliability of a double median difference score or D-score,
\code{aat_splithalf()} is recommended instead.
}
\section{Methods (by generic)}{
\itemize{
\item \code{print(aat_covreliability)}: Print an \code{aat_covreliability} object

}}
\section{Functions}{
\itemize{
\item \code{print(aat_covreliability_jackknife)}: Print an \code{aat_covreliability_jackknife} object

\item \code{plot(aat_covreliability_jackknife)}: Plot an \code{aat_covreliability_jackknife} object

}}
\examples{
#We generate a dataset with 16 stimuli in each category
ds<-aat_simulate(biasfx_jitter=40,nstims=16)
ds$stim<-paste0(ds$stim,"-",ds$is_target)

# If Lord's formula and
# bootstrapped splithalf measure something similar,
# then the outcomes should be close to each other.
aat_covreliability(ds=ds,subjvar="subj",stimvar="stim",pullvar="is_pull",
                           targetvar="is_target",rtvar="rt")
aat_splithalf(ds=ds,subjvar="subj",pullvar="is_pull",targetvar="is_target",rtvar="rt",
              algorithm="aat_doublemeandiff",iters=100,plot=FALSE)

#Testing reliability for single-difference scores
ds<-ds[ds$is_target==1,]
aat_covreliability(ds=ds,subjvar="subj",stimvar="stim",pullvar="is_pull",rtvar="rt")
hh<-aat_simulate()
test<-aat_covreliability_jackknife(ds=hh,subjvar="subj",stimvar="stim",pullvar="is_pull",
                                   targetvar="is_target",rtvar="rt",holdout="cross")
print(test)
plot(test)
}
\references{
Lord, F.Y. (1963), "Elementary Models for Measuring Change",
in Problems in Measuring Change, C.W. Harris, ed.. Madison. Wisconsin:
University of Wisconsin.
}
