#' @title Calculate the AUK Estimator
#'
#' @description 
#' Computes the Area Under the Kendall (AUK) estimator for multivariate independence. 
#' The AUK estimator is based on the survival copula and quantifies the deviation 
#' from the null hypothesis of independence.
#'
#' @details 
#' The AUK estimator utilizes the empirical distribution function \eqn{F_n} and a 
#' generalized function \eqn{g(t)} to quantify independence in \eqn{d} dimensions. 
#' Under the null hypothesis of independence, the AUK value tends toward 0.5. 
#' The implementation includes numerical stability fixes for high-dimensional data.
#'
#' @param data A matrix or data frame of observations where columns represent variables. 
#' Must contain at least 2 observations and 2 variables, with no missing values.
#'
#' @return A numeric value in the interval [0, 1] representing the AUK estimator.
#'
#' @references 
#' Afendras, G., Markatou, M., & Papantonis, N. (2025). AUK-based test for 
#' mutual independence and an index of mutual dependence. Journal of 
#' Multivariate Analysis, 105589. <doi:10.1016/j.jmva.2025.105589>
#'
#' @examples
#' # Example 1: Independent Data
#' set.seed(123)
#' data_indep <- matrix(runif(300), 100, 3)
#' AUKtest(data_indep)
#'
#' # Example 2: Dependent Data
#' x <- runif(100)
#' data_dep <- cbind(x, x^2, sqrt(x))
#' AUKtest(data_dep)
#'
#' @export
AUKtest <- function(data) {
  
  # --- Defensive Programming: Input Validation ---
  
  if (!is.matrix(data) && !is.data.frame(data)) {
    stop("Input 'data' must be a matrix or a data frame.")
  }
  
  X <- as.matrix(data)
  
  if (any(is.na(X))) {
    stop("Input 'data' contains missing values (NA).")
  }
  
  if (!is.numeric(X)) {
    stop("Input 'data' must contain only numeric values.")
  }
  
  n <- nrow(X)
  d <- ncol(X)
  
  if (n < 2 || d < 2) {
    stop("The dataset must have at least 2 observations and 2 variables.")
  }
  
  # --- Main Calculation with Numerical Stability ---
  
  # 1. Empirical distribution function (Fn)
  fn_values <- vapply(1:n, function(i) {
    sum(rowSums(sweep(X, 2, X[i, ], "<=")) == d) / n
  }, numeric(1))
  
  # 2. Stable Generalized g function
  g_generic_stable <- function(t, d) {
    if (t <= .Machine$double.eps) return(1)
    if (t >= 1 - .Machine$double.eps) return(0)
    
    log_t <- log(t)
    neg_log_t <- -log_t
    
    # Using log-sum-exp logic for numerical stability
    terms <- vapply(0:(d - 1), function(i) {
      exp(log_t + i * log(neg_log_t) - lfactorial(i))
    }, numeric(1))
    
    val <- 1 - sum(terms)
    
    return(max(0, min(1, val)))
  }
  
  auk_values <- vapply(fn_values, g_generic_stable, numeric(1), d = d)
  
  return(mean(auk_values))
}