% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/abc_optim.R
\name{abc_optim}
\alias{abc_optim}
\alias{abc_answer}
\alias{print.abc_answer}
\alias{abc_cpp}
\alias{plot.abc_answer}
\title{Artificial Bee Colony Optimization}
\usage{
abc_optim(par, fn, ..., FoodNumber = 20, lb = rep(-Inf, length(par)),
  ub = rep(+Inf, length(par)), limit = 100, maxCycle = 1000,
  optiinteger = FALSE, criter = 50, parscale = rep(1, length(par)),
  fnscale = 1)

\method{print}{abc_answer}(x, ...)

abc_cpp(par, fn, ..., FoodNumber = 20, lb = rep(-Inf, length(par)),
  ub = rep(+Inf, length(par)), limit = 100, maxCycle = 1000,
  criter = 50, parscale = rep(1, length(par)), fnscale = 1)

\method{plot}{abc_answer}(x, y = NULL,
  main = "Trace of the Objective Function", xlab = "Number of iteration",
  ylab = "Value of the objective Function", type = "l", ...)
}
\arguments{
\item{par}{Initial values for the parameters to be optimized over}

\item{fn}{A function to be minimized, with first argument of the vector of
parameters over which minimization is to take place. It should return a
scalar result.}

\item{...}{In the case of \code{abc_*}, further arguments to be passed to 'fn',
otherwise, further arguments passed to the method.}

\item{FoodNumber}{Number of food sources to exploit. Notice that the param
\code{NP} has been deprecated.}

\item{lb}{Lower bound of the parameters to be optimized.}

\item{ub}{Upper bound of the parameters to be optimized.}

\item{limit}{Limit of a food source.}

\item{maxCycle}{Maximum number of iterations.}

\item{optiinteger}{Whether to optimize binary parameters or not.}

\item{criter}{Stop criteria (numer of unchanged results) until stopping}

\item{parscale}{Numeric vector of length \code{length(par)}. Scale applied
to the parameters (see \code{\link[stats:optim]{optim}}).}

\item{fnscale}{Numeric scalar. Scale applied function. If \code{fnscale < 0},
then the problem becomes a maximization problem (see \code{\link[stats:optim]{optim}}).}

\item{x}{An object of class \code{abc_answer}.}

\item{y}{Ignored}

\item{main}{Passed to \code{\link[graphics:plot.default]{plot}}.}

\item{xlab}{Passed to \code{\link[graphics:plot.default]{plot}}.}

\item{ylab}{Passed to \code{\link[graphics:plot.default]{plot}}.}

\item{type}{Passed to \code{\link[graphics:plot.default]{plot}}.}
}
\value{
An list of class \code{abc_answer}, holding the following elements:
\item{Foods}{Numeric matrix. Last position of the bees.}
\item{f}{Numeric vector. Value of the function evaluated at each set of \code{Foods}.}
\item{fitness}{Numeric vector. Fitness of each \code{Foods}.}
\item{trial}{Integer vector. Number of trials at each \code{Foods}.}
\item{value}{Numeric scalar. Value of the function evaluated at the optimum.}
\item{par}{Numeric vector. Optimum found.}
\item{counts}{Integer scalar. Number of cycles.}
\item{hist}{Numeric matrix. Trace of the global optimums.}
}
\description{
Implements Karaboga (2005) Artificial Bee Colony (ABC) Optimization algorithm.
}
\details{
This implementation of the ABC algorithm was developed based on the basic
version written in \code{C} and published at the algorithm's official
website (see references).

\code{abc_optim} and \code{abc_cpp} are two different implementations of the
algorithm, the former using pure \code{R} code, and the later using \code{C++},
via the \pkg{Rcpp} package. Besides of the output, another important
difference between the two implementations is speed, with \code{abc_cpp}
showing between 50\% and 100\% faster performance.

Upper and Lower bounds (\code{ub}, \code{lb}) equal to infinite will be replaced
by either \code{.Machine$double.xmax} or \code{-.Machine$double.xmax}.

If \code{D} (the number of parameters to be optimzed) is greater than one,
then \code{lb} and \code{ub} can be either scalars (assuming that all the
parameters share the same boundaries) or vectors (the parameters have
different boundaries each other).

The \code{plot} method shows the trace of the objective function
as the algorithm unfolds. The line is merely the result of the objective
function evaluated at each point (row) of the \code{hist} matrix return by
\code{abc_optim}/\code{abc_cpp}.

For now, the function will return with error if \code{...} was passed to
\code{abc_optim}/\code{abc_cpp}, since those argumens are not stored with the
result.
}
\examples{

# EXAMPLE 1: The minimum is at (pi,pi) --------------------------------------

fun <- function(x) {
  -cos(x[1])*cos(x[2])*exp(-((x[1] - pi)^2 + (x[2] - pi)^2))
}

abc_optim(rep(0,2), fun, lb=-10, ub=10, criter=50)

# This should be equivalent
abc_cpp(rep(0,2), fun, lb=-10, ub=10, criter=50)

# We can also turn this into a maximization problem, and get the same
# results 
fun <- function(x) {
  # We've removed the '-' from the equation
  cos(x[1])*cos(x[2])*exp(-((x[1] - pi)^2 + (x[2] - pi)^2))
}

abc_cpp(rep(0,2), fun, lb=-10, ub=10, criter=50, fnscale = -1)

# EXAMPLE 2: global minimum at about (-15.81515) ----------------------------

fw <- function (x)
  10*sin(0.3*x)*sin(1.3*x^2) + 0.00001*x^4 + 0.2*x+80

ans <- abc_optim(50, fw, lb=-100, ub=100, criter=100)
ans[c("par", "counts", "value")]


# EXAMPLE 3: 5D sphere, global minimum at about (0,0,0,0,0) -----------------
fs <- function(x) sum(x^2)

ans <- abc_optim(rep(10,5), fs, lb=-100, ub=100, criter=200)
ans[c("par", "counts", "value")]


# EXAMPLE 4: An Ordinary Linear Regression ----------------------------------

set.seed(1231)
k <- 4
n <- 5e2

# Data generating process
w <- matrix(rnorm(k), ncol=1)     # This are the model parameters
X <- matrix(rnorm(k*n), ncol = k) # This are the controls
y <- X \%*\% w                      # This is the observed data

# Objective function
fun <- function(x) {
  sum((y - X\%*\%x)^2)
}

# Running the regression
ans <- abc_optim(rep(0,k), fun, lb = -10000, ub=10000)

# Here are the outcomes: Both columns should be the same
cbind(ans$par, w)
#             [,1]        [,2]
# [1,] -0.08051177 -0.08051177
# [2,]  0.69528553  0.69528553
# [3,] -1.75956316 -1.75956316
# [4,]  0.36156427  0.36156427


# This is just like OLS, with no constant
coef(lm(y~0+X))
#         X1          X2          X3          X4 
#-0.08051177  0.69528553 -1.75956316  0.36156427 

}
\references{
D. Karaboga, \emph{An Idea based on Honey Bee Swarm for
Numerical Optimization}, tech. report TR06,Erciyes University, Engineering
Faculty, Computer Engineering Department, 2005
\url{http://mf.erciyes.edu.tr/abc/pub/tr06_2005.pdf}

Artificial Bee Colony (ABC) Algorithm (website)
\url{http://mf.erciyes.edu.tr/abc/index.htm}

Basic version of the algorithm implemented in \code{C} (ABC's official
website) \url{http://mf.erciyes.edu.tr/abc/form.aspx}
}
\author{
George Vega Yon \email{g.vegayon@gmail.com}
}
\keyword{optimization}
