use super::*;

#[derive(PartialEq, Clone)]
pub struct Environment {
    pub(crate) robj: Robj,
}

impl Environment {








    pub fn new_with_parent(parent: Environment) -> Self {

        Environment::new_with_capacity(parent, 14)
    }














    pub fn new_with_capacity(parent: Environment, capacity: usize) -> Self {
        if capacity <= 5 {

            new_env(parent, false, 0)
        } else {

            new_env(parent, true, capacity as i32 * 2 + 1)
        }
    }











    #[allow(clippy::wrong_self_convention)]
    pub fn from_pairs<NV>(parent: Environment, names_and_values: NV) -> Self
    where
        NV: IntoIterator,
        NV::Item: SymPair,
    {
        single_threaded(|| {
            let dict_len = 29;
            let env = new_env(parent, true, dict_len);
            for nv in names_and_values {
                let (n, v) = nv.sym_pair();
                if let Some(n) = n {
                    unsafe { Rf_defineVar(n.get(), v.get(), env.get()) }
                }
            }
            env
        })
    }


    pub fn parent(&self) -> Option<Environment> {
        unsafe {
            let sexp = self.robj.get();
            let robj = Robj::from_sexp(ENCLOS(sexp));
            robj.try_into().ok()
        }
    }


    pub fn set_parent(&mut self, parent: Environment) -> &mut Self {
        single_threaded(|| unsafe {
            let sexp = self.robj.get();
            SET_ENCLOS(sexp, parent.robj.get());
        });
        self
    }


    pub fn envflags(&self) -> i32 {
        unsafe {
            let sexp = self.robj.get();
            ENVFLAGS(sexp)
        }
    }


    pub fn set_envflags(&mut self, flags: i32) -> &mut Self {
        unsafe {
            let sexp = self.robj.get();
            SET_ENVFLAGS(sexp, flags)
        }
        self
    }


    pub fn iter(&self) -> EnvIter {
        unsafe {
            let hashtab = Robj::from_sexp(HASHTAB(self.get()));
            let frame = Robj::from_sexp(FRAME(self.get()));
            if hashtab.is_null() && frame.is_pairlist() {
                EnvIter {
                    hash_table: ListIter::new(),
                    pairlist: frame.as_pairlist().unwrap().iter(),
                }
            } else {
                EnvIter {
                    hash_table: hashtab.as_list().unwrap().values(),
                    pairlist: PairlistIter::new(),
                }
            }
        }
    }










    pub fn names(&self) -> impl Iterator<Item = &str> {
        self.iter().map(|(k, _)| k)
    }











    pub fn set_local<K: Into<Robj>, V: Into<Robj>>(&self, key: K, value: V) {
        let key = key.into();
        let value = value.into();
        if key.is_symbol() {
            single_threaded(|| unsafe {
                Rf_defineVar(key.get(), value.get(), self.get());
            })
        }
    }










    pub fn local<K: Into<Robj>>(&self, key: K) -> Result<Robj> {
        let key = key.into();
        if key.is_symbol() {
            unsafe {
                Ok(Robj::from_sexp(Rf_findVarInFrame3(
                    self.get(),
                    key.get(),
                    1,
                )))
            }
        } else {
            Err(Error::NotFound(key))
        }
    }
}
























#[derive(Clone)]
pub struct EnvIter {
    hash_table: ListIter,
    pairlist: PairlistIter,
}

impl Iterator for EnvIter {
    type Item = (&'static str, Robj);

    fn next(&mut self) -> Option<Self::Item> {
        loop {


            for (key, value) in &mut self.pairlist {

                if !key.is_na() && !value.is_unbound_value() {
                    return Some((key, value));
                }
            }


            loop {
                if let Some(obj) = self.hash_table.next() {
                    if !obj.is_null() && obj.is_pairlist() {
                        self.pairlist = obj.as_pairlist().unwrap().iter();
                        break;
                    }

                } else {

                    return None;
                }
            }
        }
    }
}

impl std::fmt::Debug for Environment {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        unsafe {
            let sexp = self.get();
            if sexp == R_GlobalEnv {
                write!(f, "global_env()")
            } else if sexp == R_BaseEnv {
                write!(f, "base_env()")
            } else if sexp == R_EmptyEnv {
                write!(f, "empty_env()")
            } else {
                write!(f, "{}", self.deparse().unwrap())
            }
        }
    }
}
