% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simTrack.R
\name{simTelemetryTrack}
\alias{simTelemetryTrack}
\title{Simulate telemetry track based on known true track obtained using simTrueTrack}
\usage{
simTelemetryTrack(
  trueTrack,
  pingType,
  sbi_mean = NULL,
  sbi_sd = NULL,
  rbi_min = NULL,
  rbi_max = NULL
)
}
\arguments{
\item{trueTrack}{Know track obtained using simTrueTrack}

\item{pingType}{Type of transmitter to simulate - either stable burst interval ('sbi'), random burst interval ('rbi') or random burst interval but where the random sequence is known a priori}

\item{sbi_mean, sbi_sd}{Mean and SD of burst interval when pingType = 'sbi'}

\item{rbi_min}{Minimum and maximum BI for random burst interval transmitters}

\item{rbi_max}{Minimum and maximum BI for random burst interval transmitters}
}
\value{
\code{data.frame} containing time of ping and true positions
}
\description{
Based on a known true track obtained using simTrueTrack, this function will give true positions at time-of-pings, which are also in the output. TOPs are determined by user-specified transmitter type.
Number of pings are determined automatically based on track length and transmitter specifications.
}
\examples{
\donttest{
library(yaps)
set.seed(42)
# Simulate true track of animal movement of n seconds
trueTrack <- simTrueTrack(model='crw', n = 1000, deltaTime=1, shape=1, 
	scale=0.5, addDielPattern=TRUE, ss='rw')

# Simulate telemetry observations from true track.
# Format and parameters depend on type of transmitter burst interval (BI).
pingType <- 'sbi'

if(pingType == 'sbi') { # stable BI
    sbi_mean <- 30; sbi_sd <- 1e-4;
    teleTrack <- simTelemetryTrack(trueTrack, pingType=pingType, sbi_mean=sbi_mean, sbi_sd=sbi_sd)
} else if(pingType == 'rbi'){ # random BI
    pingType <- 'rbi'; rbi_min <- 20; rbi_max <- 40;
    teleTrack <- simTelemetryTrack(trueTrack, pingType=pingType, rbi_min=rbi_min, rbi_max=rbi_max)
}

# Simulate hydrophone array
hydros <- simHydros(auto=TRUE, trueTrack=trueTrack)
toa_list <- simToa(teleTrack, hydros, pingType, sigmaToa=1e-4, pNA=0.25, pMP=0.01)
toa <- toa_list$toa

# Specify whether to use ss_data from measured water temperature (ss_data_what <- 'data') or 
#	to estimate ss in the model (ss_data_what <- 'est')
ss_data_what <- 'data'
if(ss_data_what == 'data') {ss_data <- teleTrack$ss} else {ss_data <- 0}


if(pingType == 'sbi'){
    inp <- getInp(hydros, toa, E_dist="Mixture", n_ss=10, pingType=pingType, sdInits=0, 
		ss_data_what=ss_data_what, ss_data=ss_data)
} else if(pingType == 'rbi'){
    inp <- getInp(hydros, toa, E_dist="Mixture", n_ss=10, pingType=pingType, sdInits=0, 
		rbi_min=rbi_min, rbi_max=rbi_max, ss_data_what=ss_data_what, ss_data=ss_data)
} 

pl <- c()
maxIter <- ifelse(pingType=="sbi", 500, 5000)
outTmb <- runYaps(inp, maxIter=maxIter, getPlsd=TRUE, getRep=TRUE)

# Estimates in pl
pl <- outTmb$pl
# Correcting for hydrophone centering
pl$X <- outTmb$pl$X + inp$inp_params$Hx0
pl$Y <- outTmb$pl$Y + inp$inp_params$Hy0


# Error estimates in plsd
plsd <- outTmb$plsd

# plot the resulting estimated track
plot(y~x, data=trueTrack, type="l", xlim=range(hydros$hx), ylim=range(hydros$hy), asp=1)
lines(y~x, data=teleTrack)
points(hy~hx, data=hydros, col="green", pch=20, cex=3)
lines(pl$Y~pl$X, col="red")
}
}
