#' List pages that meet certain criteria
#'
#' See [API:Lists](https://www.mediawiki.org/wiki/API:Lists) for available
#' list actions. Each list action returns a list of pages, typically including
#' their pageid, [namespace](https://www.mediawiki.org/wiki/Manual:Namespace)
#' and title. Individual lists have particular properties that can be requested,
#' which are usually prefaced with a two-word code based on the name of the
#' list (e.g. specific properties for the `categorymembers` list action are
#' prefixed with `cm`).
#'
#' When the request is performed, the data is returned in the body of the
#' request under the `query` object, labeled by the chosen list action.
#'
#' If you want to study the actual pages listed, it is advisable to retrieve
#' the pages directly using a generator, rather than listing their IDs using a
#' list action. When using a list action, a second request is required to get
#' further information about each page. Using a generator, you can query pages
#' and retrieve their relevant properties in a single API call.
#'
#' @param .req A httr2_request, e.g. generated by `wiki_action_request`
#' @param list The [type of list](https://www.mediawiki.org/wiki/API:Lists) to return
#' @param ... <[`dynamic-dots`][rlang::dyn-dots]> Additional parameters to the query, e.g. to set configure list
#'
#' @return An HTTP response: an S3 list with class httr2_request
#' @export
#'
#' @examples
#' # Get the ten most recently added pages in Category:Physics
#' physics_pages <- wiki_action_request() %>%
#'   query_list_pages("categorymembers",
#'     cmsort = "timestamp",
#'     cmdir = "desc", cmtitle = "Category:Physics"
#'   ) %>%
#'   next_batch()
#'
#' physics_pages
query_list_pages <- function(.req, list, ...) {
  check_module(list, "list")
  # TODO: check_params
  new_list_query(.req, list, ...)
}

#' @rdname query_list_pages
#' @export
list_all_list_modules <- function() {
  schema_query_modules %>%
    dplyr::filter(group == "list") %>%
    dplyr::select(name)
}

#' Constructor for [list](https://www.mediawiki.org/wiki/API:Lists) queries
#'
#' This low-level constructor only performs basic type checking.
#'
#' @param .req A [`query/action_api/httr2_request`][wiki_action_request()]
#'   object, or a `list/query/action_api/httr2_request` as returned by this
#'   function.
#' @param list The [list module](https://www.mediawiki.org/wiki/API:Lists) to
#'   add to the query
#' @param ... <[`dynamic-dots`][rlang::dyn-dots]> Parameters to the list module
#'
#' @keywords low_level_action_api
#'
#' @return An object of type `list/query/action_api/httr2_request`.
#' @export
#' @examples
#' # Create a query to list all members of Category:Physics
#' physics_query <- wiki_action_request() %>%
#'   new_list_query("categorymembers", cmtitle="Category:Physics")
#'
new_list_query <- function(.req, list, ...) {
  UseMethod("new_list_query")
}

#' @rdname new_list_query
#' @export
new_list_query.list <- function(.req, list, ...) {
  req <- set_action(.req, "list", list, ...)
  req
}

#' @rdname new_list_query
#' @export
new_list_query.generator <- function(.req, list, ...) {
  incompatible_query_error("list", "generator")
}

#' @rdname new_list_query
#' @export
new_list_query.prop <- function(.req, list, ...) {
  incompatible_query_error("list", "prop")
}

#' @rdname new_list_query
#' @export
new_list_query.query <- function(.req, list, ...) {
  req <- set_action(.req, "list", list, ...)
  class(req) <- c("list", class(req))
  req
}

is_list_query <- function(.req) {
  is_query_subtype(.req, "list")
}
