% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/convertToPA.R
\name{convertToPA}
\alias{convertToPA}
\title{Convert a virtual species distribution (or a suitability raster) into presence-absence}
\usage{
convertToPA(x, PA.method = "probability", beta = "random",
  alpha = -0.05, species.prevalence = NULL, plot = TRUE)
}
\arguments{
\item{x}{a suitability raster, or the output from functions 
\code{\link{generateSpFromFun}}, \code{\link{generateSpFromPCA}} 
or \code{\link{generateRandomSp}}}

\item{PA.method}{\code{"threshold"} or \code{"probability"}. If 
\code{"threshold"}, then occurrence probabilities are simply converted into
presence-absence according to the threshold \code{beta}. If \code{"probability"}, then
probabilities are converted according to a logistic function of threshold 
\code{beta} and slope \code{alpha}.}

\item{beta}{\code{"random"}, a numeric value in the range of your 
probabilities or \code{NULL}. This is the threshold of conversion into
presence-absence (= the inflexion point if \code{PA.method = "probability"}).
If \code{"random"}, a numeric value will be randomly generated within the range
of \code{x}.}

\item{alpha}{\code{NULL} or a negative numeric value. Only useful if 
\code{PA.method = "probability"}. The value of \code{alpha} will
shape the logistic function transforming occurrences into presence-absences.
See \code{\link{logisticFun}} and examples therein for the choice of \code{alpha}}

\item{species.prevalence}{\code{NULL} or a numeric value between 0 and 1.
The species prevalence is the proportion of sites actually occupied by the
species.}

\item{plot}{\code{TRUE} or \code{FALSE}. If \code{TRUE}, maps of probabilities
of occurrence and presence-absence will be plotted.}
}
\value{
a \code{list} containing 5 elements:
\itemize{
\item{\code{approach}: the approach used to generate the species, \emph{i.e.}, \code{"response"}}
\item{\code{details}: the details and parameters used to generate the species}
\item{\code{suitab.raster}: the virtual species distribution, as a Raster object containing the
environmental suitability)}
\item{\code{PA.conversion}: the parameters used to convert the suitability into presence-absence}
\item{\code{pa.raster}: the presence-absence map, as a Raster object containing 0 (absence) / 1 (presence) / NA}
}
The structure of the virtualspecies object can be seen using \code{str()}
}
\description{
This functions converts the probabilities of presence from the output of
 \code{\link{generateSpFromFun}}, \code{\link{generateSpFromPCA}}, \code{\link{generateRandomSp}}
or a suitability raster into
a presence-absence raster. The conversion can be threshold-based, or based
on a probability of conversion (see details).
}
\details{
The conversion of probabilities of occurrence into presence - absence is
usually performed by selecting a threshold above which presence always occurs,
and never below. However, this approach may be too much unrealistic because
species may sometime be present in areas with a low probability of occurrence,
or be absent from areas with a high probability of occurrence. In addition,
when using a threshold you erase the previously generated response shapes: 
it all becomes threshold. Thus, this threshold approach should be avoided.

 
Hence, a more
realistic conversion consists in converting probabilities into presence -
absence with a probability function (see references). Such a probability 
conversion can be performed here with a logit function 
(see \code{\link{logisticFun}}).

To perform the probability conversion you have to define two of the
three following parameters:
\itemize{
\item{\code{beta}: the 'threshold' of the logistic function (i.e. the 
inflexion point)}
\item{\code{alpha}: the slope of the logistic function}
\item{\code{species.prevalence}: the proportion of sites in which the species
occur}
}

If you provide \code{beta} and \code{alpha}, the \code{species.prevalence}
is calculated immediately calculated after conversion into presence-absence.

On the other hand, if you provide \code{species.prevalence} and either
\code{beta} or \code{alpha}, the function will try to determine \code{alpha}
(if you provided \code{beta}) or \code{beta} (if you provided \code{alpha}).

The relationship between species prevalence, alpha and beta is dependent
on the available range of environmental conditions (see Meynard and Kaplan,
2011 and especially the Supporting Information). As a consequence, the 
desired species prevalence may not be available for the defined \code{alpha} 
or \code{beta}. In these conditions, the function will retain the \code{alpha} or
\code{beta} which provides the closest prevalence to your \code{species.prevalence},
but you may also provide another value of \code{alpha} or \code{beta} to obtain
a closer prevalence. 
 
In all cases, the \code{species.prevalence} indicated in the output is the
prevalence measured on the output presence-absence map.
}
\note{
The approximation of \code{alpha} or \code{beta} to the chosen 
\code{species.prevalence} may take time if you work on very large rasters.
}
\examples{
# Create an example stack with two environmental variables
a <- matrix(rep(dnorm(1:100, 50, sd = 25)), 
            nrow = 100, ncol = 100, byrow = TRUE)
env <- stack(raster(a * dnorm(1:100, 50, sd = 25)),
             raster(a * 1:100))
names(env) <- c("variable1", "variable2")

# Creation of the parameter list
parameters <- formatFunctions(variable1 = c(fun = 'dnorm', mean = 0.00012,
                                            sd = 0.0001),
                              variable2 = c(fun = 'linearFun', a = 1, b = 0))
sp1 <- generateSpFromFun(env, parameters, plot = FALSE)

# Conversion into presence-absence with a threshold-based approach
convertToPA(sp1, PA.method = "threshold", beta = 0.2,  plot = TRUE)
convertToPA(sp1, PA.method = "threshold", beta = "random", plot = TRUE)

# Conversion into presence-absence with a probability-based approach
convertToPA(sp1, PA.method = "probability", beta = 0.4, 
              alpha = -0.05, plot = TRUE)
convertToPA(sp1, PA.method = "probability", beta = "random", 
              alpha = -0.1, plot = TRUE)
              
# Conversion into presence-absence by choosing the prevalence
# Threshold method
convertToPA(sp1, PA.method = "threshold",
              species.prevalence = 0.3, plot = TRUE)
# Probability method, with alpha provided              
convertToPA(sp1, PA.method = "probability", alpha = -0.1, 
              species.prevalence = 0.2, plot = TRUE)        
# Probability method, with beta provided              
convertToPA(sp1, PA.method = "probability", beta = 0.5, 
              species.prevalence = 0.2, alpha = NULL, 
              plot = TRUE)    
 
# Plot the output Presence-Absence raster only             
sp1 <- convertToPA(sp1, plot = FALSE)
plot(sp1$pa.raster)
}
\references{
Meynard C.N. & Kaplan D.M. 2013. Using virtual species to study species 
distributions and model performance. 
\emph{Journal of Biogeography} \bold{40}:1-8

Meynard C.N. & Kaplan D.M. 2011. The effect of a gradual response to the 
environment on species distribution model performance.
\emph{Ecography} \bold{35}:499-509
}
\author{
Boris Leroy \email{leroy.boris@gmail.com}

with help from C. N. Meynard, C. Bellard & F. Courchamp
}
